/**
 * Barcode.WASM Easy-8-Steps サンプル (TypeScript版)
 * 
 * Barcode.WASMの最小限の使い方を示すサンプルです。
 * たった8ステップでバーコードを生成できます。
 */

// =============================================================================
// 型定義
// =============================================================================

/** 1次元バーコードクラス（Code128等） */
interface Barcode1D {
  setFitWidth(fit: boolean): void;
  setOutputFormat(format: string): void;
  draw(data: string, width: number, height: number): string;
  delete(): void;
}

/** 2次元バーコードクラス（QR等） */
interface Barcode2D {
  setFitWidth(fit: boolean): void;
  setOutputFormat(format: string): void;
  draw(data: string, size: number): string;
  delete(): void;
}

/** WASMモジュール */
interface BarcodeModule {
  Code128: new () => Barcode1D;
  QR: new () => Barcode2D;
  // 他のバーコードタイプも必要に応じて追加
}

/** グローバルのModule関数 */
declare function Module(): Promise<BarcodeModule>;

// =============================================================================
// メイン処理
// =============================================================================

let barcodeModule: BarcodeModule | null = null;

/**
 * 1次元バーコード生成（PNG）
 */
function generate1D_PNG(): void {
  if (!barcodeModule) {
    alert('WASMモジュールが初期化されていません');
    return;
  }

  const resultDiv = document.getElementById('result');
  if (!resultDiv) return;

  // 描画領域をクリア
  resultDiv.innerHTML = '';

  // Step 3: バーコード(今回はCODE128)インスタンス作成
  const barcode = new barcodeModule.Code128();

  // Step 4: 指定幅ぴったり調整（省略可）
  // 省略した場合、指定幅以内でピクセルにはまる高精度なバーコードを出力します
  barcode.setFitWidth(true);

  // Step 5: 出力形式設定（PNG出力時は省略可）
  // barcode.setOutputFormat('png');

  // Step 6: バーコード生成
  const barcodeString: string = barcode.draw("Pao@Office 12345!", 600, 150);

  // Step 7: 結果をHTMLに埋め込み
  // PNG出力時: Base64エンコードされたデータURLが返される
  resultDiv.innerHTML = '<img src="' + barcodeString + '" alt="Code128 Barcode">';

  // Step 8: メモリ解放（重要！）
  barcode.delete();

  console.log('✅ 1次元バーコード生成完了（PNG）');
}

/**
 * 1次元バーコード生成（SVG）
 */
function generate1D_SVG(): void {
  if (!barcodeModule) {
    alert('WASMモジュールが初期化されていません');
    return;
  }

  const resultDiv = document.getElementById('result');
  if (!resultDiv) return;

  // 描画領域をクリア
  resultDiv.innerHTML = '';

  // Step 3: バーコード(今回はCODE128)インスタンス作成
  const barcode = new barcodeModule.Code128();

  // Step 4: 指定幅ぴったり調整（省略可）
  // 省略した場合、指定幅以内でピクセルにはまる高精度なバーコードを出力します
  barcode.setFitWidth(true);

  // Step 5: 出力形式設定（PNG出力時は省略可）
  barcode.setOutputFormat('svg');

  // Step 6: バーコード生成
  const barcodeString: string = barcode.draw("Pao@Office 12345!", 600, 150);

  // Step 7: 結果をHTMLに埋め込み
  // SVG出力時: SVG文字列がそのまま返されるので直接埋め込み
  resultDiv.innerHTML = barcodeString;

  // Step 8: メモリ解放（重要！）
  barcode.delete();

  console.log('✅ 1次元バーコード生成完了（SVG）');
}

/**
 * 2次元バーコード生成（PNG）
 */
function generate2D_PNG(): void {
  if (!barcodeModule) {
    alert('WASMモジュールが初期化されていません');
    return;
  }

  const resultDiv = document.getElementById('result');
  if (!resultDiv) return;

  // 描画領域をクリア
  resultDiv.innerHTML = '';

  // Step 3: バーコード(今回はQRコード)インスタンス作成
  const barcode = new barcodeModule.QR();

  // Step 4: 指定幅ぴったり調整（省略可）
  // 省略した場合、指定幅以内でピクセルにはまる高精度なバーコードを出力します
  barcode.setFitWidth(true);

  // Step 5: 出力形式設定（PNG出力時は省略可）
  // barcode.setOutputFormat('png');

  // Step 6: バーコード生成
  const barcodeString: string = barcode.draw("https://www.pao.ac/", 200);

  // Step 7: 結果をHTMLに埋め込み
  // PNG出力時: Base64エンコードされたデータURLが返される
  resultDiv.innerHTML = '<img src="' + barcodeString + '" alt="QR Code">';

  // Step 8: メモリ解放（重要！）
  barcode.delete();

  console.log('✅ 2次元バーコード生成完了（PNG）');
}

/**
 * 2次元バーコード生成（SVG）
 */
function generate2D_SVG(): void {
  if (!barcodeModule) {
    alert('WASMモジュールが初期化されていません');
    return;
  }

  const resultDiv = document.getElementById('result');
  if (!resultDiv) return;

  // 描画領域をクリア
  resultDiv.innerHTML = '';

  // Step 3: バーコード(今回はQRコード)インスタンス作成
  const barcode = new barcodeModule.QR();

  // Step 4: 指定幅ぴったり調整（省略可）
  // 省略した場合、指定幅以内でピクセルにはまる高精度なバーコードを出力します
  barcode.setFitWidth(true);

  // Step 5: 出力形式設定（PNG出力時は省略可）
  barcode.setOutputFormat('svg');

  // Step 6: バーコード生成
  const barcodeString: string = barcode.draw("https://www.pao.ac/", 200);

  // Step 7: 結果をHTMLに埋め込み
  // SVG出力時: SVG文字列がそのまま返されるので直接埋め込み
  resultDiv.innerHTML = barcodeString;

  // Step 8: メモリ解放（重要！）
  barcode.delete();

  console.log('✅ 2次元バーコード生成完了（SVG）');
}

/**
 * 初期化
 */
async function init(): Promise<void> {
  const resultDiv = document.getElementById('result');
  
  try {
    // Step 1: モジュールのインポート（import文で実行済み）
    // Step 2: モジュールの初期化
    barcodeModule = await Module();
    
    if (resultDiv) {
      resultDiv.textContent = '✅ 初期化完了！ボタンをクリックしてください';
    }
    console.log('✅ Barcode.WASM 初期化完了');
  } catch (error) {
    const errorMsg = error instanceof Error ? error.message : String(error);
    if (resultDiv) {
      resultDiv.textContent = '❌ 初期化失敗: ' + errorMsg;
    }
    console.error('❌ Barcode.WASM 初期化失敗:', error);
    return;
  }

  // ボタンにイベントハンドラを設定
  document.getElementById('btn1d_png')?.addEventListener('click', generate1D_PNG);
  document.getElementById('btn1d_svg')?.addEventListener('click', generate1D_SVG);
  document.getElementById('btn2d_png')?.addEventListener('click', generate2D_PNG);
  document.getElementById('btn2d_svg')?.addEventListener('click', generate2D_SVG);
}

// DOMContentLoaded時に初期化
document.addEventListener('DOMContentLoaded', init);
