// Barcode.Swift (C++ WASM) - Easy 2 Steps
//
// QR code generation in just 2 steps using Vapor + WASM engine.
//
//   swift run
//   -> http://localhost:5762
//
import Vapor
import Leaf
import BarcodePaoWasm

// MARK: - Request / Response Models

struct DrawRequest: Content {
    var barcodeType: String?
    var code: String?
    var format: String?
    var width: Int?
    var height: Int?
    var size: Int?
}

struct Base64Response: Content {
    var ok: Bool
    var base64: String?
    var error: String?
}

struct SVGResponse: Content {
    var ok: Bool
    var svg: String?
    var error: String?
}

// MARK: - Barcode Helper

func generateBarcode(req: DrawRequest) throws -> (base64: String?, svg: String?) {
    let barcodeType = req.barcodeType ?? "qr"
    let code = req.code ?? "https://www.pao.ac/"
    let format = req.format ?? BarcodeFormat.png
    let width = req.width ?? 400
    let height = req.height ?? 120
    let size = req.size ?? 300

    switch barcodeType {
    case "qr":
        let bc = QR(outputFormat: format)
        let result = try bc.draw(code: code, size: size)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    case "code128":
        let bc = Code128(outputFormat: format)
        let result = try bc.draw(code: code, width: width, height: height)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    case "code39":
        let bc = Code39(outputFormat: format)
        let result = try bc.draw(code: code, width: width, height: height)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    case "jan13":
        let bc = JAN13(outputFormat: format)
        let result = try bc.draw(code: code, width: width, height: height)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    case "jan8":
        let bc = JAN8(outputFormat: format)
        let result = try bc.draw(code: code, width: width, height: height)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    case "datamatrix":
        let bc = DataMatrix(outputFormat: format)
        let result = try bc.draw(code: code, size: size)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    case "pdf417":
        let bc = PDF417(outputFormat: format)
        let result = try bc.draw(code: code, width: width, height: height)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    case "code93":
        let bc = Code93(outputFormat: format)
        let result = try bc.draw(code: code, width: width, height: height)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    case "nw7":
        let bc = NW7(outputFormat: format)
        let result = try bc.draw(code: code, width: width, height: height)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    case "itf":
        let bc = ITF(outputFormat: format)
        let result = try bc.draw(code: code, width: width, height: height)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    case "upc_a":
        let bc = UPCA(outputFormat: format)
        let result = try bc.draw(code: code, width: width, height: height)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    case "upc_e":
        let bc = UPCE(outputFormat: format)
        let result = try bc.draw(code: code, width: width, height: height)
        if format == BarcodeFormat.svg { return (nil, result) }
        return (result, nil)

    default:
        throw Abort(.badRequest, reason: "Unknown barcode type: \(barcodeType)")
    }
}

// MARK: - App Configuration

let app = try Application(.detect())
defer { app.shutdown() }

app.views.use(.leaf)
app.leaf.configuration.rootDirectory = app.directory.viewsDirectory

app.http.server.configuration.hostname = "0.0.0.0"
app.http.server.configuration.port = 5762

// MARK: - Routes

app.get { req async throws -> View in
    return try await req.view.render("index")
}

app.post("draw-base64") { req -> Base64Response in
    let drawReq = try req.content.decode(DrawRequest.self)
    do {
        let result = try generateBarcode(req: drawReq)
        if result.svg != nil {
            return Base64Response(ok: false, base64: nil, error: "Use /draw-svg for SVG format")
        }
        return Base64Response(ok: true, base64: result.base64, error: nil)
    } catch {
        return Base64Response(ok: false, base64: nil, error: "\(error)")
    }
}

app.post("draw-svg") { req -> SVGResponse in
    var drawReq = try req.content.decode(DrawRequest.self)
    drawReq.format = BarcodeFormat.svg
    do {
        let result = try generateBarcode(req: drawReq)
        return SVGResponse(ok: true, svg: result.svg, error: nil)
    } catch {
        return SVGResponse(ok: false, svg: nil, error: "\(error)")
    }
}

app.post("draw-canvas") { req -> Base64Response in
    let drawReq = try req.content.decode(DrawRequest.self)
    do {
        let code = drawReq.code ?? "https://www.pao.ac/"
        let qr = QR(outputFormat: BarcodeFormat.png)
        let b64 = try qr.draw(code: code, size: 300)
        return Base64Response(ok: true, base64: b64, error: nil)
    } catch {
        return Base64Response(ok: false, base64: nil, error: "\(error)")
    }
}

app.get("pdf") { req -> Response in
    let code = req.query[String.self, at: "code"] ?? "https://www.pao.ac/"

    let qr = QR(outputFormat: BarcodeFormat.png)
    let qrBase64 = try qr.draw(code: code, size: 200)

    let c128 = Code128(outputFormat: BarcodeFormat.png)
    let invNo = "INV-\(formatDate())"
    let c128Base64 = try c128.draw(code: invNo, width: 300, height: 60)

    let jan = JAN13(outputFormat: BarcodeFormat.png)
    let janBase64 = try jan.draw(code: "490123456789", width: 200, height: 60)

    let today = formatDate()

    let html = """
    <!DOCTYPE html>
    <html>
    <head>
    <meta charset="UTF-8">
    <title>Delivery Note - \(invNo)</title>
    <style>
    *{margin:0;padding:0;box-sizing:border-box}
    body{font-family:'Segoe UI',system-ui,sans-serif;background:#fff;color:#0F172A}
    .header{background:linear-gradient(135deg,#1E40AF,#7C3AED);color:#fff;padding:30px 40px}
    .header h1{font-size:24px;font-weight:800}
    .header-right{float:right;text-align:right;font-size:12px;line-height:1.8}
    .accent{height:4px;background:linear-gradient(90deg,#FA7343,#FF6B35,#10B981)}
    .content{padding:30px 40px}
    .recipient{font-size:14px;font-weight:600;margin-bottom:20px}
    .barcode-area{background:#F1F5F9;border-radius:8px;padding:20px;margin-bottom:30px;display:flex;gap:40px;align-items:flex-start}
    .barcode-section{flex:1}
    .barcode-label{font-size:10px;color:#64748B;font-weight:700;text-transform:uppercase;letter-spacing:0.08em;margin-bottom:8px}
    .barcode-section img{max-width:100%}
    table{width:100%;border-collapse:collapse;margin-bottom:20px}
    th{background:#1E40AF;color:#fff;padding:8px 12px;text-align:left;font-size:12px}
    td{padding:8px 12px;font-size:12px;border-bottom:1px solid #E2E8F0}
    tr:nth-child(even){background:#F8FAFC}
    .total-row{text-align:right;font-size:16px;color:#1E40AF;font-weight:700;padding:10px 0;border-top:2px solid #1E40AF}
    .jan-section{margin-top:20px}
    .footer{margin-top:40px;padding-top:15px;border-top:1px solid #E2E8F0;font-size:9px;color:#94A3B8;display:flex;justify-content:space-between}
    @media print{body{-webkit-print-color-adjust:exact;print-color-adjust:exact}}
    </style>
    </head>
    <body>
    <div class="header">
        <div class="header-right">
            Pao@Office<br>
            No: \(invNo)<br>
            Date: \(today)<br>
            Tokyo, Japan
        </div>
        <h1>DELIVERY NOTE</h1>
    </div>
    <div class="accent"></div>
    <div class="content">
        <div class="recipient">Sample Corporation</div>
        <div class="barcode-area">
            <div class="barcode-section">
                <div class="barcode-label">Invoice Barcode (Code128)</div>
                <img src="\(c128Base64)" alt="Code128">
            </div>
            <div class="barcode-section">
                <div class="barcode-label">QR Code</div>
                <img src="\(qrBase64)" alt="QR Code" style="max-width:140px">
            </div>
        </div>
        <table>
            <tr><th>Item</th><th>Qty</th><th>Unit Price</th><th>Amount</th></tr>
            <tr><td>Barcode.Swift C++ WASM Edition</td><td>1</td><td>&yen;11,000</td><td>&yen;11,000</td></tr>
            <tr><td>Annual Support (3 years)</td><td>1</td><td>&yen;9,900</td><td>&yen;9,900</td></tr>
        </table>
        <div class="total-row">Total: &yen;20,900</div>
        <div class="jan-section">
            <div class="barcode-label">Product Barcode (JAN-13)</div>
            <img src="\(janBase64)" alt="JAN-13">
        </div>
        <div class="footer">
            <span>Generated by Barcode.Swift (C++ WASM) &mdash; BarcodePaoWasm + Vapor</span>
            <span>pao.ac</span>
        </div>
    </div>
    </body>
    </html>
    """

    var headers = HTTPHeaders()
    headers.add(name: .contentType, value: "text/html; charset=utf-8")
    return Response(status: .ok, headers: headers, body: .init(string: html))
}

// MARK: - Helpers

func formatDate() -> String {
    let df = DateFormatter()
    df.dateFormat = "yyyy-MM-dd"
    return df.string(from: Date())
}

// MARK: - Start

print("Barcode.Swift (C++ WASM) Easy 2 Steps")
print("-> http://localhost:5762")
try app.run()
