import Foundation

private let yubinPtnN = ["144", "114", "132", "312", "123",
                         "141", "321", "213", "231", "411"]
private let yubinPtnC = ["324", "342", "234", "432", "243", "423", "441", "111"]
private let yubinStart = "13"
private let yubinStop  = "31"
private let yubinPtnHI = "414"
private let yubinAscChrs = "0123456789-ABCDEFGHIJKLMNOPQRSTUVWXYZ"

private func yubinGetPattern(_ index: Int) -> String {
    if index < 0 || index > 36 { return "" }
    if index <= 9 { return yubinPtnN[index] }
    if index == 10 { return yubinPtnHI }
    if index <= 20 { return yubinPtnC[0] + yubinPtnN[index - 11] }
    if index <= 30 { return yubinPtnC[1] + yubinPtnN[index - 21] }
    return yubinPtnC[2] + yubinPtnN[index - 31]
}

private func yubinGetCheckPattern(_ chkD: Int) -> String {
    if chkD < 0 || chkD > 18 { return "" }
    if chkD <= 9 { return yubinPtnN[chkD] }
    if chkD == 10 { return yubinPtnHI }
    return yubinPtnC[chkD - 11]
}

private func yubinPatternToCheckValue(_ pattern: String) -> Int {
    for j in 0..<10 {
        if pattern == yubinPtnN[j] { return j }
    }
    if pattern == yubinPtnHI { return 10 }
    for j in 0..<8 {
        if pattern == yubinPtnC[j] { return 11 + j }
    }
    return -1
}

/// Japan Post Customer Barcode (Yubin Customer) encoder.
public class YubinCustomer: BarcodeBase1D {
    private var bars: [Int] = []
    private(set) public var checkDigitValue: Int = -1

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    public func getBars() -> [Int] { bars }

    /// Encodes the Japan Post Customer Barcode.
    public func encode(_ code: String) throws -> [Int] {
        if code.isEmpty {
            throw BarcodeError.emptyString
        }

        let upperCode = code.uppercased()
        var barsResult = [Int]()
        var chkStr = ""

        // Start code
        for ch in yubinStart {
            barsResult.append(Int(String(ch))!)
        }

        // Data part (20 character positions)
        var codeLen = 0
        var i = upperCode.startIndex
        while i < upperCode.endIndex && codeLen < 20 {
            let ch = upperCode[i]
            guard let p = yubinAscChrs.firstIndex(of: ch) else {
                i = upperCode.index(after: i)
                continue
            }
            let idx = yubinAscChrs.distance(from: yubinAscChrs.startIndex, to: p)
            let pattern = yubinGetPattern(idx)

            // Special case: at position 19, if pattern > 3 bars (alpha char), use CC1 only
            if codeLen == 19 && pattern.count > 3 {
                for ch2 in yubinPtnC[0] {
                    barsResult.append(Int(String(ch2))!)
                    chkStr.append(ch2)
                }
                codeLen += 1
                break
            }

            for ch2 in pattern {
                barsResult.append(Int(String(ch2))!)
                chkStr.append(ch2)
            }

            codeLen += pattern.count <= 3 ? 1 : 2
            i = upperCode.index(after: i)
        }

        // Pad with CC4 to fill 20 positions
        while codeLen < 20 {
            for ch in yubinPtnC[3] {
                barsResult.append(Int(String(ch))!)
            }
            chkStr += yubinPtnC[3]
            codeLen += 1
        }

        // Check digit calculation (modulo 19)
        var chkSum = 0
        let chkChars = Array(chkStr)
        var j = 0
        while j <= chkChars.count - 3 {
            let pat3 = String(chkChars[j..<j+3])
            let value = yubinPatternToCheckValue(pat3)
            if value >= 0 {
                chkSum += value
            }
            j += 3
        }

        var chkD = 19 - (chkSum % 19)
        if chkD == 19 { chkD = 0 }
        checkDigitValue = chkD

        // Check digit bars
        let chkPattern = yubinGetCheckPattern(chkD)
        for ch in chkPattern {
            barsResult.append(Int(String(ch))!)
        }

        // Stop code
        for ch in yubinStop {
            barsResult.append(Int(String(ch))!)
        }

        self.bars = barsResult
        return barsResult
    }

    /// Draws the YubinCustomer barcode (4-state bars).
    public func draw(code: String, height: Int) throws {
        let barsResult = try encode(code)
        if barsResult.isEmpty {
            throw BarcodeError.encodingFailed("empty bar pattern")
        }

        // Calculate width from bar count: each bar + gap
        let width = barsResult.count * 4 // Default width based on bars

        if isSVGOutput() {
            try drawSVG4State(barsResult, width: width, height: height)
        } else {
            try drawPNG4State(barsResult, width: width, height: height)
        }
    }

    /// Draws with explicit width.
    public func draw(code: String, width: Int, height: Int) throws {
        let barsResult = try encode(code)
        if barsResult.isEmpty {
            throw BarcodeError.encodingFailed("empty bar pattern")
        }

        if isSVGOutput() {
            try drawSVG4State(barsResult, width: width, height: height)
        } else {
            try drawPNG4State(barsResult, width: width, height: height)
        }
    }
}
