import Foundation

// MARK: - UPC-E Constants

private let upceOE: [Int] = [0x00, 0x0b, 0x0d, 0x0e, 0x13, 0x19, 0x1c, 0x15, 0x16, 0x1a]

private let upceLeft: [[Int]] = [
    [0x27, 0x33, 0x1b, 0x21, 0x1d, 0x39, 0x05, 0x11, 0x09, 0x17], // G-pattern
    [0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b], // L-pattern
]

// MARK: - UPC-E Helper Functions

/// Expands a 7-digit UPC-E code (with leading 0) to an 11-digit UPC-A code (without check digit).
private func expandToUPCA(_ code: String) -> String? {
    guard code.count == 7 else { return nil }
    let chars = Array(code)
    let c0 = chars[0]
    let c6 = chars[6]

    guard c0 == "0" else { return nil }

    switch c6 {
    case "0", "1", "2":
        // code[:3] + c6 + "0000" + code[3:6]
        return String(chars[0..<3]) + String(c6) + "0000" + String(chars[3..<6])
    case "3":
        // code[:4] + "00000" + code[4:6]
        return String(chars[0..<4]) + "00000" + String(chars[4..<6])
    case "4":
        // code[:5] + "00000" + code[5:6]
        return String(chars[0..<5]) + "00000" + String(chars[5..<6])
    case "5", "6", "7", "8", "9":
        // code[:6] + "0000" + c6
        return String(chars[0..<6]) + "0000" + String(c6)
    default:
        return nil
    }
}

// MARK: - Check Digit

/// Computes the UPC-E check digit via UPC-A expansion.
public func calculateCheckDigitUPCE(_ src: String) -> String {
    var expanded: String
    if src.count == 7 {
        guard let exp = expandToUPCA(src) else { return "" }
        expanded = exp
    } else {
        expanded = src
    }

    let chars = Array(expanded)
    var odds = 0
    var evens = 0
    var odd = true
    for i in stride(from: chars.count, through: 1, by: -1) {
        let n = Int(chars[i - 1].asciiValue!) - 48
        if odd {
            odds += n
        } else {
            evens += n
        }
        odd = !odd
    }

    let s = odds * 3 + evens
    var cd = s % 10
    if cd != 0 {
        cd = 10 - cd
    }
    return String(cd)
}

// MARK: - UPCE Class

/// UPC-E barcode encoder.
public class UPCE: BarcodeBase1D, Encoder1D {
    public var extendedGuard: Bool = true

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    /// Returns the bar/space width pattern for UPC-E.
    public func encode(_ code: String) throws -> [Int] {
        guard !code.isEmpty else {
            throw BarcodeError.emptyString
        }

        var displayCode = code
        if displayCode.count == 6 {
            displayCode = "0" + displayCode
            displayCode += calculateCheckDigitUPCE(displayCode)
        } else if displayCode.count == 7 {
            displayCode += calculateCheckDigitUPCE(displayCode)
        } else if displayCode.count != 8 {
            throw BarcodeError.invalidInput("UPC-E requires 6, 7 or 8 digits")
        }

        for c in displayCode {
            guard c >= "0" && c <= "9" else {
                throw BarcodeError.invalidInput("UPC-E barcode requires numeric digits only")
            }
        }

        return try encodeUPCE(displayCode)
    }

    /// Internal encoding of 8-digit UPC-E code.
    private func encodeUPCE(_ src: String) throws -> [Int] {
        guard src.count == 8 else {
            throw BarcodeError.encodingFailed("EncodeUPCE: code must be 8 digits")
        }

        let chars = Array(src)
        let cd = Int(chars[7].asciiValue!) - 48
        let data = Array(chars[1..<7])

        let pre = cd
        let chkOE = 0x20

        // Build module sequence
        var modules = [Int]()

        // Start guard "101"
        modules.append(contentsOf: [1, 0, 1])

        // Data 6 digits
        for i in 0..<6 {
            let d = Int(data[i].asciiValue!) - 48
            let flgOE = (upceOE[pre] & (chkOE >> i)) != 0 ? 1 : 0
            let bits = upceLeft[flgOE][d]
            let mask = 0x40

            for j in 0..<7 {
                if (bits & (mask >> j)) != 0 {
                    modules.append(1)
                } else {
                    modules.append(0)
                }
            }
        }

        // End guard "010101"
        for j in 0..<6 {
            if j % 2 == 0 {
                modules.append(0)
            } else {
                modules.append(1)
            }
        }

        // Run-length encode
        guard !modules.isEmpty, modules[0] == 1 else {
            throw BarcodeError.encodingFailed("EncodeUPCE: internal module sequence error")
        }

        var pattern = [Int]()
        var current = modules[0]
        var count = 1

        for k in 1..<modules.count {
            if modules[k] == current {
                count += 1
            } else {
                pattern.append(count)
                current = modules[k]
                count = 1
            }
        }
        pattern.append(count)

        return pattern
    }

    /// Draws the UPC-E barcode with extended guard bars.
    public func draw(code: String, width: Int, height: Int) throws {
        let pattern = try encode(code)

        // Get display code
        var displayCode = code
        if displayCode.count == 6 {
            displayCode = "0" + displayCode
            displayCode += calculateCheckDigitUPCE(displayCode)
        } else if displayCode.count == 7 {
            displayCode += calculateCheckDigitUPCE(displayCode)
        }

        if isSVGOutput() {
            if showText && extendedGuard {
                drawSVGUPCE(pattern, displayCode: displayCode, width: width, height: height)
                return
            }
            let display = showText ? displayCode : ""
            try drawSVGBars(pattern, width: width, height: height, text: display)
        } else {
            try drawPNGJANUPC(pattern, displayCode: displayCode, width: width, height: height, clsName: "UPCE")
        }
    }
}
