import Foundation

// MARK: - UPC-A Constants

private let upcaLeftPatterns: [Int] = [0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b]
private let upcaRightPatterns: [Int] = [0x72, 0x66, 0x6c, 0x42, 0x5c, 0x4e, 0x50, 0x44, 0x48, 0x74]

// MARK: - Check Digit

/// Computes the UPC-A check digit (Modulus 10, weight 3-1).
public func calculateCheckDigitUPCA(_ src: String) -> String {
    let chars = Array(src)
    var total = 0
    for i in 0..<11 {
        let d = Int(chars[i].asciiValue!) - 48 // '0' = 48
        if i % 2 == 0 {
            total += d * 3
        } else {
            total += d
        }
    }
    let cd = (10 - (total % 10)) % 10
    return String(cd)
}

// MARK: - UPCA Class

/// UPC-A barcode encoder.
public class UPCA: BarcodeBase1D, Encoder1D {
    public var extendedGuard: Bool = true

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    /// Returns the bar/space width pattern for UPC-A.
    public func encode(_ code: String) throws -> [Int] {
        guard !code.isEmpty else {
            throw BarcodeError.emptyString
        }
        for c in code {
            guard c >= "0" && c <= "9" else {
                throw BarcodeError.invalidInput("UPC-A barcode requires numeric digits only")
            }
        }

        var s = code
        if s.count == 12 {
            let prefix11 = String(s.prefix(11))
            let expected = calculateCheckDigitUPCA(prefix11)
            let lastIdx = s.index(s.startIndex, offsetBy: 11)
            guard String(s[lastIdx...]) == expected else {
                throw BarcodeError.invalidInput("invalid check digit")
            }
        } else if s.count == 11 {
            s = s + calculateCheckDigitUPCA(s)
        } else {
            throw BarcodeError.invalidInput("UPC-A barcode requires 11 or 12 digits")
        }

        let chars = Array(s)

        var result = [Int]()
        // Start guard (101)
        result.append(contentsOf: [1, 1, 1])

        // Left 6 digits (L-pattern)
        for i in 0..<6 {
            let digit = Int(chars[i].asciiValue!) - 48
            let pattern = upcaLeftPatterns[digit]
            result.append(contentsOf: bitsToRunlength(pattern: pattern, numBits: 7))
        }

        // Center guard (01010)
        result.append(contentsOf: [1, 1, 1, 1, 1])

        // Right 6 digits (R-pattern)
        for i in 0..<6 {
            let digit = Int(chars[6 + i].asciiValue!) - 48
            let pattern = upcaRightPatterns[digit]
            result.append(contentsOf: bitsToRunlength(pattern: pattern, numBits: 7))
        }

        // End guard (101)
        result.append(contentsOf: [1, 1, 1])

        return result
    }

    /// Draws the UPC-A barcode with extended guard bars.
    public func draw(code: String, width: Int, height: Int) throws {
        let pattern = try encode(code)

        var s = code
        if s.count == 11 {
            s = s + calculateCheckDigitUPCA(s)
        }

        if isSVGOutput() {
            if showText && extendedGuard {
                drawSVGUPCA(pattern, displayCode: s, width: width, height: height)
                return
            }
            let display = showText ? s : ""
            try drawSVGBars(pattern, width: width, height: height, text: display)
        } else {
            try drawPNGJANUPC(pattern, displayCode: s, width: width, height: height, clsName: "UPCA")
        }
    }
}
