import Foundation

// MARK: - 4-state barcode rendering (YubinCustomer)

extension BarcodeBase1D {
    /// Renders a 4-state postal barcode as SVG.
    func drawSVG4State(_ bars: [Int], width: Int, height: Int) throws {
        if bars.isEmpty {
            throw BarcodeError.encodingFailed("empty bar pattern")
        }
        let n = bars.count
        let pitch = Double(width) / Double(n)
        let barW = max(1.0, (pitch * 0.45).rounded())
        let h3 = Double(height) / 3.0

        svgBegin(width, height)
        svgRect(0, 0, Double(width), Double(height), backColor)

        for (i, bt) in bars.enumerated() {
            let bx = (Double(i) * pitch + (pitch - barW) / 2.0).rounded()
            var by: Double, bh: Double
            switch bt {
            case 1: by = 0; bh = Double(height)
            case 2: by = 0; bh = (h3 * 2).rounded()
            case 3: by = h3.rounded(); bh = (h3 * 2).rounded()
            case 4: by = h3.rounded(); bh = h3.rounded()
            default: continue
            }
            svgRect(bx, by, barW, bh, foreColor)
        }

        svgEnd()
    }

    /// Renders a 4-state postal barcode as PNG.
    func drawPNG4State(_ bars: [Int], width: Int, height: Int) throws {
        if bars.isEmpty {
            throw BarcodeError.encodingFailed("empty bar pattern")
        }
        let n = bars.count

        let img = PixelBuffer(width: width, height: height)
        img.fill(backColor)

        let pitch = Double(width) / Double(n)
        let barW = max(1, Int((pitch * 0.45).rounded()))
        let h3 = Double(height) / 3.0

        for (i, bt) in bars.enumerated() {
            let bx = Int((Double(i) * pitch + (pitch - Double(barW)) / 2.0).rounded())
            var by: Int, bh: Int
            switch bt {
            case 1: by = 0; bh = height
            case 2: by = 0; bh = Int((h3 * 2).rounded())
            case 3: by = Int(h3.rounded()); bh = Int((h3 * 2).rounded())
            case 4: by = Int(h3.rounded()); bh = Int(h3.rounded())
            default: continue
            }
            img.fillRect(bx, by, bx + barW, by + bh, foreColor)
        }

        if isTrialMode() {
            drawSampleOverlayPNG(img, x: 0, y: 0, width: width, height: height)
        }

        try encodeImageBuffer(img)
    }
}

// MARK: - 2D barcode rendering

extension BarcodeBase2D {
    /// Renders a 2D matrix barcode as SVG.
    func drawSVG2D(_ matrix: [[Bool]], width: Int, height: Int) throws {
        let rows = matrix.count
        guard rows > 0 else { throw BarcodeError.encodingFailed("empty matrix") }
        let cols = matrix[0].count
        guard cols > 0 else { throw BarcodeError.encodingFailed("empty matrix") }

        svgBegin(width, height)
        svgRect(0, 0, Double(width), Double(height), backColor)

        let cellW = Double(width) / Double(cols)
        let cellH = Double(height) / Double(rows)

        for row in 0..<rows {
            for col in 0..<cols {
                if matrix[row][col] {
                    let x = (Double(col) * cellW).rounded()
                    let y = (Double(row) * cellH).rounded()
                    let w = ((Double(col + 1) * cellW).rounded() - x)
                    let h = ((Double(row + 1) * cellH).rounded() - y)
                    svgRect(x, y, w, h, foreColor)
                }
            }
        }

        svgEnd()
    }

    /// Renders a 2D matrix barcode as PNG.
    func drawPNG2D(_ matrix: [[Bool]], width: Int, height: Int) throws {
        let rows = matrix.count
        guard rows > 0 else { throw BarcodeError.encodingFailed("empty matrix") }
        let cols = matrix[0].count
        guard cols > 0 else { throw BarcodeError.encodingFailed("empty matrix") }

        let img = PixelBuffer(width: width, height: height)
        img.fill(backColor)

        let cellW = Double(width) / Double(cols)
        let cellH = Double(height) / Double(rows)

        for row in 0..<rows {
            for col in 0..<cols {
                if matrix[row][col] {
                    let x1 = Int((Double(col) * cellW).rounded())
                    let y1 = Int((Double(row) * cellH).rounded())
                    let x2 = Int((Double(col + 1) * cellW).rounded())
                    let y2 = Int((Double(row + 1) * cellH).rounded())
                    img.fillRect(x1, y1, x2, y2, foreColor)
                }
            }
        }

        if isTrialMode() {
            drawSampleOverlayPNG(img, x: 0, y: 0, width: width, height: height)
        }

        try encodeImageBuffer(img)
    }
}
