import Foundation
import PNG

/// Simple pixel buffer for NRGBA images.
class PixelBuffer {
    let width: Int
    let height: Int
    var pixels: [UInt8] // RGBA, 4 bytes per pixel

    init(width: Int, height: Int) {
        self.width = width
        self.height = height
        self.pixels = [UInt8](repeating: 0, count: width * height * 4)
    }

    func fill(_ c: RGBA) {
        for i in stride(from: 0, to: pixels.count, by: 4) {
            pixels[i]     = c.r
            pixels[i + 1] = c.g
            pixels[i + 2] = c.b
            pixels[i + 3] = c.a
        }
    }

    func setPixel(_ x: Int, _ y: Int, _ c: RGBA) {
        guard x >= 0 && x < width && y >= 0 && y < height else { return }
        let i = (y * width + x) * 4
        pixels[i]     = c.r
        pixels[i + 1] = c.g
        pixels[i + 2] = c.b
        pixels[i + 3] = c.a
    }

    func fillRect(_ x1: Int, _ y1: Int, _ x2: Int, _ y2: Int, _ c: RGBA) {
        let minX = max(0, min(x1, x2))
        let maxX = min(width, max(x1, x2))
        let minY = max(0, min(y1, y2))
        let maxY = min(height, max(y1, y2))
        for py in minY..<maxY {
            let rowBase = py * width * 4
            for px in minX..<maxX {
                let i = rowBase + px * 4
                pixels[i]     = c.r
                pixels[i + 1] = c.g
                pixels[i + 2] = c.b
                pixels[i + 3] = c.a
            }
        }
    }

    /// Encodes the pixel buffer as PNG.
    func encodePNG() -> [UInt8]? {
        // Convert RGBA pixels to swift-png format
        let rgba: [PNG.RGBA<UInt8>] = stride(from: 0, to: pixels.count, by: 4).map { i in
            PNG.RGBA(pixels[i], pixels[i+1], pixels[i+2], pixels[i+3])
        }

        let image = PNG.Image(packing: rgba,
                              size: (x: width, y: height),
                              layout: .init(format: .rgba8(palette: [], fill: nil)))

        // Encode to memory
        var output = [UInt8]()
        do {
            var stream = ArrayStream()
            try image.compress(stream: &stream, level: 9)
            output = stream.data
        } catch {
            return nil
        }
        return output.isEmpty ? nil : output
    }

    /// Encodes the pixel buffer as JPEG (simplified - outputs PNG since pure Swift JPEG encoding is complex).
    /// For JPEG, we use a minimal JFIF encoder.
    func encodeJPEG(quality: Int = 95) -> [UInt8]? {
        // Simple JPEG encoding using basic JFIF format
        return encodeSimpleJPEG(quality: quality)
    }
}

/// In-memory PNG stream for swift-png.
struct ArrayStream: PNG.BytestreamDestination {
    var data: [UInt8] = []

    mutating func write(_ buffer: [UInt8]) -> Void? {
        data.append(contentsOf: buffer)
        return ()
    }
}

// MARK: - Simple JPEG Encoder

extension PixelBuffer {
    /// Minimal baseline JPEG encoder (quality 75-95).
    func encodeSimpleJPEG(quality: Int) -> [UInt8]? {
        // For simplicity, generate an uncompressed JFIF or use PNG as fallback
        // In practice, barcode users primarily use PNG or SVG
        // We implement a basic baseline JPEG encoder

        var jpeg = [UInt8]()

        // Standard quantization tables scaled by quality
        let scaleFactor = quality < 50 ? (5000 / max(quality, 1)) : (200 - 2 * min(quality, 100))

        let baseLumQ: [Int] = [
            16,11,10,16,24,40,51,61, 12,12,14,19,26,58,60,55,
            14,13,16,24,40,57,69,56, 14,17,22,29,51,87,80,62,
            18,22,37,56,68,109,103,77, 24,35,55,64,81,104,113,92,
            49,64,78,87,103,121,120,101, 72,92,95,98,112,100,103,99
        ]
        let baseChromQ: [Int] = [
            17,18,24,47,99,99,99,99, 18,21,26,66,99,99,99,99,
            24,26,56,99,99,99,99,99, 47,66,99,99,99,99,99,99,
            99,99,99,99,99,99,99,99, 99,99,99,99,99,99,99,99,
            99,99,99,99,99,99,99,99, 99,99,99,99,99,99,99,99
        ]

        let lumQ = baseLumQ.map { UInt8(max(1, min(255, ($0 * scaleFactor + 50) / 100))) }
        let chromQ = baseChromQ.map { UInt8(max(1, min(255, ($0 * scaleFactor + 50) / 100))) }

        // SOI
        jpeg.append(contentsOf: [0xFF, 0xD8])

        // APP0 (JFIF)
        jpeg.append(contentsOf: [0xFF, 0xE0])
        let app0: [UInt8] = [0x00, 0x10, 0x4A, 0x46, 0x49, 0x46, 0x00, 0x01, 0x01, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00]
        jpeg.append(contentsOf: app0)

        // DQT - Luminance
        jpeg.append(contentsOf: [0xFF, 0xDB, 0x00, 0x43, 0x00])
        jpeg.append(contentsOf: lumQ)

        // DQT - Chrominance
        jpeg.append(contentsOf: [0xFF, 0xDB, 0x00, 0x43, 0x01])
        jpeg.append(contentsOf: chromQ)

        // SOF0
        jpeg.append(contentsOf: [0xFF, 0xC0])
        let h1 = UInt8(height >> 8), h2 = UInt8(height & 0xFF)
        let w1 = UInt8(width >> 8), w2 = UInt8(width & 0xFF)
        jpeg.append(contentsOf: [0x00, 0x11, 0x08, h1, h2, w1, w2, 0x03])
        jpeg.append(contentsOf: [0x01, 0x11, 0x00]) // Y component
        jpeg.append(contentsOf: [0x02, 0x11, 0x01]) // Cb component
        jpeg.append(contentsOf: [0x03, 0x11, 0x01]) // Cr component

        // DHT tables (standard Huffman tables)
        appendStandardHuffmanTables(&jpeg)

        // SOS
        jpeg.append(contentsOf: [0xFF, 0xDA])
        jpeg.append(contentsOf: [0x00, 0x0C, 0x03])
        jpeg.append(contentsOf: [0x01, 0x00]) // Y: DC=0, AC=0
        jpeg.append(contentsOf: [0x02, 0x11]) // Cb: DC=1, AC=1
        jpeg.append(contentsOf: [0x03, 0x11]) // Cr: DC=1, AC=1
        jpeg.append(contentsOf: [0x00, 0x3F, 0x00]) // Spectral selection

        // Encode MCU data
        var bits = BitWriter()
        var prevDCY = 0, prevDCCb = 0, prevDCCr = 0

        let zigzag: [Int] = [
            0,1,8,16,9,2,3,10, 17,24,32,25,18,11,4,5,
            12,19,26,33,40,48,41,34, 27,20,13,6,7,14,21,28,
            35,42,49,56,57,50,43,36, 29,22,15,23,30,37,44,51,
            58,59,52,45,38,31,39,46, 53,60,61,54,47,55,62,63
        ]

        let mcuW = (width + 7) / 8
        let mcuH = (height + 7) / 8

        for mcuY in 0..<mcuH {
            for mcuX in 0..<mcuW {
                // Extract 8x8 blocks for Y, Cb, Cr
                var blockY = [Double](repeating: 0, count: 64)
                var blockCb = [Double](repeating: 0, count: 64)
                var blockCr = [Double](repeating: 0, count: 64)

                for by in 0..<8 {
                    for bx in 0..<8 {
                        let px = min(mcuX * 8 + bx, width - 1)
                        let py = min(mcuY * 8 + by, height - 1)
                        let i = (py * width + px) * 4
                        let r = Double(pixels[i])
                        let g = Double(pixels[i + 1])
                        let b = Double(pixels[i + 2])
                        blockY[by * 8 + bx]  =  0.299 * r + 0.587 * g + 0.114 * b - 128.0
                        blockCb[by * 8 + bx] = -0.1687 * r - 0.3313 * g + 0.5 * b
                        blockCr[by * 8 + bx] =  0.5 * r - 0.4187 * g - 0.0813 * b
                    }
                }

                // DCT + quantize + encode each component
                prevDCY = encodeBlock(&bits, block: blockY, prevDC: prevDCY,
                                      quantTable: lumQ, zigzag: zigzag, isDC_lum: true)
                prevDCCb = encodeBlock(&bits, block: blockCb, prevDC: prevDCCb,
                                       quantTable: chromQ, zigzag: zigzag, isDC_lum: false)
                prevDCCr = encodeBlock(&bits, block: blockCr, prevDC: prevDCCr,
                                       quantTable: chromQ, zigzag: zigzag, isDC_lum: false)
            }
        }

        bits.flush()
        // Byte-stuff the scan data (escape 0xFF with 0x00)
        for byte in bits.bytes {
            jpeg.append(byte)
            if byte == 0xFF {
                jpeg.append(0x00)
            }
        }

        // EOI
        jpeg.append(contentsOf: [0xFF, 0xD9])

        return jpeg
    }

    private func fdct(_ block: inout [Double]) {
        // Simple 8x8 forward DCT
        let c1 = 0.980785280
        let c2 = 0.923879533
        let c3 = 0.831469612
        let c4 = 0.707106781
        let c5 = 0.555570233
        let c6 = 0.382683432
        let c7 = 0.195090322

        // Row DCT
        for i in 0..<8 {
            let off = i * 8
            let s07 = block[off] + block[off + 7]
            let d07 = block[off] - block[off + 7]
            let s16 = block[off + 1] + block[off + 6]
            let d16 = block[off + 1] - block[off + 6]
            let s25 = block[off + 2] + block[off + 5]
            let d25 = block[off + 2] - block[off + 5]
            let s34 = block[off + 3] + block[off + 4]
            let d34 = block[off + 3] - block[off + 4]

            let s0734 = s07 + s34
            let s1625 = s16 + s25
            block[off]     = (s0734 + s1625) * 0.5
            block[off + 4] = (s0734 - s1625) * 0.5
            block[off + 2] = ((s07 - s34) * c6 + (s16 - s25) * c2) * 0.5
            block[off + 6] = ((s07 - s34) * c2 - (s16 - s25) * c6) * 0.5

            block[off + 1] = (d07 * c1 + d16 * c3 + d25 * c5 + d34 * c7) * 0.5
            block[off + 3] = (d07 * c3 - d16 * c7 - d25 * c1 - d34 * c5) * 0.5
            block[off + 5] = (d07 * c5 - d16 * c1 + d25 * c7 + d34 * c3) * 0.5
            block[off + 7] = (d07 * c7 - d16 * c5 + d25 * c3 - d34 * c1) * 0.5
        }

        // Column DCT
        for i in 0..<8 {
            let s07 = block[i] + block[56 + i]
            let d07 = block[i] - block[56 + i]
            let s16 = block[8 + i] + block[48 + i]
            let d16 = block[8 + i] - block[48 + i]
            let s25 = block[16 + i] + block[40 + i]
            let d25 = block[16 + i] - block[40 + i]
            let s34 = block[24 + i] + block[32 + i]
            let d34 = block[24 + i] - block[32 + i]

            let s0734 = s07 + s34
            let s1625 = s16 + s25
            block[i]      = (s0734 + s1625) * 0.5
            block[32 + i] = (s0734 - s1625) * 0.5
            block[16 + i] = ((s07 - s34) * c6 + (s16 - s25) * c2) * 0.5
            block[48 + i] = ((s07 - s34) * c2 - (s16 - s25) * c6) * 0.5

            block[8 + i]  = (d07 * c1 + d16 * c3 + d25 * c5 + d34 * c7) * 0.5
            block[24 + i] = (d07 * c3 - d16 * c7 - d25 * c1 - d34 * c5) * 0.5
            block[40 + i] = (d07 * c5 - d16 * c1 + d25 * c7 + d34 * c3) * 0.5
            block[56 + i] = (d07 * c7 - d16 * c5 + d25 * c3 - d34 * c1) * 0.5
        }
    }

    private func encodeBlock(_ bits: inout BitWriter, block: [Double], prevDC: Int,
                              quantTable: [UInt8], zigzag: [Int], isDC_lum: Bool) -> Int {
        var dctBlock = block
        fdct(&dctBlock)

        // Quantize
        var quantized = [Int](repeating: 0, count: 64)
        for i in 0..<64 {
            quantized[i] = Int(round(dctBlock[zigzag[i]] / Double(quantTable[i])))
        }

        // Encode DC coefficient
        let dc = quantized[0]
        let dcDiff = dc - prevDC
        let (dcSize, dcBits) = encodeDCValue(dcDiff)

        if isDC_lum {
            bits.writeHuffman(dcLumSizeCode[dcSize], dcLumSizeBits[dcSize])
        } else {
            bits.writeHuffman(dcChromSizeCode[dcSize], dcChromSizeBits[dcSize])
        }
        if dcSize > 0 {
            bits.writeBits(dcBits, dcSize)
        }

        // Encode AC coefficients
        var zeroRun = 0
        for i in 1..<64 {
            let ac = quantized[i]
            if ac == 0 {
                zeroRun += 1
            } else {
                while zeroRun >= 16 {
                    // ZRL (zero run length 16)
                    if isDC_lum {
                        bits.writeHuffman(acLumCode[0xF0], acLumBits[0xF0])
                    } else {
                        bits.writeHuffman(acChromCode[0xF0], acChromBits[0xF0])
                    }
                    zeroRun -= 16
                }
                let (acSize, acBits) = encodeDCValue(ac)
                let rs = zeroRun * 16 + acSize
                if isDC_lum {
                    bits.writeHuffman(acLumCode[rs], acLumBits[rs])
                } else {
                    bits.writeHuffman(acChromCode[rs], acChromBits[rs])
                }
                bits.writeBits(acBits, acSize)
                zeroRun = 0
            }
        }

        if zeroRun > 0 {
            // EOB
            if isDC_lum {
                bits.writeHuffman(acLumCode[0x00], acLumBits[0x00])
            } else {
                bits.writeHuffman(acChromCode[0x00], acChromBits[0x00])
            }
        }

        return dc
    }

    private func encodeDCValue(_ value: Int) -> (size: Int, bits: Int) {
        if value == 0 { return (0, 0) }
        let absVal = abs(value)
        var size = 0
        var tmp = absVal
        while tmp > 0 { size += 1; tmp >>= 1 }
        let bits = value > 0 ? value : value + (1 << size) - 1
        return (size, bits)
    }
}

// MARK: - Bit Writer

struct BitWriter {
    var bytes = [UInt8]()
    var currentByte: UInt32 = 0
    var bitCount: Int = 0

    mutating func writeBits(_ value: Int, _ numBits: Int) {
        var v = UInt32(value & ((1 << numBits) - 1))
        var remaining = numBits
        while remaining > 0 {
            let available = 8 - bitCount
            if remaining >= available {
                currentByte = (currentByte << available) | (v >> (remaining - available))
                v &= (1 << (remaining - available)) - 1
                remaining -= available
                bytes.append(UInt8(currentByte & 0xFF))
                currentByte = 0
                bitCount = 0
            } else {
                currentByte = (currentByte << remaining) | v
                bitCount += remaining
                remaining = 0
            }
        }
    }

    mutating func writeHuffman(_ code: UInt16, _ length: UInt8) {
        writeBits(Int(code), Int(length))
    }

    mutating func flush() {
        if bitCount > 0 {
            currentByte <<= (8 - bitCount)
            bytes.append(UInt8(currentByte & 0xFF))
            currentByte = 0
            bitCount = 0
        }
    }
}

// MARK: - Standard JPEG Huffman Tables

// DC Luminance
private let dcLumSizeCode: [UInt16] = [0x00, 0x02, 0x03, 0x04, 0x05, 0x06, 0x0E, 0x1E, 0x3E, 0x7E, 0xFE, 0x1FE]
private let dcLumSizeBits: [UInt8]  = [2, 3, 3, 3, 3, 3, 4, 5, 6, 7, 8, 9]

// DC Chrominance
private let dcChromSizeCode: [UInt16] = [0x00, 0x01, 0x02, 0x06, 0x0E, 0x1E, 0x3E, 0x7E, 0xFE, 0x1FE, 0x3FE, 0x7FE]
private let dcChromSizeBits: [UInt8]  = [2, 2, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11]

// AC Luminance codes (162 entries for RRRRSSSS combinations)
private let acLumCode: [UInt16] = buildACLumCodes()
private let acLumBits: [UInt8] = buildACLumBits()
// AC Chrominance codes
private let acChromCode: [UInt16] = buildACChromCodes()
private let acChromBits: [UInt8] = buildACChromBits()

private func buildACLumCodes() -> [UInt16] {
    var codes = [UInt16](repeating: 0, count: 256)
    // Standard JPEG AC luminance Huffman table values
    let lengths: [Int] = [0,2,1,3,3,2,4,3,5,5,4,4,0,0,1,0x7D]
    let values: [UInt8] = [
        0x01,0x02,0x03,0x00,0x04,0x11,0x05,0x12,0x21,0x31,0x41,0x06,0x13,0x51,0x61,
        0x07,0x22,0x71,0x14,0x32,0x81,0x91,0xA1,0x08,0x23,0x42,0xB1,0xC1,0x15,0x52,
        0xD1,0xF0,0x24,0x33,0x62,0x72,0x82,0x09,0x0A,0x16,0x17,0x18,0x19,0x1A,0x25,
        0x26,0x27,0x28,0x29,0x2A,0x34,0x35,0x36,0x37,0x38,0x39,0x3A,0x43,0x44,0x45,
        0x46,0x47,0x48,0x49,0x4A,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x63,0x64,
        0x65,0x66,0x67,0x68,0x69,0x6A,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A,0x83,
        0x84,0x85,0x86,0x87,0x88,0x89,0x8A,0x92,0x93,0x94,0x95,0x96,0x97,0x98,0x99,
        0x9A,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,0xA8,0xA9,0xAA,0xB2,0xB3,0xB4,0xB5,0xB6,
        0xB7,0xB8,0xB9,0xBA,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,0xC8,0xC9,0xCA,0xD2,0xD3,
        0xD4,0xD5,0xD6,0xD7,0xD8,0xD9,0xDA,0xE1,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,0xE8,
        0xE9,0xEA,0xF1,0xF2,0xF3,0xF4,0xF5,0xF6,0xF7,0xF8,0xF9,0xFA
    ]
    var code: UInt16 = 0
    var vi = 0
    for bits in 1...16 {
        let count = bits <= lengths.count ? lengths[bits - 1] : 0
        for _ in 0..<count {
            if vi < values.count {
                codes[Int(values[vi])] = code
                vi += 1
            }
            code += 1
        }
        code <<= 1
    }
    return codes
}

private func buildACLumBits() -> [UInt8] {
    var bits = [UInt8](repeating: 0, count: 256)
    let lengths: [Int] = [0,2,1,3,3,2,4,3,5,5,4,4,0,0,1,0x7D]
    let values: [UInt8] = [
        0x01,0x02,0x03,0x00,0x04,0x11,0x05,0x12,0x21,0x31,0x41,0x06,0x13,0x51,0x61,
        0x07,0x22,0x71,0x14,0x32,0x81,0x91,0xA1,0x08,0x23,0x42,0xB1,0xC1,0x15,0x52,
        0xD1,0xF0,0x24,0x33,0x62,0x72,0x82,0x09,0x0A,0x16,0x17,0x18,0x19,0x1A,0x25,
        0x26,0x27,0x28,0x29,0x2A,0x34,0x35,0x36,0x37,0x38,0x39,0x3A,0x43,0x44,0x45,
        0x46,0x47,0x48,0x49,0x4A,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x63,0x64,
        0x65,0x66,0x67,0x68,0x69,0x6A,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A,0x83,
        0x84,0x85,0x86,0x87,0x88,0x89,0x8A,0x92,0x93,0x94,0x95,0x96,0x97,0x98,0x99,
        0x9A,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,0xA8,0xA9,0xAA,0xB2,0xB3,0xB4,0xB5,0xB6,
        0xB7,0xB8,0xB9,0xBA,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,0xC8,0xC9,0xCA,0xD2,0xD3,
        0xD4,0xD5,0xD6,0xD7,0xD8,0xD9,0xDA,0xE1,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,0xE8,
        0xE9,0xEA,0xF1,0xF2,0xF3,0xF4,0xF5,0xF6,0xF7,0xF8,0xF9,0xFA
    ]
    var vi = 0
    for b in 1...16 {
        let count = b <= lengths.count ? lengths[b - 1] : 0
        for _ in 0..<count {
            if vi < values.count {
                bits[Int(values[vi])] = UInt8(b)
                vi += 1
            }
        }
    }
    return bits
}

private func buildACChromCodes() -> [UInt16] {
    var codes = [UInt16](repeating: 0, count: 256)
    let lengths: [Int] = [0,2,1,2,4,4,3,4,7,5,4,4,0,1,2,0x77]
    let values: [UInt8] = [
        0x00,0x01,0x02,0x03,0x11,0x04,0x05,0x21,0x31,0x06,0x12,0x41,0x51,0x07,0x61,
        0x71,0x13,0x22,0x32,0x81,0x08,0x14,0x42,0x91,0xA1,0xB1,0xC1,0x09,0x23,0x33,
        0x52,0xF0,0x15,0x62,0x72,0xD1,0x0A,0x16,0x24,0x34,0xE1,0x25,0xF1,0x17,0x18,
        0x19,0x1A,0x26,0x27,0x28,0x29,0x2A,0x35,0x36,0x37,0x38,0x39,0x3A,0x43,0x44,
        0x45,0x46,0x47,0x48,0x49,0x4A,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x63,
        0x64,0x65,0x66,0x67,0x68,0x69,0x6A,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A,
        0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8A,0x92,0x93,0x94,0x95,0x96,0x97,
        0x98,0x99,0x9A,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,0xA8,0xA9,0xAA,0xB2,0xB3,0xB4,
        0xB5,0xB6,0xB7,0xB8,0xB9,0xBA,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,0xC8,0xC9,0xCA,
        0xD2,0xD3,0xD4,0xD5,0xD6,0xD7,0xD8,0xD9,0xDA,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,
        0xE8,0xE9,0xEA,0xF2,0xF3,0xF4,0xF5,0xF6,0xF7,0xF8,0xF9,0xFA
    ]
    var code: UInt16 = 0
    var vi = 0
    for b in 1...16 {
        let count = b <= lengths.count ? lengths[b - 1] : 0
        for _ in 0..<count {
            if vi < values.count {
                codes[Int(values[vi])] = code
                vi += 1
            }
            code += 1
        }
        code <<= 1
    }
    return codes
}

private func buildACChromBits() -> [UInt8] {
    var bits = [UInt8](repeating: 0, count: 256)
    let lengths: [Int] = [0,2,1,2,4,4,3,4,7,5,4,4,0,1,2,0x77]
    let values: [UInt8] = [
        0x00,0x01,0x02,0x03,0x11,0x04,0x05,0x21,0x31,0x06,0x12,0x41,0x51,0x07,0x61,
        0x71,0x13,0x22,0x32,0x81,0x08,0x14,0x42,0x91,0xA1,0xB1,0xC1,0x09,0x23,0x33,
        0x52,0xF0,0x15,0x62,0x72,0xD1,0x0A,0x16,0x24,0x34,0xE1,0x25,0xF1,0x17,0x18,
        0x19,0x1A,0x26,0x27,0x28,0x29,0x2A,0x35,0x36,0x37,0x38,0x39,0x3A,0x43,0x44,
        0x45,0x46,0x47,0x48,0x49,0x4A,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x63,
        0x64,0x65,0x66,0x67,0x68,0x69,0x6A,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A,
        0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8A,0x92,0x93,0x94,0x95,0x96,0x97,
        0x98,0x99,0x9A,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,0xA8,0xA9,0xAA,0xB2,0xB3,0xB4,
        0xB5,0xB6,0xB7,0xB8,0xB9,0xBA,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,0xC8,0xC9,0xCA,
        0xD2,0xD3,0xD4,0xD5,0xD6,0xD7,0xD8,0xD9,0xDA,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,
        0xE8,0xE9,0xEA,0xF2,0xF3,0xF4,0xF5,0xF6,0xF7,0xF8,0xF9,0xFA
    ]
    var vi = 0
    for b in 1...16 {
        let count = b <= lengths.count ? lengths[b - 1] : 0
        for _ in 0..<count {
            if vi < values.count {
                bits[Int(values[vi])] = UInt8(b)
                vi += 1
            }
        }
    }
    return bits
}

private func appendStandardHuffmanTables(_ jpeg: inout [UInt8]) {
    // DC Luminance DHT
    let dcLumLengths: [UInt8] = [0,1,5,1,1,1,1,1,1,0,0,0,0,0,0,0]
    let dcLumValues: [UInt8] = [0,1,2,3,4,5,6,7,8,9,10,11]
    writeDHT(&jpeg, tableClass: 0, tableId: 0, lengths: dcLumLengths, values: dcLumValues)

    // AC Luminance DHT
    let acLumLengths: [UInt8] = [0,2,1,3,3,2,4,3,5,5,4,4,0,0,1,0x7D]
    let acLumValues: [UInt8] = [
        0x01,0x02,0x03,0x00,0x04,0x11,0x05,0x12,0x21,0x31,0x41,0x06,0x13,0x51,0x61,
        0x07,0x22,0x71,0x14,0x32,0x81,0x91,0xA1,0x08,0x23,0x42,0xB1,0xC1,0x15,0x52,
        0xD1,0xF0,0x24,0x33,0x62,0x72,0x82,0x09,0x0A,0x16,0x17,0x18,0x19,0x1A,0x25,
        0x26,0x27,0x28,0x29,0x2A,0x34,0x35,0x36,0x37,0x38,0x39,0x3A,0x43,0x44,0x45,
        0x46,0x47,0x48,0x49,0x4A,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x63,0x64,
        0x65,0x66,0x67,0x68,0x69,0x6A,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A,0x83,
        0x84,0x85,0x86,0x87,0x88,0x89,0x8A,0x92,0x93,0x94,0x95,0x96,0x97,0x98,0x99,
        0x9A,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,0xA8,0xA9,0xAA,0xB2,0xB3,0xB4,0xB5,0xB6,
        0xB7,0xB8,0xB9,0xBA,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,0xC8,0xC9,0xCA,0xD2,0xD3,
        0xD4,0xD5,0xD6,0xD7,0xD8,0xD9,0xDA,0xE1,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,0xE8,
        0xE9,0xEA,0xF1,0xF2,0xF3,0xF4,0xF5,0xF6,0xF7,0xF8,0xF9,0xFA
    ]
    writeDHT(&jpeg, tableClass: 1, tableId: 0, lengths: acLumLengths, values: acLumValues)

    // DC Chrominance DHT
    let dcChromLengths: [UInt8] = [0,3,1,1,1,1,1,1,1,1,1,0,0,0,0,0]
    let dcChromValues: [UInt8] = [0,1,2,3,4,5,6,7,8,9,10,11]
    writeDHT(&jpeg, tableClass: 0, tableId: 1, lengths: dcChromLengths, values: dcChromValues)

    // AC Chrominance DHT
    let acChromLengths: [UInt8] = [0,2,1,2,4,4,3,4,7,5,4,4,0,1,2,0x77]
    let acChromValues: [UInt8] = [
        0x00,0x01,0x02,0x03,0x11,0x04,0x05,0x21,0x31,0x06,0x12,0x41,0x51,0x07,0x61,
        0x71,0x13,0x22,0x32,0x81,0x08,0x14,0x42,0x91,0xA1,0xB1,0xC1,0x09,0x23,0x33,
        0x52,0xF0,0x15,0x62,0x72,0xD1,0x0A,0x16,0x24,0x34,0xE1,0x25,0xF1,0x17,0x18,
        0x19,0x1A,0x26,0x27,0x28,0x29,0x2A,0x35,0x36,0x37,0x38,0x39,0x3A,0x43,0x44,
        0x45,0x46,0x47,0x48,0x49,0x4A,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x63,
        0x64,0x65,0x66,0x67,0x68,0x69,0x6A,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A,
        0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8A,0x92,0x93,0x94,0x95,0x96,0x97,
        0x98,0x99,0x9A,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,0xA8,0xA9,0xAA,0xB2,0xB3,0xB4,
        0xB5,0xB6,0xB7,0xB8,0xB9,0xBA,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,0xC8,0xC9,0xCA,
        0xD2,0xD3,0xD4,0xD5,0xD6,0xD7,0xD8,0xD9,0xDA,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,
        0xE8,0xE9,0xEA,0xF2,0xF3,0xF4,0xF5,0xF6,0xF7,0xF8,0xF9,0xFA
    ]
    writeDHT(&jpeg, tableClass: 1, tableId: 1, lengths: acChromLengths, values: acChromValues)
}

private func writeDHT(_ jpeg: inout [UInt8], tableClass: UInt8, tableId: UInt8,
                       lengths: [UInt8], values: [UInt8]) {
    let length = 2 + 1 + 16 + values.count
    jpeg.append(contentsOf: [0xFF, 0xC4])
    jpeg.append(UInt8(length >> 8))
    jpeg.append(UInt8(length & 0xFF))
    jpeg.append((tableClass << 4) | tableId)
    jpeg.append(contentsOf: lengths)
    jpeg.append(contentsOf: values)
}

// MARK: - BarcodeBase1D PNG Rendering Extension

extension BarcodeBase1D {
    /// Renders a generic 1D barcode to PNG/JPEG.
    func drawPNG1D(_ pattern: [Int], code: String, width: Int, height: Int) throws {
        let display = showText ? code : ""
        try renderBarsToPNG(pattern, width: width, height: height, text: display)
    }

    func renderBarsToPNG(_ pattern: [Int], width: Int, height: Int, text: String) throws {
        let totalUnits = pattern.reduce(0, +)
        if totalUnits <= 0 {
            throw BarcodeError.encodingFailed("pattern has zero total units")
        }

        let fontSize = max(8, Int(Double(height) * 0.15 * textFontScale))

        var textH = 0
        var displayText = text
        if !displayText.isEmpty {
            textH = Int(Double(fontSize) * 1.4)
        }
        var barH = height - textH
        if barH <= 0 {
            barH = height
            displayText = ""
        }

        let img = PixelBuffer(width: width, height: height)
        img.fill(backColor)

        let unitW = Double(width) / Double(totalUnits)
        var accum = 0.0
        var isBar = true
        for units in pattern {
            let x1 = Int((accum * unitW).rounded())
            accum += Double(units)
            let x2 = Int((accum * unitW).rounded())
            if isBar && x2 > x1 {
                img.fillRect(x1, 0, x2, barH, foreColor)
            }
            isBar = !isBar
        }

        // Draw text
        if !displayText.isEmpty {
            let textY = barH + 2
            drawTextOnImage(img, text: displayText, x: 0, y: textY, width: width, fontSize: fontSize)
        }

        // Trial mode watermark
        if isTrialMode() {
            drawSampleOverlayPNG(img, x: 0, y: 0, width: width, height: height)
        }

        try encodeImageBuffer(img)
    }

    func drawTextOnImage(_ img: PixelBuffer, text: String, x: Int, y: Int, width: Int, fontSize: Int) {
        guard let font = getDefaultFont() else { return }

        if textEvenSpacing {
            drawTextEvenPNG(img, font: font, text: text, x: Double(x), y: Double(y),
                           width: Double(width), fontSize: fontSize, color: foreColor,
                           hScale: textHorizontalSpacingScale)
        } else {
            drawTextCenteredPNG(img, font: font, text: text, cx: x + width / 2, y: y,
                               fontSize: fontSize, color: foreColor)
        }
    }
}

// MARK: - Text Drawing

func drawTextEvenPNG(_ img: PixelBuffer, font: TrueTypeFont, text: String, x: Double, y: Double,
                     width: Double, fontSize: Int, color: RGBA, hScale: Double) {
    let numChars = text.count
    if numChars == 0 { return }

    var marginRatio = 0.05
    if width < 100 { marginRatio = 0.03 }
    else if width > 400 { marginRatio = 0.07 }

    let horizontalMargin = width * marginRatio * hScale
    let textWidth = width - 2 * horizontalMargin

    var charSpacing = textWidth
    if numChars > 1 {
        charSpacing = textWidth / Double(numChars - 1)
    }

    let minSpacing = Double(fontSize) / 2
    if charSpacing < minSpacing { charSpacing = minSpacing }

    var totalTextWidth = 0.0
    if numChars > 1 {
        totalTextWidth = charSpacing * Double(numChars - 1)
    }
    let startX = x + (width - totalTextWidth) / 2

    let pixelSize = Double(fontSize)
    let ascent = font.ascent(pixelSize: pixelSize)
    let chars = Array(text)

    for i in 0..<numChars {
        let charX = startX + Double(i) * charSpacing
        drawCharCenteredPNG(img, font: font, ch: chars[i], cx: Int(charX.rounded()),
                           y: Int(y.rounded()), fontSize: fontSize, ascent: ascent, color: color)
    }
}

func drawTextCenteredPNG(_ img: PixelBuffer, font: TrueTypeFont, text: String, cx: Int, y: Int,
                         fontSize: Int, color: RGBA) {
    let pixelSize = Double(fontSize)
    let ascent = font.ascent(pixelSize: pixelSize)
    let scale = pixelSize / 1000.0 // approximate

    var totalWidth = 0.0
    for ch in text {
        let gi = font.glyphIndex(for: UInt16(ch.asciiValue ?? 0))
        let adv = font.advanceWidth(glyphIndex: gi)
        totalWidth += Double(adv) * pixelSize / Double(2048) // Roboto unitsPerEm = 2048
    }
    let startX = cx - Int(totalWidth / 2)

    var curX = Double(startX)
    for ch in text {
        let gi = font.glyphIndex(for: UInt16(ch.asciiValue ?? 0))
        let adv = font.advanceWidth(glyphIndex: gi)
        let advPx = Double(adv) * pixelSize / 2048.0

        if let glyph = font.rasterizeGlyph(glyphIndex: gi, pixelSize: pixelSize) {
            let gx = Int(curX.rounded()) + glyph.bearingX
            let gy = y + ascent - glyph.bearingY
            for gy2 in 0..<glyph.height {
                for gx2 in 0..<glyph.width {
                    let alpha = glyph.bitmap[gy2 * glyph.width + gx2]
                    if alpha > 0 {
                        img.setPixel(gx + gx2, gy + gy2,
                                    RGBA(color.r, color.g, color.b, alpha))
                    }
                }
            }
        }
        curX += advPx
    }
}

func drawCharCenteredPNG(_ img: PixelBuffer, font: TrueTypeFont, ch: Character, cx: Int, y: Int,
                         fontSize: Int, ascent: Int, color: RGBA) {
    let pixelSize = Double(fontSize)
    let gi = font.glyphIndex(for: UInt16(ch.asciiValue ?? 0))
    let adv = font.advanceWidth(glyphIndex: gi)
    let advPx = Double(adv) * pixelSize / 2048.0
    let startX = cx - Int(advPx / 2)

    if let glyph = font.rasterizeGlyph(glyphIndex: gi, pixelSize: pixelSize) {
        let gx = startX + glyph.bearingX
        let gy = y + ascent - glyph.bearingY
        for gy2 in 0..<glyph.height {
            for gx2 in 0..<glyph.width {
                let alpha = glyph.bitmap[gy2 * glyph.width + gx2]
                if alpha > 0 {
                    img.setPixel(gx + gx2, gy + gy2,
                                RGBA(color.r, color.g, color.b, alpha))
                }
            }
        }
    }
}

func drawSampleOverlayPNG(_ img: PixelBuffer, x: Int, y: Int, width: Int, height: Int) {
    var fontSize = Int(Double(height) * 0.12)
    if fontSize < 8 { fontSize = 8 }
    if fontSize > 40 { fontSize = 40 }
    var margin = Int(Double(height) * 0.01)
    if margin < 2 { margin = 2 }
    let text = getTrialText()
    let red = RGBA.red

    guard let font = getDefaultFont() else { return }
    let ascent = font.ascent(pixelSize: Double(fontSize))

    var curX = Double(x + margin)
    for ch in text {
        let gi = font.glyphIndex(for: UInt16(ch.asciiValue ?? 0))
        let adv = font.advanceWidth(glyphIndex: gi)
        let advPx = Double(adv) * Double(fontSize) / 2048.0

        if let glyph = font.rasterizeGlyph(glyphIndex: gi, pixelSize: Double(fontSize)) {
            let gx = Int(curX.rounded()) + glyph.bearingX
            let gy = y + margin + ascent - glyph.bearingY
            for gy2 in 0..<glyph.height {
                for gx2 in 0..<glyph.width {
                    let alpha = glyph.bitmap[gy2 * glyph.width + gx2]
                    if alpha > 0 {
                        img.setPixel(gx + gx2, gy + gy2, RGBA(red.r, red.g, red.b, alpha))
                    }
                }
            }
        }
        curX += advPx
    }
}

// MARK: - Image Encoding

extension BarcodeBase {
    /// Encodes a PixelBuffer as PNG or JPEG.
    func encodeImageBuffer(_ img: PixelBuffer) throws {
        if format == FORMAT_JPEG {
            guard let data = img.encodeJPEG() else {
                throw BarcodeError.renderFailed("JPEG encoding failed")
            }
            imageBuffer = data
        } else {
            guard let data = img.encodePNG() else {
                throw BarcodeError.renderFailed("PNG encoding failed")
            }
            imageBuffer = data
        }
    }
}
