import Foundation

// MARK: - Shared RSS/DataBar Utility Algorithms

/// Calculates nCr (binomial coefficient).
func getCombinations(_ n: Int, _ r: Int) -> Int {
    let minDenom: Int
    let maxDenom: Int
    if n - r > r {
        minDenom = r
        maxDenom = n - r
    } else {
        minDenom = n - r
        maxDenom = r
    }

    var val = 1
    var j = 1
    var i = n
    while i > maxDenom {
        val *= i
        if j <= minDenom {
            val /= j
            j += 1
        }
        i -= 1
    }
    while j <= minDenom {
        val /= j
        j += 1
    }
    return val
}

/// RSS subset width algorithm (ISO 24724).
/// Converts a value into an element width pattern.
func getWidths(_ val: Int, _ n: Int, _ elements: Int, _ maxWidth: Int, _ noNarrow: Int) -> [Int] {
    var widths = [Int](repeating: 0, count: elements)
    var narrowMask = 0
    var currentVal = val
    var currentN = n

    for bar in 0..<(elements - 1) {
        var elmWidth = 1
        narrowMask |= (1 << bar)
        var subVal = 0

        while true {
            subVal = getCombinations(currentN - elmWidth - 1, elements - bar - 2)

            if noNarrow == 0 && narrowMask == 0 &&
                currentN - elmWidth - (elements - bar - 1) >= elements - bar - 1 {
                subVal -= getCombinations(currentN - elmWidth - (elements - bar),
                                          elements - bar - 2)
            }

            if elements - bar - 1 > 1 {
                var lessVal = 0
                var mxwElement = currentN - elmWidth - (elements - bar - 2)
                while mxwElement > maxWidth {
                    lessVal += getCombinations(currentN - elmWidth - mxwElement - 1,
                                               elements - bar - 3)
                    mxwElement -= 1
                }
                subVal -= lessVal * (elements - 1 - bar)
            } else if currentN - elmWidth > maxWidth {
                subVal -= 1
            }

            currentVal -= subVal
            if currentVal < 0 {
                break
            }

            elmWidth += 1
            narrowMask &= ~(1 << bar)
        }

        currentVal += subVal
        currentN -= elmWidth
        widths[bar] = elmWidth
    }

    widths[elements - 1] = currentN
    return widths
}

/// Calculates GS1 GTIN-14 check digit from 13 digits.
func calculateGTINCheckDigit(_ src13: String) -> Int {
    let chars = Array(src13.utf8)
    var total = 0
    for i in 0..<13 {
        let weight = (i % 2 != 0) ? 1 : 3
        total += Int(chars[i] - UInt8(ascii: "0")) * weight
    }
    return (10 - (total % 10)) % 10
}

/// Converts a binary string to run-length encoded pattern string.
func bin2pat(_ binary: String) -> String {
    if binary.isEmpty { return "" }

    let chars = Array(binary.utf8)
    var result = [UInt8]()
    var count = 1
    var lastChar = chars[0]

    for i in 1..<chars.count {
        if chars[i] == lastChar {
            count += 1
        } else {
            result.append(UInt8(ascii: "0") + UInt8(count))
            count = 1
            lastChar = chars[i]
        }
    }
    result.append(UInt8(ascii: "0") + UInt8(count))

    return String(bytes: result, encoding: .ascii) ?? ""
}

/// Left-pads a string with '0' to the given length.
func padLeft(_ s: String, _ length: Int) -> String {
    var result = s
    while result.count < length {
        result = "0" + result
    }
    return result
}

/// Parses a numeric string to Int64.
func parseInt64(_ s: String) -> Int64 {
    var v: Int64 = 0
    for c in s.utf8 {
        v = v * 10 + Int64(c - UInt8(ascii: "0"))
    }
    return v
}
