import Foundation

// MARK: - QR Code Error Correction Level Constants

/// Low error correction (~7% recovery).
public let QR_ECC_L = 0
/// Medium error correction (~15% recovery).
public let QR_ECC_M = 1
/// Quartile error correction (~25% recovery).
public let QR_ECC_Q = 2
/// High error correction (~30% recovery).
public let QR_ECC_H = 3

// MARK: - QR Code Encode Mode Constants

private let qrModeNumeric      = "N"
private let qrModeAlphaNumeric = "A"
private let qrModeBinary       = "B"
private let qrModeKanji        = "K"

// MARK: - Internal Text Type Constants

private let qrTextAuto: Int         = 0
private let qrTextNumeric: Int      = 1
private let qrTextAlphaNumeric: Int = 2
private let qrTextKanji: Int        = 3
private let qrTextBinary: Int       = 4

// MARK: - Internal ECC Rate Constants

private let qrECCLow7: Int     = 0
private let qrECCMedium15: Int = 1
private let qrECCQual25: Int   = 2
private let qrECCHighQ30: Int  = 3

// MARK: - Lookup Tables

/// gDataWords[4][40]
private let gDataWords: [[Int]] = [
    [19, 34, 55, 80, 108, 68, 78, 97, 116, 68,
     81, 92, 107, 115, 87, 98, 107, 120, 113, 107,
     116, 111, 121, 117, 106, 114, 122, 117, 116, 115,
     115, 115, 115, 115, 121, 121, 122, 122, 117, 118],
    [16, 28, 44, 32, 43, 27, 31, 38, 36, 43,
     50, 36, 37, 40, 41, 45, 46, 43, 44, 41,
     42, 46, 47, 45, 47, 46, 45, 45, 45, 47,
     46, 46, 46, 46, 47, 47, 46, 46, 47, 47],
    [13, 22, 17, 24, 15, 19, 14, 18, 16, 19,
     22, 20, 20, 16, 24, 19, 22, 22, 21, 24,
     22, 24, 24, 24, 24, 22, 23, 24, 23, 24,
     24, 24, 24, 24, 24, 24, 24, 24, 24, 24],
    [9, 16, 13, 9, 11, 15, 13, 14, 12, 15,
     12, 14, 11, 12, 12, 15, 14, 14, 13, 15,
     16, 13, 15, 16, 15, 16, 15, 15, 15, 15,
     15, 15, 15, 16, 15, 15, 15, 15, 15, 15],
]

/// gExtraDataWords[4][40]
private let gExtraDataWords: [[Int]] = [
    [0, 0, 0, 0, 0, 0, 0, 0, 0, 2,
     0, 2, 0, 1, 1, 1, 5, 1, 4, 5,
     4, 7, 5, 4, 4, 2, 4, 10, 7, 10,
     3, 0, 1, 6, 7, 14, 4, 18, 4, 6],
    [0, 0, 0, 0, 0, 0, 0, 2, 2, 1,
     4, 2, 1, 5, 5, 3, 1, 4, 11, 13,
     0, 0, 14, 14, 13, 4, 3, 23, 7, 10,
     29, 23, 21, 23, 26, 34, 14, 32, 7, 31],
    [0, 0, 0, 0, 2, 0, 4, 2, 4, 2,
     4, 6, 4, 5, 7, 2, 15, 1, 4, 5,
     6, 16, 14, 16, 22, 6, 26, 31, 37, 25,
     1, 35, 19, 7, 14, 10, 10, 14, 22, 34],
    [0, 0, 0, 0, 2, 0, 1, 2, 4, 2,
     8, 4, 4, 5, 7, 13, 17, 19, 16, 10,
     6, 0, 14, 2, 13, 4, 28, 31, 26, 25,
     28, 35, 46, 1, 41, 64, 46, 32, 67, 61],
]

/// gECCWords[4][40]
private let gECCWords: [[Int]] = [
    [7, 10, 15, 20, 26, 18, 20, 24, 30, 18,
     20, 24, 26, 30, 22, 24, 28, 30, 28, 28,
     28, 28, 30, 30, 26, 28, 30, 30, 30, 30,
     30, 30, 30, 30, 30, 30, 30, 30, 30, 30],
    [10, 16, 26, 18, 24, 16, 18, 22, 22, 26,
     30, 22, 22, 24, 24, 28, 28, 26, 26, 26,
     26, 28, 28, 28, 28, 28, 28, 28, 28, 28,
     28, 28, 28, 28, 28, 28, 28, 28, 28, 28],
    [13, 22, 18, 26, 18, 24, 18, 22, 20, 24,
     28, 26, 24, 20, 30, 24, 28, 28, 26, 30,
     28, 30, 30, 30, 30, 28, 30, 30, 30, 30,
     30, 30, 30, 30, 30, 30, 30, 30, 30, 30],
    [17, 28, 22, 16, 22, 28, 26, 26, 24, 28,
     24, 28, 22, 24, 24, 30, 28, 28, 26, 28,
     30, 24, 30, 30, 30, 30, 30, 30, 30, 30,
     30, 30, 30, 30, 30, 30, 30, 30, 30, 30],
]

/// gRSBlockCount[4][40]
private let gRSBlockCount: [[Int]] = [
    [1, 1, 1, 1, 1, 2, 2, 2, 2, 4,
     4, 4, 4, 4, 6, 6, 6, 6, 7, 8,
     8, 9, 9, 10, 12, 12, 12, 13, 14, 15,
     16, 17, 18, 19, 19, 20, 21, 22, 24, 25],
    [1, 1, 1, 2, 2, 4, 4, 4, 5, 5,
     5, 8, 9, 9, 10, 10, 11, 13, 14, 16,
     17, 17, 18, 20, 21, 23, 25, 26, 28, 29,
     31, 33, 35, 37, 38, 40, 43, 45, 47, 49],
    [1, 1, 2, 2, 4, 4, 6, 6, 8, 8,
     8, 10, 12, 16, 12, 17, 16, 18, 21, 20,
     23, 23, 25, 27, 29, 34, 34, 35, 38, 40,
     43, 45, 48, 51, 53, 56, 59, 62, 65, 68],
    [1, 1, 2, 4, 4, 4, 5, 6, 8, 8,
     11, 11, 16, 16, 18, 16, 19, 21, 25, 25,
     25, 34, 30, 32, 35, 37, 40, 42, 45, 48,
     51, 54, 57, 60, 63, 66, 70, 74, 77, 81],
]

/// Alignment pattern positions by version (1..40)
private let qrAlignmentPatterns: [Int: [Int]] = [
    1: [6], 2: [6, 18], 3: [6, 22], 4: [6, 26], 5: [6, 30], 6: [6, 34],
    7: [6, 22, 38], 8: [6, 24, 42], 9: [6, 26, 46], 10: [6, 28, 50],
    11: [6, 30, 54], 12: [6, 32, 58], 13: [6, 34, 62],
    14: [6, 26, 46, 66], 15: [6, 26, 48, 70], 16: [6, 26, 50, 74],
    17: [6, 30, 54, 78], 18: [6, 30, 56, 82], 19: [6, 30, 58, 86],
    20: [6, 34, 62, 90],
    21: [6, 28, 50, 72, 94], 22: [6, 26, 50, 74, 98],
    23: [6, 30, 54, 78, 102], 24: [6, 28, 54, 80, 106],
    25: [6, 32, 58, 84, 110], 26: [6, 30, 58, 86, 114],
    27: [6, 34, 62, 90, 118],
    28: [6, 26, 50, 74, 98, 122], 29: [6, 30, 54, 78, 102, 126],
    30: [6, 26, 52, 78, 104, 130], 31: [6, 30, 56, 82, 108, 134],
    32: [6, 34, 60, 86, 112, 138], 33: [6, 30, 58, 86, 114, 142],
    34: [6, 34, 62, 90, 118, 146],
    35: [6, 30, 54, 78, 102, 126, 150], 36: [6, 24, 50, 76, 102, 128, 154],
    37: [6, 28, 54, 80, 106, 132, 158], 38: [6, 32, 58, 84, 110, 136, 162],
    39: [6, 26, 54, 82, 110, 138, 166], 40: [6, 30, 58, 86, 114, 142, 170],
]

/// Version information bits (versions 7-40)
private let qrVersionInfo: [Int] = [
    31892, 34236, 39577, 42195, 48118, 51042, 55367, 58893,
    63784, 68472, 70749, 76311, 79154, 84390, 87683, 92361,
    96236, 102084, 102881, 110507, 110734, 117786, 119615, 126325,
    127568, 133589, 136944, 141498, 145311, 150283, 152622, 158308,
    161089, 167017,
]

/// GF(256) exp table
private let gfExp: [Int] = [
    1, 2, 4, 8, 16, 32, 64, 128, 29, 58, 116, 232, 205, 135, 19, 38, 76, 152, 45, 90,
    180, 117, 234, 201, 143, 3, 6, 12, 24, 48, 96, 192, 157, 39, 78, 156, 37, 74, 148, 53,
    106, 212, 181, 119, 238, 193, 159, 35, 70, 140, 5, 10, 20, 40, 80, 160, 93, 186, 105, 210,
    185, 111, 222, 161, 95, 190, 97, 194, 153, 47, 94, 188, 101, 202, 137, 15, 30, 60, 120, 240,
    253, 231, 211, 187, 107, 214, 177, 127, 254, 225, 223, 163, 91, 182, 113, 226, 217, 175, 67, 134,
    17, 34, 68, 136, 13, 26, 52, 104, 208, 189, 103, 206, 129, 31, 62, 124, 248, 237, 199, 147,
    59, 118, 236, 197, 151, 51, 102, 204, 133, 23, 46, 92, 184, 109, 218, 169, 79, 158, 33, 66,
    132, 21, 42, 84, 168, 77, 154, 41, 82, 164, 85, 170, 73, 146, 57, 114, 228, 213, 183, 115,
    230, 209, 191, 99, 198, 145, 63, 126, 252, 229, 215, 179, 123, 246, 241, 255, 227, 219, 171, 75,
    150, 49, 98, 196, 149, 55, 110, 220, 165, 87, 174, 65, 130, 25, 50, 100, 200, 141, 7, 14,
    28, 56, 112, 224, 221, 167, 83, 166, 81, 162, 89, 178, 121, 242, 249, 239, 195, 155, 43, 86,
    172, 69, 138, 9, 18, 36, 72, 144, 61, 122, 244, 245, 247, 243, 251, 235, 203, 139, 11, 22,
    44, 88, 176, 125, 250, 233, 207, 131, 27, 54, 108, 216, 173, 71, 142, 1,
]

/// GF(256) log table
private let gfLog: [Int] = [
    0, 0, 1, 25, 2, 50, 26, 198, 3, 223, 51, 238, 27, 104, 199, 75, 4, 100, 224, 14,
    52, 141, 239, 129, 28, 193, 105, 248, 200, 8, 76, 113, 5, 138, 101, 47, 225, 36, 15, 33,
    53, 147, 142, 218, 240, 18, 130, 69, 29, 181, 194, 125, 106, 39, 249, 185, 201, 154, 9, 120,
    77, 228, 114, 166, 6, 191, 139, 98, 102, 221, 48, 253, 226, 152, 37, 179, 16, 145, 34, 136,
    54, 208, 148, 206, 143, 150, 219, 189, 241, 210, 19, 92, 131, 56, 70, 64, 30, 66, 182, 163,
    195, 72, 126, 110, 107, 58, 40, 84, 250, 133, 186, 61, 202, 94, 155, 159, 10, 21, 121, 43,
    78, 212, 229, 172, 115, 243, 167, 87, 7, 112, 192, 247, 140, 128, 99, 13, 103, 74, 222, 237,
    49, 197, 254, 24, 227, 165, 153, 119, 38, 184, 180, 124, 17, 68, 146, 217, 35, 32, 137, 46,
    55, 63, 209, 91, 149, 188, 207, 205, 144, 135, 151, 178, 220, 252, 190, 97, 242, 86, 211, 171,
    20, 42, 93, 158, 132, 60, 57, 83, 71, 109, 65, 162, 31, 45, 67, 216, 183, 123, 164, 118,
    196, 23, 73, 236, 127, 12, 111, 246, 108, 161, 59, 82, 41, 157, 85, 170, 251, 96, 134, 177,
    187, 204, 62, 90, 203, 89, 95, 176, 156, 169, 160, 81, 11, 245, 22, 235, 122, 117, 44, 215,
    79, 174, 213, 233, 230, 231, 173, 232, 116, 214, 244, 234, 168, 80, 88, 175,
]

/// Generator polynomials keyed by ECC word count
private let qrGenPolys: [Int: [Int]] = [
    7:  [127, 122, 154, 164, 11, 68, 117],
    10: [216, 194, 159, 111, 199, 94, 95, 113, 157, 193],
    13: [137, 73, 227, 17, 177, 17, 52, 13, 46, 43, 83, 132, 120],
    15: [29, 196, 111, 163, 112, 74, 10, 105, 105, 139, 132, 151, 32, 134, 26],
    16: [59, 13, 104, 189, 68, 209, 30, 8, 163, 65, 41, 229, 98, 50, 36, 59],
    17: [119, 66, 83, 120, 119, 22, 197, 83, 249, 41, 143, 134, 85, 53, 125, 99, 79],
    18: [239, 251, 183, 113, 149, 175, 199, 215, 240, 220, 73, 82, 173, 75, 32, 67, 217, 146],
    20: [152, 185, 240, 5, 111, 99, 6, 220, 112, 150, 69, 36, 187, 22, 228, 198, 121, 121, 165, 174],
    22: [89, 179, 131, 176, 182, 244, 19, 189, 69, 40, 28, 137, 29, 123, 67, 253, 86, 218, 230, 26, 145, 245],
    24: [122, 118, 169, 70, 178, 237, 216, 102, 115, 150, 229, 73, 130, 72, 61, 43, 206, 1, 237, 247, 127, 217, 144, 117],
    26: [246, 51, 183, 4, 136, 98, 199, 152, 77, 56, 206, 24, 145, 40, 209, 117, 233, 42, 135, 68, 70, 144, 146, 77, 43, 94],
    28: [252, 9, 28, 13, 18, 251, 208, 150, 103, 174, 100, 41, 167, 12, 247, 56, 117, 119, 233, 127, 181, 100, 121, 147, 176, 74, 58, 197],
    30: [212, 246, 77, 73, 195, 192, 75, 98, 5, 70, 103, 177, 22, 217, 138, 51, 181, 246, 72, 25, 18, 46, 228, 74, 216, 195, 11, 106, 130, 150],
]

/// Capacity tables
private let qrArrayCap: [Int] = [
    0, 26, 44, 70, 100, 134, 172, 196, 242, 292,
    346, 404, 466, 532, 581, 655, 733, 815, 901, 991,
    1085, 1156, 1258, 1364, 1474, 1588, 1706, 1828, 1921, 2051,
    2185, 2323, 2465, 2611, 2761, 2876, 3034, 3196, 3362, 3532,
    3706,
]

private let qrArray2Extra: [Int] = [
    0, 0, 7, 7, 7, 7, 7, 0, 0, 0,
    0, 0, 0, 0, 3, 3, 3, 3, 3, 3,
    3, 4, 4, 4, 4, 4, 4, 4, 3, 3,
    3, 3, 3, 3, 3, 0, 0, 0, 0, 0,
    0,
]

/// qrArray3Cap[4][41] - bit capacity by ECC level and version
private let qrArray3Cap: [[Int]] = [
    [0, 152, 272, 440, 640, 864, 1088, 1248, 1552, 1856,
     2192, 2592, 2960, 3424, 3688, 4184, 4712, 5176, 5768, 6360,
     6888, 7456, 8048, 8752, 9392, 10208, 10960, 11744, 12248, 13048,
     13880, 14744, 15640, 16568, 17528, 18448, 19472, 20528, 21616, 22496,
     23648],
    [0, 128, 224, 352, 512, 688, 864, 992, 1232, 1456,
     1728, 2032, 2320, 2672, 2920, 3320, 3624, 4056, 4504, 5016,
     5352, 5712, 6256, 6880, 7312, 8000, 8496, 9024, 9544, 10136,
     10984, 11640, 12328, 13048, 13800, 14496, 15312, 15936, 16816, 17728,
     18672],
    [0, 104, 176, 272, 384, 496, 608, 704, 880, 1056,
     1232, 1440, 1648, 1952, 2088, 2360, 2600, 2936, 3176, 3560,
     3880, 4096, 4544, 4912, 5312, 5744, 6032, 6464, 6968, 7288,
     7880, 8264, 8920, 9368, 9848, 10288, 10832, 11408, 12016, 12656,
     13328],
    [0, 72, 128, 208, 288, 368, 480, 528, 688, 800,
     976, 1120, 1264, 1440, 1576, 1784, 2024, 2264, 2504, 2728,
     3080, 3248, 3536, 3712, 4112, 4304, 4768, 5024, 5288, 5608,
     5960, 6344, 6760, 7208, 7688, 7888, 8432, 8768, 9136, 9776,
     10208],
]

/// Format information strings (32 patterns)
private let qrFormatInfo: [String] = [
    "111011111000100", "111001011110011", "111110110101010", "111100010011101",
    "110011000101111", "110001100011000", "110110001000001", "110100101110110",
    "101010000010010", "101000100100101", "101111001111100", "101101101001011",
    "100010111111001", "100000011001110", "100111110010111", "100101010100000",
    "011010101011111", "011000001101000", "011111100110001", "011101000000110",
    "010010010110100", "010000110000011", "010111011011010", "010101111101101",
    "001011010001001", "001001110111110", "001110011100111", "001100111010000",
    "000011101100010", "000001001010101", "000110100001100", "000100000111011",
]

// MARK: - QRCode Public Class

/// QR Code barcode encoder.
public class QRCode: BarcodeBase2D {
    /// QR version (0 = auto, 1-40 for specific version).
    public var version: Int = 0
    /// Error correction level (QR_ECC_L, QR_ECC_M, QR_ECC_Q, QR_ECC_H).
    public var errorCorrectionLevel: Int = QR_ECC_M
    /// Mask pattern (-1 = auto, 0-7 for specific mask).
    public var maskPattern: Int = -1

    private var encodeMode: String = qrModeBinary

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    /// Sets the encoding mode ("N" = Numeric, "A" = AlphaNumeric, "B" = Binary, "K" = Kanji).
    public func setEncodeMode(_ mode: String) {
        switch mode {
        case qrModeNumeric, qrModeAlphaNumeric, qrModeBinary, qrModeKanji:
            encodeMode = mode
        default:
            encodeMode = qrModeBinary
        }
    }

    /// Returns the current encoding mode.
    public func getEncodeMode() -> String { encodeMode }

    /// Renders the QR code into the internal buffer.
    /// - Parameters:
    ///   - code: The string to encode.
    ///   - size: Width and height of the output image (square).
    public func draw(code: String, size: Int) throws {
        let data = Array(code.utf8)
        guard !data.isEmpty else {
            throw BarcodeError.emptyString
        }

        let jam = createQRJam()
        try jam.makeArr(data)
        let pattern = jam.patt
        guard !pattern.isEmpty else {
            throw BarcodeError.encodingFailed("QR encoding produced empty pattern")
        }

        if isSVGOutput() {
            try drawSVG2D(pattern, width: size, height: size)
        } else {
            try drawPNG2D(pattern, width: size, height: size)
        }
    }

    /// Generates and returns the QR code pattern as a 2D bool matrix.
    public func getPattern(code: String) throws -> [[Bool]] {
        let data = Array(code.utf8)
        guard !data.isEmpty else {
            throw BarcodeError.emptyString
        }

        let jam = createQRJam()
        try jam.makeArr(data)
        return jam.patt
    }

    private func createQRJam() -> QRJam {
        let jam = QRJam()
        jam.setEncodeMode(encodeMode)
        jam.setVersion(version)
        jam.setECCRate(errorCorrectionLevel)
        return jam
    }
}

// MARK: - QRJam Internal Engine

/// Internal QR code generation engine (port of Go qrJam / C++ QRJam).
private class QRJam {
    var lFormatFirstX:  [Int] = [0, 1, 2, 3, 4, 5, 7, 8, 8, 8, 8, 8, 8, 8, 8]
    var lFormatFirstY:  [Int] = [8, 8, 8, 8, 8, 8, 8, 8, 7, 5, 4, 3, 2, 1, 0]
    var lFormatSecondX: [Int] = Array(repeating: 0, count: 15)
    var lFormatSecondY: [Int] = Array(repeating: 0, count: 15)

    var lDeployTableX: [Int] = []
    var lDeployTableY: [Int] = []
    var lDeployOrderX: [Int] = []
    var lDeployOrderY: [Int] = []
    var lRSBlock: [Int] = []

    var lQRSetVersion: Int  = 0
    var lQRSetECCRate: Int  = qrECCMedium15
    var lQRSetTextType: Int = qrTextAuto
    var lQRQuitZone: Int    = 4

    var lQRInquireVersion: Int          = 0
    var lQRInquireTextType: Int         = 0
    var lQRInquireNumericBits: Int      = 0
    var lQRInquireAlphaNumericBits: Int = 0
    var lQRInquireKanjiBits: Int        = 0
    var lQRInquireBinaryBits: Int       = 0
    var lQRInquireModuleSize: Int       = 0

    var encodeMode: String = "Z"

    var patt: [[Bool]] = []

    // MARK: - Configuration

    func setEncodeMode(_ mode: String) {
        encodeMode = mode
        switch mode {
        case "N":
            lQRSetTextType = qrTextNumeric
        case "A":
            lQRSetTextType = qrTextAlphaNumeric
        default:
            lQRSetTextType = qrTextBinary
        }
    }

    func setVersion(_ version: Int) {
        if version >= 0 && version <= 40 {
            lQRSetVersion = version
        }
    }

    func setECCRate(_ rate: Int) {
        lQRSetECCRate = rate
    }

    // MARK: - Helper Functions

    private func qrGet1DAddress(_ px: Int, _ py: Int, _ symbolSize: Int) -> Int {
        return px + py * (symbolSize + 1)
    }

    private func qrCodeGetFormatX(_ pVersion: Int) -> [Int] {
        var arr = [Int](repeating: 0, count: 15)
        for i in 0..<7 {
            arr[i] = 8
        }
        for i in 0..<8 {
            arr[i + 7] = 4 * pVersion + 9 + i
        }
        return arr
    }

    private func qrCodeGetFormatY(_ pVersion: Int) -> [Int] {
        var arr = [Int](repeating: 0, count: 15)
        for i in 0..<8 {
            arr[i + 7] = 8
        }
        for i in 0..<7 {
            arr[i] = 4 * pVersion + 16 - i
        }
        return arr
    }

    // MARK: - Bit Inquiry

    private func qrCodeInquireBits(_ data: [UInt8]) {
        let length = data.count

        if encodeMode == "N" {
            let rem = length % 3
            if rem == 0 {
                lQRInquireNumericBits = Int((Double(length) / 3.0).rounded(.down) * 10.0)
            } else if rem == 1 {
                lQRInquireNumericBits = Int((Double(length) / 3.0).rounded(.down) * 10.0 + 4.0)
            } else if rem == 2 {
                lQRInquireNumericBits = Int((Double(length) / 3.0).rounded(.down) * 10.0 + 7.0)
            }
            lQRInquireAlphaNumericBits = 0
            lQRInquireKanjiBits = 0
        } else if encodeMode == "A" {
            let rem = length % 2
            if rem == 0 {
                lQRInquireAlphaNumericBits = Int((Double(length) / 2.0).rounded(.down) * 11.0)
            } else if rem == 1 {
                lQRInquireAlphaNumericBits = Int((Double(length) / 2.0).rounded(.down) * 11.0 + 6.0)
            }
            lQRInquireNumericBits = 0
            lQRInquireKanjiBits = 0
        } else {
            lQRInquireNumericBits = 0
            lQRInquireAlphaNumericBits = 0
            lQRInquireKanjiBits = 0
        }
        lQRInquireBinaryBits = length * 8
    }

    private func qrCodeSetInquireTextType(_ data: [UInt8]) -> Int {
        if encodeMode == "N" {
            return qrTextNumeric
        } else if encodeMode == "A" {
            return qrTextAlphaNumeric
        }
        return qrTextBinary
    }

    // MARK: - Symbol Data

    private func qrCodeGetSymbolData(_ pVersion: Int) -> [Int] {
        let num = pVersion * 4 + 17
        var array3 = [Int](repeating: 0, count: num * (num + 1))
        let array: [Int] = [3, num - 4]

        let ap = qrAlignmentPatterns[pVersion] ?? [0]

        let ga = qrGet1DAddress

        for jj in 0..<array.count {
            for kk in 0..<array.count {
                let num8 = array[jj]
                let num9 = array[kk]
                if jj == array.count - 1 && kk == array.count - 1 {
                    continue
                }

                // Center and immediate neighbors
                for dx in -1...1 {
                    for dy in -1...1 {
                        array3[ga(num8 + dx, num9 + dy, num)] = 3
                    }
                }

                // Outer ring
                for i in -3...3 {
                    array3[ga(num8 + i, num9 - 3, num)] = 3
                    array3[ga(num8 + i, num9 + 3, num)] = 3
                }
                for i in -2...2 {
                    array3[ga(num8 - 3, num9 + i, num)] = 3
                    array3[ga(num8 + 3, num9 + i, num)] = 3
                }

                // Inner ring
                for i in -2...2 {
                    array3[ga(num8 + i, num9 - 2, num)] = 2
                    array3[ga(num8 + i, num9 + 2, num)] = 2
                }
                for i in -1...1 {
                    array3[ga(num8 - 2, num9 + i, num)] = 2
                    array3[ga(num8 + 2, num9 + i, num)] = 2
                }

                // Border markings for specific corners
                if jj == 0 && kk == 0 {
                    for i in -3...4 {
                        array3[ga(num8 + 4, num9 + i, num)] = 8
                    }
                    for i in -3...4 {
                        array3[ga(num8 + i, num9 + 4, num)] = 8
                    }
                }

                if jj == 0 && kk == 1 {
                    for i in -4...3 {
                        array3[ga(num8 + 4, num9 + i, num)] = 8
                    }
                    for i in -4...3 {
                        array3[ga(num8 - i, num9 - 4, num)] = 8
                    }
                }

                if jj == 1 && kk == 0 {
                    for i in -3...4 {
                        array3[ga(num8 - 4, num9 + i, num)] = 8
                    }
                    for i in -4...3 {
                        array3[ga(num8 + i, num9 + 4, num)] = 8
                    }
                }
            }
        }

        // Alignment patterns
        let num16 = ap.count - 1
        for lIdx in 0...num16 {
            for mIdx in 0..<ap.count {
                let num18 = ap[lIdx]
                let num19 = ap[mIdx]
                if (lIdx == 0 && mIdx == 0) ||
                   (lIdx == 0 && mIdx == ap.count - 1) ||
                   (mIdx == 0 && lIdx == ap.count - 1) {
                    continue
                }

                array3[ga(num18, num19, num)] = 5
                // Outer ring
                for i in -2...2 {
                    array3[ga(num18 + i, num19 - 2, num)] = 5
                    array3[ga(num18 + i, num19 + 2, num)] = 5
                }
                for i in -1...1 {
                    array3[ga(num18 - 2, num19 + i, num)] = 5
                    array3[ga(num18 + 2, num19 + i, num)] = 5
                }
                // Inner ring
                for dx in -1...1 {
                    for dy in -1...1 {
                        if dx == 0 && dy == 0 {
                            continue
                        }
                        array3[ga(num18 + dx, num19 + dy, num)] = 4
                    }
                }
                array3[ga(num18, num19, num)] = 5 // center
            }
        }

        // Timing patterns
        let num20 = num - 9
        var n = 8
        while n <= num20 {
            array3[ga(n, 6, num)] |= 0x11
            array3[ga(6, n, num)] |= 0x11
            if n != num20 {
                array3[ga(n + 1, 6, num)] |= 0x10
                array3[ga(6, n + 1, num)] |= 0x10
            }
            n += 2
        }

        array3[ga(8, num - 8, num)] = 129

        // Version information
        if pVersion >= 7 {
            var num21 = qrVersionInfo[pVersion - 7]
            for num22 in 0..<6 {
                for num23 in 0..<3 {
                    let num24 = num21 & 1
                    if num24 == 1 {
                        array3[ga(num - 11 + num23, num22, num)] |= 0x21
                        array3[ga(num22, num - 11 + num23, num)] |= 0x21
                    } else {
                        array3[ga(num - 11 + num23, num22, num)] |= 0x20
                        array3[ga(num22, num - 11 + num23, num)] |= 0x20
                    }
                    num21 >>= 1
                }
            }
        }

        let fmtX = qrCodeGetFormatX(lQRInquireVersion)
        let fmtY = qrCodeGetFormatY(lQRInquireVersion)
        lFormatSecondX = fmtX
        lFormatSecondY = fmtY
        for num25 in 0..<15 {
            array3[ga(lFormatFirstX[num25], lFormatFirstY[num25], num)] |= 0x40
            array3[ga(lFormatSecondX[num25], lFormatSecondY[num25], num)] |= 0x40
        }

        return array3
    }

    // MARK: - RS Block Transformation

    private func qrCodeTransRSBlock(_ pDeployData: [Int]) -> [Int] {
        let eccIdx = lQRSetECCRate
        let verIdx = lQRInquireVersion - 1
        let num  = gDataWords[eccIdx][verIdx]
        let num2 = gExtraDataWords[eccIdx][verIdx]
        let num3 = Double(gECCWords[eccIdx][verIdx])
        let num4 = gRSBlockCount[eccIdx][verIdx]

        var arr = [Int](repeating: 0, count: pDeployData.count)
        var num5 = 0
        var i = 0
        var array2 = [Int](repeating: 0, count: 129)
        var num6 = 0

        for jj in 0..<num {
            for kk in 0..<num4 {
                var num9 = 0
                if kk > num4 - num2 {
                    num9 = kk - (num4 - num2)
                    if jj == 0 {
                        array2[num6] = jj + kk * num + num9 - 1
                        num6 += 1
                    }
                }
                for num10 in 0..<8 {
                    num5 = (jj + kk * num + num9) * 8 + num10
                    if num5 < arr.count && i < pDeployData.count {
                        arr[num5] = pDeployData[i]
                    }
                    i += 1
                }
            }
        }

        if num2 > 0 {
            array2[num6] = num * num4 + num2 - 1
            num6 += 1
        }

        for lIdx in 0..<num6 {
            for num12 in 0..<8 {
                num5 = array2[lIdx] * 8 + num12
                if num5 < arr.count && i < pDeployData.count {
                    arr[num5] = pDeployData[i]
                }
                i += 1
            }
        }

        let num13 = i
        let num14 = Int((num3 - 1.0).rounded())
        for m in 0...num14 {
            for n in 0..<num4 {
                for num16 in 0..<8 {
                    num5 = Int((Double(num13) + (Double(m) + Double(n) * num3) * 8.0 + Double(num16)).rounded())
                    if num5 < arr.count && i < pDeployData.count {
                        arr[num5] = pDeployData[i]
                    }
                    i += 1
                }
            }
        }

        while i < pDeployData.count {
            if i < arr.count {
                arr[i] = pDeployData[i]
            }
            i += 1
        }

        return arr
    }

    // MARK: - AlphaNumeric Code

    private func qrGetAlphaNumericCode(_ c: UInt8) -> Int {
        if c >= 0x30 && c <= 0x39 {
            return Int(c) - 0x30
        }
        if c >= 0x41 && c <= 0x5A {
            return Int(c) - 0x41 + 10
        }
        let mapping: [UInt8: Int] = [
            0x20: 36, 0x24: 37, 0x25: 38, 0x2A: 39, 0x2B: 40,
            0x2D: 41, 0x2E: 42, 0x2F: 43, 0x3A: 44,
        ]
        if let v = mapping[c] {
            return v
        }
        return 0
    }

    // MARK: - 8-Bit Data

    private func qrCodeGet8BitData(_ pInputData: [Int], _ pInputDataBits: [Int], _ pMaxBytes: Int) -> [Int] {
        var arr = [Int](repeating: 0, count: pMaxBytes)
        let numItems = pInputDataBits.count
        var num2 = 0
        var num3 = 8
        var num4 = 0

        for i in 0..<numItems {
            num4 += pInputDataBits[i]
        }

        let num6 = Int((Double(num4 - 1) / 8.0).rounded(.down))
        for jj in 0...num6 {
            if jj < arr.count {
                arr[jj] = 0
            }
        }

        for k in 0..<numItems {
            var num8 = pInputData[k]
            var num9 = pInputDataBits[k]
            if num9 == 0 {
                break
            }

            var flag = true
            while flag {
                if num3 > num9 {
                    arr[num2] = ((arr[num2] << (num9 & 7)) | num8) & 0xFF
                    num3 -= num9
                    flag = false
                } else {
                    num9 -= num3
                    arr[num2] = ((arr[num2] << (num3 & 7)) | (num8 >> num9)) & 0xFF
                    if num9 == 0 {
                        flag = false
                    } else {
                        num8 &= (1 << num9) - 1
                        flag = true
                    }
                    num2 += 1
                    num3 = 8
                }
            }
        }

        if num3 != 8 {
            arr[num2] = (arr[num2] << (num3 & 7)) & 0xFF
        } else {
            num2 -= 1
        }

        if num2 < pMaxBytes - 1 {
            var flag = true
            while num2 < pMaxBytes - 1 {
                num2 += 1
                if flag {
                    arr[num2] = 236
                } else {
                    arr[num2] = 17
                }
                flag = !flag
            }
        }

        return arr
    }

    // MARK: - Byte Data XOR

    private static func qrCodeGetByteData(_ pInput1: [Int], _ pInput2: [Int]) -> [Int] {
        let arr: [Int]
        let arr2: [Int]
        if pInput1.count > pInput2.count {
            arr = pInput1
            arr2 = pInput2
        } else {
            arr = pInput2
            arr2 = pInput1
        }

        let numA = arr.count
        let num2 = arr2.count
        var arr3 = [Int](repeating: 0, count: numA + 1)

        for i in 0..<numA {
            if i < num2 {
                arr3[i] = arr[i] ^ arr2[i]
            } else {
                arr3[i] = arr[i]
            }
        }

        return arr3
    }

    // MARK: - Total Data (with ECC)

    private func qrCodeGetTotalData(_ pInputData: [Int], _ pECC: Int, _ pData: Int, _ pTotal: Int) -> [Int] {
        let eccIdx = lQRSetECCRate
        let verIdx = lQRInquireVersion - 1
        let num  = gDataWords[eccIdx][verIdx]
        let num2 = gExtraDataWords[eccIdx][verIdx]
        let num3 = Double(gECCWords[eccIdx][verIdx])
        let num4 = gRSBlockCount[eccIdx][verIdx]

        let array4 = qrGenPolys[pECC] ?? []

        // Build multiplication table
        var arrTable = [[Int]](repeating: [Int](repeating: 0, count: array4.count), count: 256)
        for i in 0..<array4.count {
            let num6Val = gfLog[array4[i]]
            for num7 in 0..<256 {
                let num8 = (num6Val + gfLog[num7]) % 255
                let num9 = gfExp[num8]
                arrTable[num7][i] = num9
            }
        }

        // RS block sizes
        lRSBlock = [Int](repeating: 0, count: num4)
        for jj in 0..<num4 {
            if jj >= num4 - num2 {
                lRSBlock[jj] = Int((Double(num) + num3 + 1.0).rounded())
            } else {
                lRSBlock[jj] = Int((Double(num) + num3).rounded())
            }
        }

        // Split input data into RS blocks
        var array5 = [[Int]](repeating: [], count: lRSBlock.count)
        var array6 = [Int](repeating: 0, count: pTotal)
        for i in 0..<min(pInputData.count, pTotal) {
            array6[i] = pInputData[i]
        }

        for num10Idx in 0..<lRSBlock.count {
            for k in 0..<lRSBlock.count {
                let size = lRSBlock[num10Idx] - pECC
                array5[k] = [Int](repeating: 0, count: size + 1)
            }
        }

        var num11 = 0
        var num12 = 0
        for num10 in 0..<pData {
            if num12 < array5.count && num11 < array5[num12].count {
                array5[num12][num11] = pInputData[num10]
            }
            num11 += 1
            if num12 < lRSBlock.count && num11 >= lRSBlock[num12] - pECC {
                num11 = 0
                num12 += 1
            }
        }

        for num12Idx in 0..<lRSBlock.count {
            var array8 = array5[num12Idx]
            let num15 = lRSBlock[num12Idx]
            let num16 = num15 - pECC

            for _ in stride(from: num16, through: 1, by: -1) {
                let b = array8[0]
                if b != 0 {
                    var array9 = [Int](repeating: 0, count: array8.count - 1)
                    for idx in 0..<array9.count {
                        array9[idx] = array8[idx + 1]
                    }
                    var array10 = [Int](repeating: 0, count: pECC)
                    for lIdx in 0..<pECC {
                        array10[lIdx] = arrTable[b][lIdx]
                    }
                    array8 = QRJam.qrCodeGetByteData(array9, array10)
                } else if pECC < array8.count {
                    array8 = Array(array8[1...])
                } else {
                    var array12 = [Int](repeating: 0, count: pECC)
                    for idx in 0..<pECC {
                        if idx < array8.count - 1 {
                            array12[idx] = array8[idx + 1]
                        }
                    }
                    array12[pECC - 1] = 0
                    array8 = array12
                }
            }

            let destStart = pInputData.count + num12Idx * pECC
            for idx in 0..<pECC {
                if destStart + idx < array6.count && idx < array8.count {
                    array6[destStart + idx] = array8[idx]
                }
            }
        }

        return array6
    }

    // MARK: - Mask Functions

    private static func qrCodeGetMaskBit(_ maskNo: Int, _ py: Int, _ px: Int) -> Bool {
        switch maskNo {
        case 0:
            return (px + py) % 2 == 0
        case 1:
            return px % 2 == 0
        case 2:
            return py % 3 == 0
        case 3:
            return (px + py) % 3 == 0
        case 4:
            return (px / 2 + py / 3) % 2 == 0
        case 5:
            return (px * py) % 2 + (px * py) % 3 == 0
        case 6:
            return ((px * py) % 2 + (px * py) % 3) % 2 == 0
        case 7:
            return ((px * py) % 3 + (px + py) % 2) % 2 == 0
        default:
            return false
        }
    }

    private static func qrCodeGetMaskBitGroup(_ py: Int, _ px: Int) -> Int {
        var num = 0
        for m in 0..<8 {
            if qrCodeGetMaskBit(m, py, px) {
                num += 1 << m
            }
        }
        return num
    }

    // MARK: - Mask Selection

    private static func qrCodeChooseMaskNumber(_ pModuleData: [[Int]], _ pExtraBits: Int) -> Int {
        let length = pModuleData.count
        var array  = [Int](repeating: 0, count: 8)
        var array2 = [Int](repeating: 0, count: 8)
        var array3 = [Int](repeating: 0, count: 8)
        var array4 = [Int](repeating: 0, count: 8)
        var array5 = [Int](repeating: 0, count: 8)
        var array6 = [Int](repeating: 0, count: 8)
        var array7 = [Int](repeating: 0, count: 9)
        var array8 = [Int](repeating: 0, count: 9)
        var array9  = [Bool](repeating: false, count: 9)
        var array10 = [Bool](repeating: false, count: 9)

        for i in 0..<length {
            for n in 0..<8 {
                array7[n] = 0
                array8[n] = 0
                array9[n] = false
                array10[n] = false
            }

            for jj in 0..<length {
                var num = 0
                var num2 = 0
                if jj > 0 && i > 0 {
                    num = pModuleData[jj][i] & pModuleData[jj - 1][i] &
                          pModuleData[jj][i - 1] & pModuleData[jj - 1][i - 1]
                    num2 = pModuleData[jj][i] | pModuleData[jj - 1][i] |
                           pModuleData[jj][i - 1] | pModuleData[jj - 1][i - 1]
                }

                for m in 0..<8 {
                    array7[m] = ((array7[m] & 0x3F) << 1) | ((pModuleData[jj][i] >> (m & 7)) & 1)
                    array8[m] = ((array8[m] & 0x3F) << 1) | ((pModuleData[i][jj] >> (m & 7)) & 1)

                    if pModuleData[jj][i] & (1 << m) != 0 {
                        array4[m] += 1
                    }
                    if array7[m] == 0b1011101 {
                        array3[m] += 40
                    }
                    if array8[m] == 0b1011101 {
                        array3[m] += 40
                    }
                    if jj > 0 && i > 0 {
                        if (num & 1) != 0 || (num2 & 1) == 0 {
                            array2[m] += 3
                        }
                        num >>= 1
                        num2 >>= 1
                    }

                    if (array7[m] & 0x1F) == 0 || (array7[m] & 0x1F) == 31 {
                        if jj > 3 {
                            if array9[m] {
                                array[m] += 1
                            } else {
                                array[m] += 3
                                array9[m] = true
                            }
                        }
                    } else {
                        array9[m] = false
                    }

                    if (array8[m] & 0x1F) == 0 || (array8[m] & 0x1F) == 31 {
                        if jj > 3 {
                            if array10[m] {
                                array[m] += 1
                            } else {
                                array[m] += 3
                                array10[m] = true
                            }
                        }
                    } else {
                        array10[m] = false
                    }
                }
            }
        }

        var num8 = 0
        var result = 0
        let array11: [Int] = [90, 80, 70, 60, 50, 40, 30, 20, 10, 0, 0, 10, 20, 30, 40, 50, 60, 70, 80, 90, 90]

        for n in 0..<8 {
            var idx = Int((20.0 * Double(array4[n]) / Double(pExtraBits)).rounded())
            if idx < 0 { idx = 0 }
            if idx > 20 { idx = 20 }
            array5[n] = array11[idx]
            array6[n] = array[n] + array2[n] + array3[n] + array5[n]
        }

        for o in 0..<8 {
            if array6[o] < num8 || o == 0 {
                result = o
                num8 = array6[o]
            }
        }

        return result
    }

    // MARK: - Main Data Encoding

    private func qrGetData(_ data: [UInt8]) throws -> [[Int]] {
        let num4 = data.count
        qrCodeInquireBits(data)
        lQRInquireTextType = qrCodeSetInquireTextType(data)
        var qrTextTypes = lQRInquireTextType

        if qrTextTypes == qrTextKanji {
            qrTextTypes = qrTextBinary
            lQRInquireTextType = qrTextBinary
        }

        var array6 = [Int](repeating: 0, count: num4 + 32)
        var array7 = [Int](repeating: 0, count: num4 + 32)

        if num4 <= 0 {
            return [[0]]
        }

        array7[0] = 4
        var num = 0 // index into array6/array7

        // Version-dependent extra bits for character count indicator
        var array5Ver: [Int]

        if qrTextTypes == qrTextNumeric {
            array5Ver = [
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                2, 2, 2, 2, 2, 2, 2, 4, 4, 4,
                4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
            ]
            array6[num] = 1
            num += 1
            array6[num] = num4
            array7[num] = 10
            num += 1

            for jj in 0..<num4 {
                let b = Int(data[jj]) - 0x30
                if jj % 3 == 0 {
                    array6[num] = b
                    array7[num] = 4
                } else {
                    array6[num] = array6[num] * 10 + b
                    if jj % 3 == 1 {
                        array7[num] = 7
                    } else {
                        array7[num] = 10
                        if jj < num4 - 1 {
                            num += 1
                        }
                    }
                }
            }
            num += 1
        } else {
            // Binary / default
            array5Ver = [
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
                8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
                8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
            ]
            array6[num] = 4 // Binary mode indicator
            num += 1
            array6[num] = num4
            array7[num] = 8
            num += 1

            for i in 0..<num4 {
                array6[i + num] = Int(data[i]) & 0xFF
                array7[i + num] = 8
            }
            num += num4
        }

        // Calculate total bits
        var num2 = 0
        for lIdx in 0..<num {
            num2 += array7[lIdx]
        }

        // num5 is the index of the character count indicator
        let num5 = 1

        // Auto version selection
        lQRInquireVersion = 1
        var num12 = 1
        var num3 = 0
        while num12 <= 40 {
            if qrArray3Cap[lQRSetECCRate][num12] >= num2 + array5Ver[lQRInquireVersion] {
                num3 = qrArray3Cap[lQRSetECCRate][num12]
                break
            }
            lQRInquireVersion += 1
            num12 += 1
        }

        if lQRInquireVersion > 40 {
            throw BarcodeError.encodingFailed("data too long for QR Code")
        }

        // Version adjustment - BUG FIX: use lQRInquireVersion != 0 (always true after auto-detect)
        // When user specifies a version via lQRSetVersion, we must still update num3.
        // The fix: check lQRInquireVersion != 0 and handle both branches to update num3.
        if lQRInquireVersion != 0 {
            if lQRInquireVersion > lQRSetVersion {
                num3 = qrArray3Cap[lQRSetECCRate][lQRInquireVersion]
            } else {
                lQRInquireVersion = lQRSetVersion
                num3 = qrArray3Cap[lQRSetECCRate][lQRInquireVersion]
            }
        }

        num2 += array5Ver[lQRInquireVersion]
        array7[num5] = array7[num5] + array5Ver[lQRInquireVersion]
        let num14 = qrArrayCap[lQRInquireVersion]
        let num15 = qrArray2Extra[lQRInquireVersion] + (num14 << 3)

        let fmtX = qrCodeGetFormatX(lQRInquireVersion)
        let fmtY = qrCodeGetFormatY(lQRInquireVersion)
        lFormatSecondX = fmtX
        lFormatSecondY = fmtY

        let pECC = gECCWords[lQRSetECCRate][lQRInquireVersion - 1]
        let num16 = Int((Double(num3) / 8.0).rounded())
        let num17 = 4 * lQRInquireVersion + 17
        lQRInquireModuleSize = num17

        let array9 = qrCodeGetSymbolData(lQRInquireVersion)

        lDeployTableX = [Int](repeating: 0, count: num15 + 1)
        lDeployTableY = [Int](repeating: 0, count: num15 + 1)
        lDeployOrderX = [Int](repeating: 0, count: num15)
        lDeployOrderY = [Int](repeating: 0, count: num15)

        // Data placement order
        var num19 = 0
        var mVal = num17 - 1
        while mVal >= 3 {
            for nVal in stride(from: num17 - 1, through: 0, by: -1) {
                var num20 = nVal * (num17 + 1) + mVal
                if (array9[num20] & 0xFE) == 0 {
                    if num19 < lDeployTableX.count {
                        lDeployTableX[num19] = mVal
                        lDeployTableY[num19] = nVal
                        num19 += 1
                    }
                }
                num20 = nVal * (num17 + 1) + mVal - 1
                if (array9[num20] & 0xFE) == 0 {
                    if num19 < lDeployTableX.count {
                        lDeployTableX[num19] = mVal - 1
                        lDeployTableY[num19] = nVal
                        num19 += 1
                    }
                }
            }

            if mVal == 8 {
                mVal = 7
            }
            for num22 in 0..<num17 {
                var num23 = num22 * (num17 + 1) + mVal - 2
                if (array9[num23] & 0xFE) == 0 {
                    if num19 < lDeployTableX.count {
                        lDeployTableX[num19] = mVal - 2
                        lDeployTableY[num19] = num22
                        num19 += 1
                    }
                }
                num23 = num22 * (num17 + 1) + mVal - 3
                if (array9[num23] & 0xFE) == 0 {
                    if num19 < lDeployTableX.count {
                        lDeployTableX[num19] = mVal - 3
                        lDeployTableY[num19] = num22
                        num19 += 1
                    }
                }
            }
            mVal -= 4
        }

        lDeployOrderX = qrCodeTransRSBlock(lDeployTableX)
        lDeployOrderY = qrCodeTransRSBlock(lDeployTableY)

        // Terminator
        if num2 <= num3 - 4 {
            array6[num] = 0
            array7[num] = 4
        } else if num2 < num3 {
            array6[num] = 0
            array7[num] = num3 - num2
        }

        // Convert to 8-bit data
        let pInputData = qrCodeGet8BitData(array6, array7, num16)

        // Add error correction data
        let array10 = qrCodeGetTotalData(pInputData, pECC, num16, num14)

        // Initialize QR matrix
        var array11 = [[Int]](repeating: [Int](repeating: 0, count: num17), count: num17)

        // Place data into matrix
        for num29 in 0..<num14 {
            var b5 = array10[num29]
            for num30 in stride(from: 7, through: 0, by: -1) {
                let num31 = num29 * 8 + num30
                if num31 < lDeployOrderX.count && num31 < lDeployOrderY.count {
                    let ox = lDeployOrderX[num31]
                    let oy = lDeployOrderY[num31]
                    let num32 = QRJam.qrCodeGetMaskBitGroup(ox, oy)
                    if ox < num17 && oy < num17 {
                        array11[ox][oy] = (255 * (b5 & 1)) ^ num32
                    }
                }
                b5 >>= 1
            }
        }

        for num33 in stride(from: qrArray2Extra[lQRInquireVersion], through: 1, by: -1) {
            let num34 = num33 + num14 * 8 - 1
            if num34 < lDeployOrderX.count && num34 < lDeployOrderY.count {
                let ox = lDeployOrderX[num34]
                let oy = lDeployOrderY[num34]
                let num32 = QRJam.qrCodeGetMaskBitGroup(ox, oy)
                if ox < num17 && oy < num17 {
                    array11[ox][oy] = 0xFF ^ num32
                }
            }
        }

        // Choose mask pattern
        let b6 = QRJam.qrCodeChooseMaskNumber(
            array11, qrArray2Extra[lQRInquireVersion] + num14 * 8)
        let b7 = Int(pow(2.0, Double(b6)).rounded())
        let b8 = b6 + lQRSetECCRate * 8

        // Place format information
        for num35 in 0..<15 {
            let formatStr = qrFormatInfo[b8]
            let charIndex = formatStr.index(formatStr.startIndex, offsetBy: num35)
            let b9 = Int(String(formatStr[charIndex]))! // '0' or '1'
            array11[lFormatFirstX[num35]][lFormatFirstY[num35]] = b9 * 255
            array11[lFormatSecondX[num35]][lFormatSecondY[num35]] = b9 * 255
        }

        // Build final matrix
        var array12 = [[Int]](repeating: [Int](repeating: 0, count: num17), count: num17)
        var num36 = 0
        for num38 in 0..<num17 {
            for num40 in 0..<num17 {
                if (array11[num40][num38] & b7) != 0 || (array9[num36] & 1) == 1 {
                    array12[num40][num38] = 1 | (array9[num36] & 0xFE)
                } else {
                    array12[num40][num38] = 0 | (array9[num36] & 0xFE)
                }
                num36 += 1
            }
            num36 += 1
        }

        return array12
    }

    // MARK: - Make Pattern Array

    func makeArr(_ data: [UInt8]) throws {
        guard !data.isEmpty else {
            throw BarcodeError.encodingFailed("empty data")
        }

        let array = try qrGetData(data)
        let size = array.count
        guard size > 0 else {
            throw BarcodeError.encodingFailed("QR encoding produced empty array")
        }

        patt = [[Bool]](repeating: [Bool](repeating: false, count: size), count: size)
        for i in 0..<size {
            for jj in 0..<size {
                let b = array[i][jj]
                patt[i][jj] = (b & 1) == 1
            }
        }
    }
}
