import Foundation

// MARK: - PDF417 Public Constants

/// PDF417 error correction level constants.
public let PDF417_ERROR_AUTO   = -1
public let PDF417_ERROR_LEVEL0 = 0
public let PDF417_ERROR_LEVEL1 = 1
public let PDF417_ERROR_LEVEL2 = 2
public let PDF417_ERROR_LEVEL3 = 3
public let PDF417_ERROR_LEVEL4 = 4
public let PDF417_ERROR_LEVEL5 = 5
public let PDF417_ERROR_LEVEL6 = 6
public let PDF417_ERROR_LEVEL7 = 7
public let PDF417_ERROR_LEVEL8 = 8

/// PDF417 size mode constants.
public let PDF417_SIZE_AUTO             = 0
public let PDF417_SIZE_ROWS             = 1
public let PDF417_SIZE_COLUMNS          = 2
public let PDF417_SIZE_COLUMNS_AND_ROWS = 3

// MARK: - Internal Constants

private let pdf417StartPattern       = 0x1fea8
private let pdf417StopPattern        = 0x3fa29
private let pdf417StartCodeSize      = 17
private let pdf417StopSize           = 18
private let pdf417Mod                = 929
private let pdf417Alpha              = 0x10000
private let pdf417Lower              = 0x20000
private let pdf417Mixed              = 0x40000
private let pdf417Punctuation        = 0x80000
private let pdf417IsByte             = 0x100000
private let pdf417ByteShift          = 913
private let pdf417PL                 = 25
private let pdf417LL                 = 27
private let pdf417AS                 = 27
private let pdf417ML                 = 28
private let pdf417AL                 = 28
private let pdf417PS                 = 29
private let pdf417PAL                = 29
private let pdf417Space              = 26
private let pdf417TextMode           = 900
private let pdf417ByteMode6          = 924
private let pdf417ByteMode           = 901
private let pdf417NumericMode        = 902
private let pdf417AbsoluteMaxTextSize = 5420
private let pdf417MaxDataCodewords   = 926

// Internal size kind constants.
private let pdf417SizeAuto          = 0
private let pdf417SizeColumns       = 1
private let pdf417SizeRows          = 2
private let pdf417SizeColumnsAndRows = 3

private let pdf417MixedSet       = "0123456789&\r\t,:#-.$/+%*=^"
private let pdf417PunctuationSet = ";<>@[\\]_`~!\r\t,:\n-.$/\"|*()?{}'"

// MARK: - Internal Segment

private struct PDF417Segment {
    var segType: UInt8 // ASCII 'T', 'N', 'B', or 0
    var start: Int
    var end: Int
}

// MARK: - Internal Encoder

private class PDF417Encoder {
    var outBits: [UInt8] = []
    var bitPtr: Int = 0
    var bitColumns: Int = 0
    var errorLevel: Int = 2
    var lenCodewords: Int = 0
    var codeColumns: Int = 0
    var codeRows: Int = 0
    var useAutoErrorLevel: Bool = true
    var aspectRatio: Double = 0.5
    var yHeight: Int = 3
    var options: Int = 0
    var sizeKind: Int = 0
    var codewords: [Int] = []
    var text: [UInt8] = []
    var cwPtr: Int = 0
    var segmentList: [PDF417Segment] = []

    func initBlock() {
        codewords = [Int](repeating: 0, count: pdf417MaxDataCodewords + 2)
    }

    // MARK: - Bit Output

    func outCodeword17(_ codeword: Int) {
        var bytePtr = bitPtr >> 3
        let bit = bitPtr - bytePtr * 8
        outBits[bytePtr] |= UInt8(truncatingIfNeeded: (codeword >> (9 + bit)) & 0xFF)
        bytePtr += 1
        outBits[bytePtr] |= UInt8(truncatingIfNeeded: (codeword >> (1 + bit)) & 0xFF)
        bytePtr += 1
        let shifted = codeword << 8
        outBits[bytePtr] |= UInt8(truncatingIfNeeded: (shifted >> (1 + bit)) & 0xFF)
        bitPtr += 17
    }

    func outCodeword18(_ codeword: Int) {
        var bytePtr = bitPtr >> 3
        let bit = bitPtr - bytePtr * 8
        outBits[bytePtr] |= UInt8(truncatingIfNeeded: (codeword >> (10 + bit)) & 0xFF)
        bytePtr += 1
        outBits[bytePtr] |= UInt8(truncatingIfNeeded: (codeword >> (2 + bit)) & 0xFF)
        bytePtr += 1
        let shifted = codeword << 8
        outBits[bytePtr] |= UInt8(truncatingIfNeeded: (shifted >> (2 + bit)) & 0xFF)
        if bit == 7 {
            bytePtr += 1
            outBits[bytePtr] |= 0x80
        }
        bitPtr += 18
    }

    func outCodeword(_ codeword: Int) {
        outCodeword17(codeword)
    }

    func outStartPattern() {
        outCodeword17(pdf417StartPattern)
    }

    func outStopPattern() {
        outCodeword18(pdf417StopPattern)
    }

    // MARK: - Paint Code

    func outPaintCode() {
        bitColumns = pdf417StartCodeSize * (codeColumns + 3) + pdf417StopSize
        let bytesPerRow = (bitColumns - 1) / 8 + 1
        let lenBits = bytesPerRow * codeRows
        outBits = [UInt8](repeating: 0, count: lenBits)

        var codePtr = 0
        for row in 0..<codeRows {
            bitPtr = bytesPerRow * 8 * row
            let rowMod = row % 3
            let cluster = pdf417Clusters[rowMod]

            outStartPattern()

            // Left edge indicator
            var edge: Int
            switch rowMod {
            case 0:
                edge = 30 * (row / 3) + (codeRows - 1) / 3
            case 1:
                edge = 30 * (row / 3) + errorLevel * 3 + (codeRows - 1) % 3
            default:
                edge = 30 * (row / 3) + codeColumns - 1
            }
            outCodeword(cluster[edge])

            // Data codewords
            for _ in 0..<codeColumns {
                if codePtr >= lenCodewords {
                    break
                }
                let cw = codewords[codePtr]
                if cw < 0 || cw >= cluster.count {
                    break
                }
                outCodeword(cluster[cw])
                codePtr += 1
            }

            // Right edge indicator
            switch rowMod {
            case 0:
                edge = 30 * (row / 3) + codeColumns - 1
            case 1:
                edge = 30 * (row / 3) + (codeRows - 1) / 3
            default:
                edge = 30 * (row / 3) + errorLevel * 3 + (codeRows - 1) % 3
            }
            outCodeword(cluster[edge])

            outStopPattern()
        }

        if (options & 0x02) != 0 { // PDF417_INVERT_BITMAP
            for k in 0..<outBits.count {
                outBits[k] ^= 0xFF
            }
        }
    }

    // MARK: - Error Correction

    func calculateErrorCorrection(_ dest: Int) {
        var el = errorLevel
        if el < 0 || el > 8 {
            el = 0
        }
        let A = pdf417ErrorLevel[el]
        let ALength = 2 << el
        for k in 0..<ALength {
            codewords[dest + k] = 0
        }
        let lastE = ALength - 1
        for k in 0..<lenCodewords {
            let t1 = codewords[k] + codewords[dest]
            for e in 0..<ALength {
                let t2 = (t1 * A[lastE - e]) % pdf417Mod
                let t3 = pdf417Mod - t2
                var nxt = 0
                if e != lastE {
                    nxt = codewords[dest + e + 1]
                }
                codewords[dest + e] = (nxt + t3) % pdf417Mod
            }
        }
        for k in 0..<ALength {
            codewords[dest + k] = (pdf417Mod - codewords[dest + k]) % pdf417Mod
        }
    }

    // MARK: - Text Type and Value

    func getTextTypeAndValue(_ maxLength: Int, _ idx: Int) -> Int {
        if idx >= maxLength {
            return 0
        }
        let c = Int(text[idx]) & 0xFF
        let ch = Character(UnicodeScalar(c)!)

        if ch >= "A" && ch <= "Z" {
            return pdf417Alpha + c - Int(Character("A").asciiValue!)
        }
        if ch >= "a" && ch <= "z" {
            return pdf417Lower + c - Int(Character("a").asciiValue!)
        }
        if ch == " " {
            return pdf417Alpha + pdf417Lower + pdf417Mixed + pdf417Space
        }

        let ms = pdf417IndexOf(pdf417MixedSet, UInt8(c))
        let ps = pdf417IndexOf(pdf417PunctuationSet, UInt8(c))

        if ms < 0 && ps < 0 {
            return pdf417IsByte + c
        }
        if ms == ps {
            return pdf417Mixed + pdf417Punctuation + ms
        }
        if ms >= 0 {
            return pdf417Mixed + ms
        }
        return pdf417Punctuation + ps
    }

    // MARK: - Text Compaction

    func textCompaction(_ start: Int, _ length: Int) throws {
        var dest = [Int](repeating: 0, count: pdf417AbsoluteMaxTextSize * 2)
        var mode = pdf417Alpha
        var ptr = 0
        var fullBytes = 0
        let lengthEnd = length + start
        var k = start
        while k < lengthEnd {
            let v = getTextTypeAndValue(lengthEnd, k)
            if (v & mode) != 0 {
                dest[ptr] = v & 0xFF
                ptr += 1
                k += 1
                continue
            }
            if (v & pdf417IsByte) != 0 {
                if (ptr & 1) != 0 {
                    if (mode & pdf417Punctuation) != 0 {
                        dest[ptr] = pdf417PAL
                    } else {
                        dest[ptr] = pdf417PS
                    }
                    ptr += 1
                    if (mode & pdf417Punctuation) != 0 {
                        mode = pdf417Alpha
                    }
                }
                dest[ptr] = pdf417ByteShift
                ptr += 1
                dest[ptr] = v & 0xFF
                ptr += 1
                fullBytes += 2
                k += 1
                continue
            }

            if mode == pdf417Alpha {
                if (v & pdf417Lower) != 0 {
                    dest[ptr] = pdf417LL
                    ptr += 1
                    dest[ptr] = v & 0xFF
                    ptr += 1
                    mode = pdf417Lower
                } else if (v & pdf417Mixed) != 0 {
                    dest[ptr] = pdf417ML
                    ptr += 1
                    dest[ptr] = v & 0xFF
                    ptr += 1
                    mode = pdf417Mixed
                } else if (getTextTypeAndValue(lengthEnd, k + 1) &
                           getTextTypeAndValue(lengthEnd, k + 2) & pdf417Punctuation) != 0 {
                    dest[ptr] = pdf417ML
                    ptr += 1
                    dest[ptr] = pdf417PL
                    ptr += 1
                    dest[ptr] = v & 0xFF
                    ptr += 1
                    mode = pdf417Punctuation
                } else {
                    dest[ptr] = pdf417PS
                    ptr += 1
                    dest[ptr] = v & 0xFF
                    ptr += 1
                }
            } else if mode == pdf417Lower {
                if (v & pdf417Alpha) != 0 {
                    if (getTextTypeAndValue(lengthEnd, k + 1) &
                        getTextTypeAndValue(lengthEnd, k + 2) & pdf417Alpha) != 0 {
                        dest[ptr] = pdf417ML
                        ptr += 1
                        dest[ptr] = pdf417AL
                        ptr += 1
                        mode = pdf417Alpha
                    } else {
                        dest[ptr] = pdf417AS
                        ptr += 1
                    }
                    dest[ptr] = v & 0xFF
                    ptr += 1
                } else if (v & pdf417Mixed) != 0 {
                    dest[ptr] = pdf417ML
                    ptr += 1
                    dest[ptr] = v & 0xFF
                    ptr += 1
                    mode = pdf417Mixed
                } else if (getTextTypeAndValue(lengthEnd, k + 1) &
                           getTextTypeAndValue(lengthEnd, k + 2) & pdf417Punctuation) != 0 {
                    dest[ptr] = pdf417ML
                    ptr += 1
                    dest[ptr] = pdf417PL
                    ptr += 1
                    dest[ptr] = v & 0xFF
                    ptr += 1
                    mode = pdf417Punctuation
                } else {
                    dest[ptr] = pdf417PS
                    ptr += 1
                    dest[ptr] = v & 0xFF
                    ptr += 1
                }
            } else if mode == pdf417Mixed {
                if (v & pdf417Lower) != 0 {
                    dest[ptr] = pdf417LL
                    ptr += 1
                    dest[ptr] = v & 0xFF
                    ptr += 1
                    mode = pdf417Lower
                } else if (v & pdf417Alpha) != 0 {
                    dest[ptr] = pdf417AL
                    ptr += 1
                    dest[ptr] = v & 0xFF
                    ptr += 1
                    mode = pdf417Alpha
                } else if (getTextTypeAndValue(lengthEnd, k + 1) &
                           getTextTypeAndValue(lengthEnd, k + 2) & pdf417Punctuation) != 0 {
                    dest[ptr] = pdf417PL
                    ptr += 1
                    dest[ptr] = v & 0xFF
                    ptr += 1
                    mode = pdf417Punctuation
                } else {
                    dest[ptr] = pdf417PS
                    ptr += 1
                    dest[ptr] = v & 0xFF
                    ptr += 1
                }
            } else if mode == pdf417Punctuation {
                dest[ptr] = pdf417PAL
                ptr += 1
                mode = pdf417Alpha
                k -= 1 // re-process this character
            }

            k += 1
        }

        if (ptr & 1) != 0 {
            dest[ptr] = pdf417PS
            ptr += 1
        }
        let size = (ptr + fullBytes) / 2
        if size + cwPtr > pdf417MaxDataCodewords {
            throw BarcodeError.encodingFailed("the text is too big")
        }
        let destEnd = ptr
        ptr = 0
        while ptr < destEnd {
            let v = dest[ptr]
            ptr += 1
            if v >= 30 {
                codewords[cwPtr] = v
                cwPtr += 1
                codewords[cwPtr] = dest[ptr]
                cwPtr += 1
                ptr += 1
            } else {
                codewords[cwPtr] = v * 30 + dest[ptr]
                cwPtr += 1
                ptr += 1
            }
        }
    }

    // MARK: - Number Compaction

    func basicNumberCompaction(_ start: Int, _ length: Int) {
        let ret = cwPtr
        let retLast = length / 3
        cwPtr += retLast + 1
        for k in 0...retLast {
            codewords[ret + k] = 0
        }
        codewords[ret + retLast] = 1
        let end = length + start
        for ni in start..<end {
            for k in stride(from: retLast, through: 0, by: -1) {
                codewords[ret + k] *= 10
            }
            codewords[ret + retLast] += Int(text[ni]) - Int(Character("0").asciiValue!)
            for k in stride(from: retLast, to: 0, by: -1) {
                codewords[ret + k - 1] += codewords[ret + k] / 900
                codewords[ret + k] %= 900
            }
        }
    }

    func numberCompaction(_ start: Int, _ length: Int) throws {
        let full = (length / 44) * 15
        let sz = length % 44
        let size: Int
        if sz == 0 {
            size = full
        } else {
            size = full + sz / 3 + 1
        }
        if size + cwPtr > pdf417MaxDataCodewords {
            throw BarcodeError.encodingFailed("the text is too big")
        }
        let end = length + start
        var k = start
        while k < end {
            var segSz = end - k
            if segSz > 44 {
                segSz = 44
            }
            basicNumberCompaction(k, segSz)
            k += 44
        }
    }

    // MARK: - Byte Compaction

    func byteCompaction6(_ start: Int) {
        let length = 6
        let ret = cwPtr
        let retLast = 4
        cwPtr += retLast + 1
        for k in 0...retLast {
            codewords[ret + k] = 0
        }
        let end = length + start
        for ni in start..<end {
            for k in stride(from: retLast, through: 0, by: -1) {
                codewords[ret + k] *= 256
            }
            codewords[ret + retLast] += Int(text[ni]) & 0xFF
            for k in stride(from: retLast, to: 0, by: -1) {
                codewords[ret + k - 1] += codewords[ret + k] / 900
                codewords[ret + k] %= 900
            }
        }
    }

    func byteCompaction(_ start: Int, _ length: Int) throws {
        let size = (length / 6) * 5 + (length % 6)
        if size + cwPtr > pdf417MaxDataCodewords {
            throw BarcodeError.encodingFailed("the text is too big")
        }
        let end = length + start
        var k = start
        while k < end {
            var sz = end - k
            if sz > 6 {
                sz = 6
            }
            if sz < 6 {
                for j in 0..<sz {
                    codewords[cwPtr] = Int(text[k + j]) & 0xFF
                    cwPtr += 1
                }
            } else {
                byteCompaction6(k)
            }
            k += 6
        }
    }

    // MARK: - Break String

    func breakString() {
        let textLength = text.count
        var lastP = 0
        var startN = 0
        var nd = 0

        for k in 0..<textLength {
            let c = text[k] & 0xFF
            let ch = Character(UnicodeScalar(c))
            if ch >= "0" && ch <= "9" {
                if nd == 0 {
                    startN = k
                }
                nd += 1
                continue
            }
            if nd >= 13 {
                if lastP != startN {
                    let c2 = text[lastP] & 0xFF
                    let ch2 = Character(UnicodeScalar(c2))
                    var lastTxt = (ch2 >= " " && ch2 < "\u{7f}") || ch2 == "\r" || ch2 == "\n" || ch2 == "\t"
                    for j in lastP..<startN {
                        let c3 = text[j] & 0xFF
                        let ch3 = Character(UnicodeScalar(c3))
                        let txt = (ch3 >= " " && ch3 < "\u{7f}") || ch3 == "\r" || ch3 == "\n" || ch3 == "\t"
                        if txt != lastTxt {
                            let segT: UInt8 = lastTxt ? UInt8(ascii: "T") : UInt8(ascii: "B")
                            segmentList.append(PDF417Segment(segType: segT, start: lastP, end: j))
                            lastP = j
                            lastTxt = txt
                        }
                    }
                    let segT: UInt8 = lastTxt ? UInt8(ascii: "T") : UInt8(ascii: "B")
                    segmentList.append(PDF417Segment(segType: segT, start: lastP, end: startN))
                }
                segmentList.append(PDF417Segment(segType: UInt8(ascii: "N"), start: startN, end: k))
                lastP = k
            }
            nd = 0
        }

        if nd < 13 {
            startN = textLength
        }
        if lastP != startN {
            let c2 = text[lastP] & 0xFF
            let ch2 = Character(UnicodeScalar(c2))
            var lastTxt = (ch2 >= " " && ch2 < "\u{7f}") || ch2 == "\r" || ch2 == "\n" || ch2 == "\t"
            for j in lastP..<startN {
                let c3 = text[j] & 0xFF
                let ch3 = Character(UnicodeScalar(c3))
                let txt = (ch3 >= " " && ch3 < "\u{7f}") || ch3 == "\r" || ch3 == "\n" || ch3 == "\t"
                if txt != lastTxt {
                    let segT: UInt8 = lastTxt ? UInt8(ascii: "T") : UInt8(ascii: "B")
                    segmentList.append(PDF417Segment(segType: segT, start: lastP, end: j))
                    lastP = j
                    lastTxt = txt
                }
            }
            let segT: UInt8 = lastTxt ? UInt8(ascii: "T") : UInt8(ascii: "B")
            segmentList.append(PDF417Segment(segType: segT, start: lastP, end: startN))
        }
        if nd >= 13 {
            segmentList.append(PDF417Segment(segType: UInt8(ascii: "N"), start: startN, end: textLength))
        }

        // Merge pass 1: single-byte segments between text segments
        var k = 0
        while k < segmentList.count {
            let v = segmentList[k]
            if v.segType == UInt8(ascii: "B") && (v.end - v.start) == 1 {
                if k > 0 && k + 1 < segmentList.count {
                    let vp = segmentList[k - 1]
                    let vn = segmentList[k + 1]
                    if vp.segType == UInt8(ascii: "T") && vn.segType == UInt8(ascii: "T") &&
                       (vp.end - vp.start) + (vn.end - vn.start) >= 3 {
                        segmentList[k - 1].end = vn.end
                        segmentList.removeSubrange(k...(k + 1))
                        k = 0
                        continue
                    }
                }
            }
            k += 1
        }

        // Merge pass 2: absorb short neighbors into long text segments
        k = 0
        while k < segmentList.count {
            if segmentList[k].segType == UInt8(ascii: "T") && (segmentList[k].end - segmentList[k].start) >= 5 {
                var redo = false
                if k > 0 {
                    let vp = segmentList[k - 1]
                    if (vp.segType == UInt8(ascii: "B") && (vp.end - vp.start) == 1) || vp.segType == UInt8(ascii: "T") {
                        redo = true
                        segmentList[k].start = vp.start
                        segmentList.remove(at: k - 1)
                        k -= 1
                    }
                }
                if k + 1 < segmentList.count {
                    let vn = segmentList[k + 1]
                    if (vn.segType == UInt8(ascii: "B") && (vn.end - vn.start) == 1) || vn.segType == UInt8(ascii: "T") {
                        redo = true
                        segmentList[k].end = vn.end
                        segmentList.remove(at: k + 1)
                    }
                }
                if redo {
                    k = 0
                    continue
                }
            }
            k += 1
        }

        // Merge pass 3: absorb short text segments into byte segments
        k = 0
        while k < segmentList.count {
            if segmentList[k].segType == UInt8(ascii: "B") {
                var redo = false
                if k > 0 {
                    let vp = segmentList[k - 1]
                    if (vp.segType == UInt8(ascii: "T") && (vp.end - vp.start) < 5) || vp.segType == UInt8(ascii: "B") {
                        redo = true
                        segmentList[k].start = vp.start
                        segmentList.remove(at: k - 1)
                        k -= 1
                    }
                }
                if k + 1 < segmentList.count {
                    let vn = segmentList[k + 1]
                    if (vn.segType == UInt8(ascii: "T") && (vn.end - vn.start) < 5) || vn.segType == UInt8(ascii: "B") {
                        redo = true
                        segmentList[k].end = vn.end
                        segmentList.remove(at: k + 1)
                    }
                }
                if redo {
                    k = 0
                    continue
                }
            }
            k += 1
        }

        // Special case: single text segment of all digits
        if segmentList.count == 1 {
            if segmentList[0].segType == UInt8(ascii: "T") && (segmentList[0].end - segmentList[0].start) >= 8 {
                var allDigits = true
                for kk in segmentList[0].start..<segmentList[0].end {
                    let ch = Character(UnicodeScalar(text[kk] & 0xFF))
                    if ch < "0" || ch > "9" {
                        allDigits = false
                        break
                    }
                }
                if allDigits {
                    segmentList[0].segType = UInt8(ascii: "N")
                }
            }
        }
    }

    // MARK: - Assemble

    func assemble() throws {
        if segmentList.isEmpty {
            return
        }
        cwPtr = 1
        for (k, v) in segmentList.enumerated() {
            let segLen = v.end - v.start
            switch v.segType {
            case UInt8(ascii: "T"):
                if k != 0 {
                    codewords[cwPtr] = pdf417TextMode
                    cwPtr += 1
                }
                try textCompaction(v.start, segLen)
            case UInt8(ascii: "N"):
                codewords[cwPtr] = pdf417NumericMode
                cwPtr += 1
                try numberCompaction(v.start, segLen)
            case UInt8(ascii: "B"):
                if segLen % 6 != 0 {
                    codewords[cwPtr] = pdf417ByteMode
                } else {
                    codewords[cwPtr] = pdf417ByteMode6
                }
                cwPtr += 1
                try byteCompaction(v.start, segLen)
            default:
                break
            }
        }
    }

    // MARK: - Max Possible Error Level

    func maxPossibleErrorLevel(_ remain: Int) -> Int {
        var level = 8
        var size = 512
        while level > 0 {
            if remain >= size {
                return level
            }
            level -= 1
            size >>= 1
        }
        return 0
    }

    // MARK: - Max Square

    func maxSquare() -> Int {
        if codeColumns > 21 {
            codeColumns = 29
            codeRows = 32
        } else {
            codeColumns = 16
            codeRows = 58
        }
        return pdf417MaxDataCodewords + 2
    }

    // MARK: - Paint Code (Main Entry)

    func paintCode() throws {
        initBlock()

        if text.isEmpty {
            throw BarcodeError.invalidInput("text cannot be empty")
        }
        if text.count > pdf417AbsoluteMaxTextSize {
            throw BarcodeError.encodingFailed("the text is too big")
        }

        segmentList = []
        breakString()
        try assemble()
        segmentList = []
        codewords[0] = cwPtr
        lenCodewords = cwPtr

        let maxErr = maxPossibleErrorLevel(pdf417MaxDataCodewords + 2 - lenCodewords)

        if useAutoErrorLevel {
            if lenCodewords < 41 {
                errorLevel = 2
            } else if lenCodewords < 161 {
                errorLevel = 3
            } else if lenCodewords < 321 {
                errorLevel = 4
            } else {
                errorLevel = 5
            }
        }

        if errorLevel < 0 {
            errorLevel = 0
        } else if errorLevel > maxErr {
            errorLevel = maxErr
        }

        if codeColumns < 1 {
            codeColumns = 1
        } else if codeColumns > 30 {
            codeColumns = 30
        }

        if codeRows < 3 {
            codeRows = 3
        } else if codeRows > 90 {
            codeRows = 90
        }

        var lenErr = 2 << errorLevel
        let fixedColumn: Bool
        var skipRowColAdjust = false
        var tot = lenCodewords + lenErr

        if sizeKind == pdf417SizeColumnsAndRows {
            tot = codeColumns * codeRows
            if tot > pdf417MaxDataCodewords + 2 {
                tot = maxSquare()
            }
            if tot < lenCodewords + lenErr {
                tot = lenCodewords + lenErr
            } else {
                skipRowColAdjust = true
            }
            fixedColumn = true // not used when skipRowColAdjust is true
        } else if sizeKind == pdf417SizeAuto {
            if aspectRatio < 0.001 {
                aspectRatio = 0.001
            } else if aspectRatio > 1000 {
                aspectRatio = 1000
            }
            let b = 73.0 * aspectRatio - 4.0
            let c = (-b + (b * b + 4.0 * 17.0 * aspectRatio *
                           Double(lenCodewords + lenErr) * Double(yHeight)).squareRoot()) /
                    (2.0 * 17.0 * aspectRatio)
            codeColumns = Int(c + 0.5)
            if codeColumns < 1 {
                codeColumns = 1
            } else if codeColumns > 30 {
                codeColumns = 30
            }
            fixedColumn = true
        } else {
            fixedColumn = sizeKind != pdf417SizeRows
        }

        if !skipRowColAdjust {
            if fixedColumn {
                codeRows = (tot - 1) / codeColumns + 1
                if codeRows < 3 {
                    codeRows = 3
                } else if codeRows > 90 {
                    codeRows = 90
                    codeColumns = (tot - 1) / 90 + 1
                }
            } else {
                codeColumns = (tot - 1) / codeRows + 1
                if codeColumns > 30 {
                    codeColumns = 30
                    codeRows = (tot - 1) / 30 + 1
                }
            }
            tot = codeRows * codeColumns
        }

        if tot > pdf417MaxDataCodewords + 2 {
            tot = maxSquare()
        }

        errorLevel = maxPossibleErrorLevel(tot - lenCodewords)
        lenErr = 2 << errorLevel
        var pad = tot - lenErr - lenCodewords
        cwPtr = lenCodewords
        while pad > 0 {
            codewords[cwPtr] = pdf417TextMode
            cwPtr += 1
            pad -= 1
        }
        codewords[0] = cwPtr
        lenCodewords = cwPtr
        calculateErrorCorrection(lenCodewords)
        lenCodewords = tot

        outPaintCode()
    }

    // MARK: - Convert to Bool Matrix

    func convertToBoolMatrix() -> [[Bool]]? {
        if outBits.isEmpty || codeRows == 0 || bitColumns == 0 {
            return nil
        }
        let rows = codeRows
        let bitCols = bitColumns
        let bytesPerRow = (bitCols + 7) / 8
        var result = [[Bool]]()
        result.reserveCapacity(rows)
        for r in 0..<rows {
            var rowData = [Bool](repeating: false, count: bitCols)
            for c in 0..<bitCols {
                let byteIndex = r * bytesPerRow + c / 8
                let bitIndex = 7 - (c % 8)
                if byteIndex < outBits.count {
                    rowData[c] = (outBits[byteIndex] & (1 << bitIndex)) != 0
                }
            }
            result.append(rowData)
        }
        return result
    }
}

// MARK: - Helper

private func pdf417IndexOf(_ s: String, _ b: UInt8) -> Int {
    let bytes = Array(s.utf8)
    for i in 0..<bytes.count {
        if bytes[i] == b {
            return i
        }
    }
    return -1
}

// MARK: - Public PDF417 Class

/// PDF417 encodes PDF417 2D barcodes.
public class PDF417: BarcodeBase2D {
    /// Error correction level (-1=Auto, 0..8).
    public var errorCorrectionLevel: Int = PDF417_ERROR_LEVEL2

    /// Number of data columns (0=auto, 1..30).
    public var columns: Int = 0

    /// Number of rows (0=auto, 3..90).
    public var rows: Int = 0

    private var sizeMode: Int = PDF417_SIZE_AUTO
    private var aspectRatio: Double = 0.5
    private var yHeight: Int = 3
    private var quietZoneWidth: Int = 2
    private var useAutoErrorLevel: Bool = true

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    // MARK: - Settings

    /// Sets the error correction level (-1=Auto, 0..8).
    public func setErrorCorrectionLevel(_ level: Int) {
        if level >= PDF417_ERROR_AUTO && level <= PDF417_ERROR_LEVEL8 {
            errorCorrectionLevel = level
        }
    }

    /// Sets the number of data columns (0=auto, 1..30).
    public func setColumns(_ cols: Int) {
        if cols >= 0 && cols <= 30 {
            columns = cols
        }
    }

    /// Sets the number of rows (0=auto, 3..90).
    public func setRows(_ rows: Int) {
        if rows >= 0 && rows <= 90 {
            self.rows = rows
        }
    }

    /// Sets the size mode (PDF417_SIZE_AUTO, PDF417_SIZE_ROWS, PDF417_SIZE_COLUMNS, PDF417_SIZE_COLUMNS_AND_ROWS).
    public func setSizeMode(_ mode: Int) {
        if mode >= PDF417_SIZE_AUTO && mode <= PDF417_SIZE_COLUMNS_AND_ROWS {
            sizeMode = mode
        }
    }

    /// Returns the current size mode.
    public func getSizeMode() -> Int { sizeMode }

    /// Sets the aspect ratio (0.001..1000).
    public func setAspectRatio(_ ratio: Double) {
        if ratio >= 0.001 && ratio <= 1000.0 {
            aspectRatio = ratio
        }
    }

    /// Returns the current aspect ratio.
    public func getAspectRatio() -> Double { aspectRatio }

    /// Sets the Y-dimension height multiplier.
    public func setYHeight(_ h: Int) {
        if h > 0 {
            yHeight = h
        }
    }

    /// Returns the current Y-height.
    public func getYHeight() -> Int { yHeight }

    /// Sets the quiet zone width in modules.
    public func setQuietZoneWidth(_ w: Int) {
        if w >= 0 {
            quietZoneWidth = w
        }
    }

    /// Returns the current quiet zone width.
    public func getQuietZoneWidth() -> Int { quietZoneWidth }

    /// Sets whether to automatically determine error level.
    public func setUseAutoErrorLevel(_ on: Bool) {
        useAutoErrorLevel = on
    }

    /// Returns the current auto error level setting.
    public func getUseAutoErrorLevel() -> Bool { useAutoErrorLevel }

    // MARK: - Internal: Apply Settings to Encoder

    private func applySettingsCal(_ enc: PDF417Encoder) {
        enc.errorLevel = errorCorrectionLevel
        enc.codeColumns = columns
        enc.codeRows = rows
        enc.aspectRatio = aspectRatio
        enc.yHeight = yHeight
        enc.useAutoErrorLevel = useAutoErrorLevel
    }

    private func applySettingsFull(_ enc: PDF417Encoder) {
        enc.errorLevel = errorCorrectionLevel
        enc.codeColumns = columns
        enc.codeRows = rows
        enc.aspectRatio = aspectRatio
        enc.yHeight = yHeight
        enc.useAutoErrorLevel = useAutoErrorLevel
        switch sizeMode {
        case PDF417_SIZE_AUTO:
            enc.sizeKind = pdf417SizeAuto
        case PDF417_SIZE_COLUMNS:
            enc.sizeKind = pdf417SizeColumns
        case PDF417_SIZE_ROWS:
            enc.sizeKind = pdf417SizeRows
        case PDF417_SIZE_COLUMNS_AND_ROWS:
            enc.sizeKind = pdf417SizeColumnsAndRows
        default:
            enc.sizeKind = pdf417SizeAuto
        }
    }

    // MARK: - Get Pattern

    /// Generates the PDF417 pattern as a [[Bool]] matrix (true=black).
    public func getPattern(_ code: String) throws -> [[Bool]] {
        let data = Array(code.utf8)
        guard !data.isEmpty else {
            throw BarcodeError.invalidInput("empty input")
        }

        let enc = PDF417Encoder()
        applySettingsCal(enc)
        enc.text = data
        try enc.paintCode()
        guard let result = enc.convertToBoolMatrix() else {
            throw BarcodeError.encodingFailed("failed to generate pattern")
        }
        return result
    }

    // MARK: - Draw

    /// Renders the PDF417 barcode to the internal buffer (SVG or PNG/JPEG).
    /// PDF417 takes width AND height (like 1D barcodes).
    public func draw(code: String, width: Int, height: Int) throws {
        guard width > 0 && height > 0 else {
            throw BarcodeError.invalidInput("width and height must be positive")
        }

        let patt = try getPattern(code)

        let numRows = patt.count
        guard numRows > 0 else {
            throw BarcodeError.encodingFailed("empty pattern")
        }
        let numCols = patt[0].count
        guard numCols > 0 else {
            throw BarcodeError.encodingFailed("empty pattern")
        }

        if isSVGOutput() {
            try drawSVGPDF417(patt, numRows: numRows, numCols: numCols, width: width, height: height)
        } else {
            try drawPNGPDF417(patt, numRows: numRows, numCols: numCols, width: width, height: height)
        }
    }

    // MARK: - SVG Rendering

    private func drawSVGPDF417(_ patt: [[Bool]], numRows: Int, numCols: Int, width: Int, height: Int) throws {
        var moduleW: Double
        var moduleH: Double
        if fitWidth {
            moduleW = Double(width) / Double(numCols)
            moduleH = Double(height) / Double(numRows)
        } else {
            moduleW = Double(width / numCols)
            moduleH = Double(height / numRows)
            if moduleW < 1.0 {
                moduleW = 1.0
            }
            if moduleH < 1.0 {
                moduleH = 1.0
            }
        }

        let aw = Int((moduleW * Double(numCols)).rounded(.up))
        let ah = Int((moduleH * Double(numRows)).rounded(.up))

        svgBegin(aw, ah)
        svgRect(0, 0, Double(aw), Double(ah), backColor)

        let adj = Double(pxAdjBlack)
        for r in 0..<numRows {
            for c in 0..<numCols {
                if patt[r][c] {
                    var dw = moduleW + adj + 0.5
                    var dh = moduleH + adj + 0.5
                    if dw < 0 { dw = 0 }
                    if dh < 0 { dh = 0 }
                    svgRect(Double(c) * moduleW, Double(r) * moduleH, dw, dh, foreColor)
                }
            }
        }

        svgEnd()
    }

    // MARK: - PNG/JPEG Rendering

    private func drawPNGPDF417(_ patt: [[Bool]], numRows: Int, numCols: Int, width: Int, height: Int) throws {
        var moduleW: Double
        var moduleH: Double
        if fitWidth {
            moduleW = Double(width) / Double(numCols)
            moduleH = Double(height) / Double(numRows)
        } else {
            moduleW = Double(width / numCols)
            moduleH = Double(height / numRows)
            if moduleW < 1.0 {
                moduleW = 1.0
            }
            if moduleH < 1.0 {
                moduleH = 1.0
            }
        }

        let aw = Int((moduleW * Double(numCols)).rounded(.up))
        let ah = Int((moduleH * Double(numRows)).rounded(.up))

        let img = PixelBuffer(width: aw, height: ah)
        img.fill(backColor)

        let adj = pxAdjBlack
        for r in 0..<numRows {
            for c in 0..<numCols {
                if patt[r][c] {
                    let drawX = Int(Double(c) * moduleW)
                    let drawY = Int(Double(r) * moduleH)
                    var drawW = Int(moduleW + Double(adj))
                    var drawH = Int(moduleH + Double(adj))
                    if drawW < 1 { drawW = 1 }
                    if drawH < 1 { drawH = 1 }
                    var endX = drawX + drawW
                    var endY = drawY + drawH
                    if endX > aw { endX = aw }
                    if endY > ah { endY = ah }
                    img.fillRect(drawX, drawY, endX, endY, foreColor)
                }
            }
        }

        // Trial mode watermark
        if isTrialMode() {
            drawSampleOverlayPNG(img, x: 0, y: 0, width: aw, height: ah)
        }

        try encodeImageBuffer(img)
    }
}
