import Foundation

private let nw7Chars = "ABCD.+:/$-0123456789"
private let nw7Patterns: [Int] = [
    0x1a, 0x29, 0x0b, 0x0e, 0x54, 0x15, 0x45, 0x51, 0x18, 0x0c,
    0x03, 0x06, 0x09, 0x60, 0x12, 0x42, 0x21, 0x24, 0x30, 0x48,
]

/// NW-7 (Codabar) barcode encoder.
public class NW7: BarcodeBase1D, Encoder1D {
    public var showStartStop: Bool = true

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    public func encode(_ code: String) throws -> [Int] {
        if code.isEmpty {
            throw BarcodeError.emptyString
        }

        let s = code.uppercased()
        let bytes = Array(s.utf8)

        let startChar = bytes[0]
        let stopChar = bytes[bytes.count - 1]
        let isStartValid = startChar >= UInt8(ascii: "A") && startChar <= UInt8(ascii: "D")
        let isStopValid = stopChar >= UInt8(ascii: "A") && stopChar <= UInt8(ascii: "D")

        let encoded: String
        if isStartValid && isStopValid {
            encoded = s
        } else if isStartValid || isStopValid {
            throw BarcodeError.invalidInput("invalid start/stop character combination in NW7: \(s)")
        } else {
            encoded = "C" + s + "C"
        }

        var result = [Int]()
        let encodedBytes = Array(encoded.utf8)
        for i in 0..<encodedBytes.count {
            if i > 0 {
                result.append(2) // inter-character gap
            }

            guard let charIdx = nw7Chars.firstIndex(of: Character(UnicodeScalar(encodedBytes[i]))) else {
                throw BarcodeError.invalidCharacter(symbology: "NW7", character: Character(UnicodeScalar(encodedBytes[i])))
            }
            let charIndex = nw7Chars.distance(from: nw7Chars.startIndex, to: charIdx)
            let pattern = nw7Patterns[charIndex]

            // 7-bit pattern: MSB first, 1=wide(3), 0=narrow(1)
            var mask = 0x40
            for _ in 0..<7 {
                if (pattern & mask) != 0 {
                    result.append(3)
                } else {
                    result.append(1)
                }
                mask >>= 1
            }
        }

        return result
    }

    public func draw(code: String, width: Int, height: Int) throws {
        try self.draw(code, width: width, height: height, encoder: self)
    }
}
