import Foundation

private let nec2of5DigitPatterns: [Int] = [
    0x18, 0x03, 0x05, 0x06, 0x09, 0x0a, 0x0c, 0x11, 0x12, 0x14,
]
private let nec2of5Start = 0x5 // 101
private let nec2of5Stop  = 0x3 // 011

/// NEC 2 of 5 barcode encoder.
public class NEC2of5: BarcodeBase1D, Encoder1D {
    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    public func encode(_ code: String) throws -> [Int] {
        if code.isEmpty { throw BarcodeError.emptyString }

        for ch in code {
            if ch < "0" || ch > "9" {
                throw BarcodeError.invalidInput("NEC 2 of 5 barcode requires numeric digits only")
            }
        }

        let wide = minLineWidth % 2 != 0 ? 3 : 2
        let narrow = 1
        var result = [Int]()

        // Start code (3 bits: 101)
        var chk = 0x4
        for j in 0..<3 {
            if j % 2 == 0 {
                result.append((nec2of5Start & (chk >> j)) != 0 ? wide : narrow)
            } else {
                result.append(narrow)
            }
        }

        result.append(narrow) // Inter-character gap

        // Data encoding
        let bytes = Array(code.utf8)
        for i in 0..<bytes.count {
            let digit = Int(bytes[i] - UInt8(ascii: "0"))
            let pattern = nec2of5DigitPatterns[digit]

            let chk2 = 0x10
            for j in 0..<5 {
                if j % 2 == 0 {
                    result.append((pattern & (chk2 >> j)) != 0 ? wide : narrow)
                } else {
                    result.append(narrow)
                }
            }

            if i < bytes.count - 1 {
                result.append(narrow)
            }
        }

        // Stop code (3 bits: 011)
        chk = 0x4
        for j in 0..<3 {
            if j % 2 == 0 {
                result.append((nec2of5Stop & (chk >> j)) != 0 ? wide : narrow)
            } else {
                result.append((nec2of5Stop & (chk >> j)) != 0 ? wide : narrow)
            }
        }

        return result
    }

    public func draw(code: String, width: Int, height: Int) throws {
        try self.draw(code, width: width, height: height, encoder: self)
    }
}
