import Foundation

private let matrix2of5DigitPatterns: [Int] = [
    0x18, 0x03, 0x05, 0x06, 0x09, 0x0a, 0x0c, 0x11, 0x12, 0x14,
]
private let matrix2of5Start = 0x5 // 101
private let matrix2of5Stop  = 0x3 // 011

/// Matrix 2 of 5 barcode encoder.
public class Matrix2of5: BarcodeBase1D, Encoder1D {
    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    public func encode(_ code: String) throws -> [Int] {
        if code.isEmpty { throw BarcodeError.emptyString }

        for ch in code {
            if ch < "0" || ch > "9" {
                throw BarcodeError.invalidInput("Matrix 2 of 5 barcode requires numeric digits only")
            }
        }

        let wide = minLineWidth % 2 != 0 ? 3 : 2
        let narrow = 1
        var result = [Int]()

        // Start code (3 bits: 101)
        var mask = 0x4
        for j in 0..<3 {
            if j % 2 == 0 { // bar position
                result.append((matrix2of5Start & (mask >> j)) != 0 ? wide : narrow)
            } else { // space position
                result.append(narrow)
            }
        }

        result.append(narrow) // Inter-character gap

        // Data encoding
        let bytes = Array(code.utf8)
        for i in 0..<bytes.count {
            let digit = Int(bytes[i] - UInt8(ascii: "0"))
            let pattern = matrix2of5DigitPatterns[digit]

            let bitMask = 0x10
            for j in 0..<5 {
                if j % 2 == 0 { // bar position
                    result.append((pattern & (bitMask >> j)) != 0 ? wide : narrow)
                } else { // space position - always narrow
                    result.append(narrow)
                }
            }

            if i < bytes.count - 1 {
                result.append(narrow)
            }
        }

        // Stop code (3 bits: 011)
        mask = 0x4
        for j in 0..<3 {
            if j % 2 == 0 { // bar position
                result.append((matrix2of5Stop & (mask >> j)) != 0 ? wide : narrow)
            } else { // space position - responds to bit pattern for stop
                result.append((matrix2of5Stop & (mask >> j)) != 0 ? wide : narrow)
            }
        }

        return result
    }

    public func draw(code: String, width: Int, height: Int) throws {
        try self.draw(code, width: width, height: height, encoder: self)
    }
}
