import Foundation

// MARK: - JAN/UPC Shared Rendering

/// JANUPCParams holds shared text/bar parameters for JAN/UPC rendering.
struct JANUPCParams {
    var textHeight: Double = 0
    var normalBarHeight: Double = 0
    var tallBarHeight: Double = 0
    var svgFontSize: Int = 0
    var textY: Double = 0
    var digitWidth: Double = 0
    var leftMargin: Double = 0
    var charWidth: Double = 0
    var barcodeWidth: Double = 0
    var barcodeStartX: Double = 0
    var moduleWidth: Double = 0
    var posA: Double = 0
}

/// Computes shared text/bar parameters for JAN/UPC rendering.
func computeJANUPCParams(height: Int, tfScale: Double, tvoScale: Double,
                          pattern: [Int], width: Int, clsName: String) -> JANUPCParams {
    var p = JANUPCParams()

    var baseRatio = 0.2
    if height < 50 {
        let nr = baseRatio * 1.25
        baseRatio = nr < 0.25 ? nr : 0.25
    } else if height > 200 {
        let nr = baseRatio * 0.75
        baseRatio = nr > 0.15 ? nr : 0.15
    }

    p.textHeight = Double(Int(Double(height) * baseRatio * tfScale))
    if p.textHeight < 8 {
        p.textHeight = 8
    }

    var voRatio = 0.05
    if height < 50 {
        let nr = voRatio * 0.6
        voRatio = nr > 0.03 ? nr : 0.03
    } else if height > 200 {
        let nr = voRatio * 1.4
        voRatio = nr < 0.08 ? nr : 0.08
    }

    var verticalOffsetBar = Double(Int(Double(height) * voRatio * tvoScale))
    if verticalOffsetBar < 2 {
        verticalOffsetBar = 2
    }

    p.normalBarHeight = Double(height) - p.textHeight - verticalOffsetBar
    if p.normalBarHeight < 1 {
        p.normalBarHeight = 1
    }

    p.svgFontSize = Int(p.textHeight * 0.8)
    if p.svgFontSize < 8 {
        p.svgFontSize = 8
    }

    var verticalOffsetText = Double(Int(Double(height) * 0.03 * tvoScale))
    if verticalOffsetText < 1 {
        verticalOffsetText = 1
    }
    p.textY = p.normalBarHeight + verticalOffsetText

    p.digitWidth = Double(p.svgFontSize) * 0.6
    p.leftMargin = p.digitWidth * 1.0

    let fontHeight = Double(height) * 0.2 * tfScale
    p.charWidth = fontHeight * 0.8

    let longBarExtra: Double
    if clsName == "JAN8" || clsName == "JAN13" {
        longBarExtra = fontHeight * 0.275
    } else {
        longBarExtra = fontHeight * 0.495
    }
    p.tallBarHeight = p.normalBarHeight + longBarExtra

    p.posA = 0

    let totalUnits = pattern.reduce(0, +)

    switch clsName {
    case "JAN8":
        p.barcodeWidth = Double(width)
        p.barcodeStartX = 0
    case "JAN13":
        let leftMarginWidth = p.charWidth * 1.2
        p.barcodeWidth = Double(width) - leftMarginWidth
        p.barcodeStartX = leftMarginWidth
    case "UPCA", "UPCE":
        let outMargin = p.charWidth * 1.0
        p.barcodeWidth = Double(width) - outMargin * 2
        p.barcodeStartX = outMargin
    default:
        p.barcodeWidth = Double(width)
        p.barcodeStartX = 0
    }

    if totalUnits > 0 {
        p.moduleWidth = p.barcodeWidth / Double(totalUnits)
    }

    return p
}

// MARK: - SVG Rendering Extensions for JAN/UPC

extension BarcodeBase1D {

    /// Renders JAN8 with extended guard bars as SVG.
    func drawSVGJAN8(_ pattern: [Int], displayCode: String, width: Int, height: Int) {
        let p = computeJANUPCParams(height: height, tfScale: textFontScale,
                                     tvoScale: textVerticalOffsetScale,
                                     pattern: pattern, width: width, clsName: "JAN8")

        svgBegin(width, height)
        svgRect(0, 0, Double(width), Double(height), backColor)

        var currentX = p.barcodeStartX
        var isBar = true
        var segIdx = 0, segCnt = 0
        var posB = 0.0, posC = 0.0, posD = 0.0, posE = 0.0

        for v in pattern {
            var barW = Double(v) * p.moduleWidth
            if barW < 1 { barW = 1 }

            if segIdx == 0 && segCnt >= 3 {
                posB = currentX; segIdx += 1; segCnt = 0
            } else if segIdx == 1 && segCnt >= 16 {
                posC = currentX; segIdx += 1; segCnt = 0
            } else if segIdx == 2 && segCnt >= 5 {
                posD = currentX; segIdx += 1; segCnt = 0
            } else if segIdx == 3 && segCnt >= 16 {
                posE = currentX; segIdx += 1; segCnt = 0
            }

            let isTall = (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
            if isBar {
                let bh = isTall ? p.tallBarHeight : p.normalBarHeight
                svgRect(currentX, 0, barW, bh, foreColor)
            }
            segCnt += 1
            currentX += barW
            isBar = !isBar
        }

        // Fallback
        let mw = p.moduleWidth
        if posB == 0 { posB = p.barcodeStartX + 3 * mw }
        if posC == 0 { posC = posB + 28 * mw }
        if posD == 0 { posD = posC + 5 * mw }
        if posE == 0 { posE = posD + 28 * mw }

        if displayCode.count >= 8 {
            let chars = Array(displayCode)

            let leftSection = posC - posB
            let effLeft = leftSection - p.leftMargin * 2
            if effLeft > 0 {
                if textEvenSpacing {
                    let spacing = effLeft / 3.0
                    for i in 0..<4 {
                        let digitX = posB + p.leftMargin + Double(i) * spacing
                        svgText(digitX, p.textY, String(chars[i]), p.svgFontSize, foreColor, anchor: "middle")
                    }
                } else {
                    let centerX = posB + leftSection / 2.0
                    svgText(centerX, p.textY, String(displayCode.prefix(4)), p.svgFontSize, foreColor, anchor: "middle")
                }
            }

            let rightSection = posE - posD
            let effRight = rightSection - p.leftMargin * 2
            if effRight > 0 {
                if textEvenSpacing {
                    let spacing = effRight / 3.0
                    for i in 0..<4 {
                        let digitX = posD + p.leftMargin + Double(i) * spacing
                        svgText(digitX, p.textY, String(chars[4 + i]), p.svgFontSize, foreColor, anchor: "middle")
                    }
                } else {
                    let centerX = posD + rightSection / 2.0
                    let startIdx = displayCode.index(displayCode.startIndex, offsetBy: 4)
                    let endIdx = displayCode.index(displayCode.startIndex, offsetBy: 8)
                    svgText(centerX, p.textY, String(displayCode[startIdx..<endIdx]), p.svgFontSize, foreColor, anchor: "middle")
                }
            }
        }

        svgEnd()
    }

    /// Renders JAN13 with extended guard bars as SVG.
    func drawSVGJAN13(_ pattern: [Int], displayCode: String, width: Int, height: Int) {
        let p = computeJANUPCParams(height: height, tfScale: textFontScale,
                                     tvoScale: textVerticalOffsetScale,
                                     pattern: pattern, width: width, clsName: "JAN13")

        svgBegin(width, height)
        svgRect(0, 0, Double(width), Double(height), backColor)

        var currentX = p.barcodeStartX
        var isBar = true
        var segIdx = 0, segCnt = 0
        var posB = 0.0, posC = 0.0, posD = 0.0, posE = 0.0

        for v in pattern {
            var barW = Double(v) * p.moduleWidth
            if barW < 1 { barW = 1 }

            if segIdx == 0 && segCnt >= 3 {
                posB = currentX; segIdx += 1; segCnt = 0
            } else if segIdx == 1 && segCnt >= 24 {
                posC = currentX; segIdx += 1; segCnt = 0
            } else if segIdx == 2 && segCnt >= 5 {
                posD = currentX; segIdx += 1; segCnt = 0
            } else if segIdx == 3 && segCnt >= 24 {
                posE = currentX; segIdx += 1; segCnt = 0
            }

            let isTall = (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
            if isBar {
                let bh = isTall ? p.tallBarHeight : p.normalBarHeight
                svgRect(currentX, 0, barW, bh, foreColor)
            }
            segCnt += 1
            currentX += barW
            isBar = !isBar
        }

        if displayCode.count >= 13 {
            let chars = Array(displayCode)

            // Prefix
            let prefixX = p.posA + p.digitWidth * 0.3
            svgText(prefixX, p.textY, String(chars[0]), p.svgFontSize, foreColor, anchor: "start")

            let leftSection = posC - posB
            let effLeft = leftSection - p.leftMargin * 2
            if effLeft > 0 {
                if textEvenSpacing {
                    let spacing = effLeft / 5.0
                    for i in 0..<6 {
                        let digitX = posB + p.leftMargin + Double(i) * spacing
                        svgText(digitX, p.textY, String(chars[1 + i]), p.svgFontSize, foreColor, anchor: "middle")
                    }
                } else {
                    let centerX = posB + leftSection / 2.0
                    let startIdx = displayCode.index(displayCode.startIndex, offsetBy: 1)
                    let endIdx = displayCode.index(displayCode.startIndex, offsetBy: 7)
                    svgText(centerX, p.textY, String(displayCode[startIdx..<endIdx]), p.svgFontSize, foreColor, anchor: "middle")
                }
            }

            let rightSection = posE - posD
            let effRight = rightSection - p.leftMargin * 2
            if effRight > 0 {
                if textEvenSpacing {
                    let spacing = effRight / 5.0
                    for i in 0..<6 {
                        let digitX = posD + p.leftMargin + Double(i) * spacing
                        svgText(digitX, p.textY, String(chars[7 + i]), p.svgFontSize, foreColor, anchor: "middle")
                    }
                } else {
                    let centerX = posD + rightSection / 2.0
                    let startIdx = displayCode.index(displayCode.startIndex, offsetBy: 7)
                    let endIdx = displayCode.index(displayCode.startIndex, offsetBy: 13)
                    svgText(centerX, p.textY, String(displayCode[startIdx..<endIdx]), p.svgFontSize, foreColor, anchor: "middle")
                }
            }
        }

        svgEnd()
    }

    /// Renders UPC-A with extended guard bars as SVG.
    func drawSVGUPCA(_ pattern: [Int], displayCode: String, width: Int, height: Int) {
        let p = computeJANUPCParams(height: height, tfScale: textFontScale,
                                     tvoScale: textVerticalOffsetScale,
                                     pattern: pattern, width: width, clsName: "UPCA")

        svgBegin(width, height)
        svgRect(0, 0, Double(width), Double(height), backColor)

        var currentX = p.barcodeStartX
        var isBar = true
        var segIdx = 0, segCnt = 0
        var posB = 0.0, posC = 0.0, posD = 0.0, posE = 0.0

        for v in pattern {
            var barW = Double(v) * p.moduleWidth
            if barW < 1 { barW = 1 }

            if segIdx == 0 && segCnt >= 7 {
                posB = currentX; segIdx += 1; segCnt = 0
            } else if segIdx == 1 && segCnt >= 21 {
                posC = currentX; segIdx += 1; segCnt = 0
            } else if segIdx == 2 && segCnt >= 3 {
                posD = currentX; segIdx += 1; segCnt = 0
            } else if segIdx == 3 && segCnt >= 21 {
                posE = currentX; segIdx += 1; segCnt = 0
            }

            let isTall = (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
            if isBar {
                let bh = isTall ? p.tallBarHeight : p.normalBarHeight
                svgRect(currentX, 0, barW, bh, foreColor)
            }
            segCnt += 1
            currentX += barW
            isBar = !isBar
        }

        let posF = currentX

        if displayCode.count >= 12 {
            let chars = Array(displayCode)

            // Prefix
            let prefixX = p.posA + p.digitWidth * 0.3
            svgText(prefixX, p.textY, String(chars[0]), p.svgFontSize, foreColor, anchor: "start")

            // Left 5
            let leftSection = posC - posB
            let effLeft = leftSection - p.leftMargin * 2
            if effLeft > 0 {
                if textEvenSpacing {
                    let spacing = effLeft / 4.0
                    for i in 0..<5 {
                        let digitX = posB + p.leftMargin + Double(i) * spacing
                        svgText(digitX, p.textY, String(chars[1 + i]), p.svgFontSize, foreColor, anchor: "middle")
                    }
                } else {
                    let centerX = posB + leftSection / 2.0
                    let startIdx = displayCode.index(displayCode.startIndex, offsetBy: 1)
                    let endIdx = displayCode.index(displayCode.startIndex, offsetBy: 6)
                    svgText(centerX, p.textY, String(displayCode[startIdx..<endIdx]), p.svgFontSize, foreColor, anchor: "middle")
                }
            }

            // Right 5
            let rightSection = posE - posD
            let effRight = rightSection - p.leftMargin * 2
            if effRight > 0 {
                if textEvenSpacing {
                    let spacing = effRight / 4.0
                    for i in 0..<5 {
                        let digitX = posD + p.leftMargin + Double(i) * spacing
                        svgText(digitX, p.textY, String(chars[6 + i]), p.svgFontSize, foreColor, anchor: "middle")
                    }
                } else {
                    let centerX = posD + rightSection / 2.0
                    let startIdx = displayCode.index(displayCode.startIndex, offsetBy: 6)
                    let endIdx = displayCode.index(displayCode.startIndex, offsetBy: 11)
                    svgText(centerX, p.textY, String(displayCode[startIdx..<endIdx]), p.svgFontSize, foreColor, anchor: "middle")
                }
            }

            // Suffix
            let suffixX = posF + p.digitWidth * 0.7
            svgText(suffixX, p.textY, String(chars[11]), p.svgFontSize, foreColor, anchor: "start")
        }

        svgEnd()
    }

    /// Renders UPC-E with extended guard bars as SVG.
    func drawSVGUPCE(_ pattern: [Int], displayCode: String, width: Int, height: Int) {
        let p = computeJANUPCParams(height: height, tfScale: textFontScale,
                                     tvoScale: textVerticalOffsetScale,
                                     pattern: pattern, width: width, clsName: "UPCE")

        svgBegin(width, height)
        svgRect(0, 0, Double(width), Double(height), backColor)

        var currentX = p.barcodeStartX
        var isBar = true
        var posB = 0.0, posC = 0.0
        var accumulatedUnits = 0

        for v in pattern {
            var barW = Double(v) * p.moduleWidth
            if barW < 1 { barW = 1 }

            if posB == 0 && accumulatedUnits >= 3 {
                posB = currentX
            }
            if posC == 0 && accumulatedUnits >= 45 {
                posC = currentX
            }

            let isStartGuard = accumulatedUnits < 3
            let isEndGuard = accumulatedUnits >= 45
            let isTall = (isStartGuard || isEndGuard) && isBar

            if isBar {
                let bh = isTall ? p.tallBarHeight : p.normalBarHeight
                svgRect(currentX, 0, barW, bh, foreColor)
            }

            accumulatedUnits += v
            currentX += barW
            isBar = !isBar
        }

        if posB == 0 {
            posB = p.barcodeStartX + 3 * p.moduleWidth
        }
        if posC == 0 {
            posC = p.barcodeStartX + 45 * p.moduleWidth
        }

        posB += p.charWidth * 0.5
        posC -= p.charWidth * 0.5
        let posF = currentX

        if displayCode.count >= 8 {
            let chars = Array(displayCode)

            // Prefix
            let prefixX = p.posA + p.digitWidth * 0.3
            svgText(prefixX, p.textY, String(chars[0]), p.svgFontSize, foreColor, anchor: "start")

            let midSection = posC - posB
            let effMid = midSection - p.leftMargin * 2
            if effMid > 0 {
                if textEvenSpacing {
                    let spacing = effMid / 5.0
                    for i in 0..<6 {
                        let digitX = posB + p.leftMargin + Double(i) * spacing
                        svgText(digitX, p.textY, String(chars[1 + i]), p.svgFontSize, foreColor, anchor: "middle")
                    }
                } else {
                    let centerX = posB + midSection / 2.0
                    let startIdx = displayCode.index(displayCode.startIndex, offsetBy: 1)
                    let endIdx = displayCode.index(displayCode.startIndex, offsetBy: 7)
                    svgText(centerX, p.textY, String(displayCode[startIdx..<endIdx]), p.svgFontSize, foreColor, anchor: "middle")
                }
            }

            // Suffix
            let suffixX = posF + p.digitWidth * 0.7
            svgText(suffixX, p.textY, String(chars[7]), p.svgFontSize, foreColor, anchor: "start")
        }

        svgEnd()
    }

    // MARK: - PNG Rendering for JAN/UPC

    /// Renders JAN/UPC barcodes as PNG with extended guard bars.
    func drawPNGJANUPC(_ pattern: [Int], displayCode: String, width: Int, height: Int, clsName: String) throws {
        let p = computeJANUPCParams(height: height, tfScale: textFontScale,
                                     tvoScale: textVerticalOffsetScale,
                                     pattern: pattern, width: width, clsName: clsName)

        let img = PixelBuffer(width: width, height: height)
        img.fill(backColor)

        var currentX = p.barcodeStartX
        var isBar = true

        // Draw bars based on barcode type
        switch clsName {
        case "JAN8":
            var segIdx = 0, segCnt = 0
            for v in pattern {
                var barW = Double(v) * p.moduleWidth
                if barW < 1 { barW = 1 }
                if segIdx == 0 && segCnt >= 3 {
                    segIdx += 1; segCnt = 0
                } else if segIdx == 1 && segCnt >= 16 {
                    segIdx += 1; segCnt = 0
                } else if segIdx == 2 && segCnt >= 5 {
                    segIdx += 1; segCnt = 0
                } else if segIdx == 3 && segCnt >= 16 {
                    segIdx += 1; segCnt = 0
                }
                let isTall = (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
                if isBar {
                    let bh = isTall ? p.tallBarHeight : p.normalBarHeight
                    let x1 = Int((currentX).rounded())
                    let x2 = Int((currentX + barW).rounded())
                    if x2 > x1 {
                        img.fillRect(x1, 0, x2, Int(bh.rounded()), foreColor)
                    }
                }
                segCnt += 1
                currentX += barW
                isBar = !isBar
            }

        case "JAN13":
            var segIdx = 0, segCnt = 0
            for v in pattern {
                var barW = Double(v) * p.moduleWidth
                if barW < 1 { barW = 1 }
                if segIdx == 0 && segCnt >= 3 {
                    segIdx += 1; segCnt = 0
                } else if segIdx == 1 && segCnt >= 24 {
                    segIdx += 1; segCnt = 0
                } else if segIdx == 2 && segCnt >= 5 {
                    segIdx += 1; segCnt = 0
                } else if segIdx == 3 && segCnt >= 24 {
                    segIdx += 1; segCnt = 0
                }
                let isTall = (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
                if isBar {
                    let bh = isTall ? p.tallBarHeight : p.normalBarHeight
                    let x1 = Int((currentX).rounded())
                    let x2 = Int((currentX + barW).rounded())
                    if x2 > x1 {
                        img.fillRect(x1, 0, x2, Int(bh.rounded()), foreColor)
                    }
                }
                segCnt += 1
                currentX += barW
                isBar = !isBar
            }

        case "UPCA":
            var segIdx = 0, segCnt = 0
            for v in pattern {
                var barW = Double(v) * p.moduleWidth
                if barW < 1 { barW = 1 }
                if segIdx == 0 && segCnt >= 7 {
                    segIdx += 1; segCnt = 0
                } else if segIdx == 1 && segCnt >= 21 {
                    segIdx += 1; segCnt = 0
                } else if segIdx == 2 && segCnt >= 3 {
                    segIdx += 1; segCnt = 0
                } else if segIdx == 3 && segCnt >= 21 {
                    segIdx += 1; segCnt = 0
                }
                let isTall = (segIdx == 0 || segIdx == 2 || segIdx == 4) && isBar
                if isBar {
                    let bh = isTall ? p.tallBarHeight : p.normalBarHeight
                    let x1 = Int((currentX).rounded())
                    let x2 = Int((currentX + barW).rounded())
                    if x2 > x1 {
                        img.fillRect(x1, 0, x2, Int(bh.rounded()), foreColor)
                    }
                }
                segCnt += 1
                currentX += barW
                isBar = !isBar
            }

        case "UPCE":
            var accUnits = 0
            for v in pattern {
                var barW = Double(v) * p.moduleWidth
                if barW < 1 { barW = 1 }
                let isStartGuard = accUnits < 3
                let isEndGuard = accUnits >= 45
                let isTall = (isStartGuard || isEndGuard) && isBar
                if isBar {
                    let bh = isTall ? p.tallBarHeight : p.normalBarHeight
                    let x1 = Int((currentX).rounded())
                    let x2 = Int((currentX + barW).rounded())
                    if x2 > x1 {
                        img.fillRect(x1, 0, x2, Int(bh.rounded()), foreColor)
                    }
                }
                accUnits += v
                currentX += barW
                isBar = !isBar
            }

        default:
            break
        }

        // Draw text using font
        let fontSize = p.svgFontSize
        if let font = getDefaultFont() {
            let textY = Int(p.textY.rounded())
            let pixelSize = Double(fontSize)
            let ascent = font.ascent(pixelSize: pixelSize)

            switch clsName {
            case "JAN8":
                if displayCode.count >= 8 {
                    let chars = Array(displayCode)
                    drawJANUPCTextPNG(img, font: font, text: String(chars[0..<4]),
                                      startPos: p.barcodeStartX + 3 * p.moduleWidth,
                                      endPos: p.barcodeStartX + 31 * p.moduleWidth,
                                      margin: p.leftMargin, textY: textY, count: 4,
                                      height: height)
                    drawJANUPCTextPNG(img, font: font, text: String(chars[4..<8]),
                                      startPos: p.barcodeStartX + 36 * p.moduleWidth,
                                      endPos: p.barcodeStartX + 64 * p.moduleWidth,
                                      margin: p.leftMargin, textY: textY, count: 4,
                                      height: height)
                }

            case "JAN13":
                if displayCode.count >= 13 {
                    let chars = Array(displayCode)
                    drawCharCenteredPNGJANUPC(img, font: font,
                                              ch: chars[0],
                                              cx: Int((p.posA + p.digitWidth * 0.3).rounded()),
                                              y: textY, fontSize: fontSize, ascent: ascent, color: foreColor)
                    drawJANUPCTextPNG(img, font: font, text: String(chars[1..<7]),
                                      startPos: p.barcodeStartX + 3 * p.moduleWidth,
                                      endPos: p.barcodeStartX + 45 * p.moduleWidth,
                                      margin: p.leftMargin, textY: textY, count: 6,
                                      height: height)
                    drawJANUPCTextPNG(img, font: font, text: String(chars[7..<13]),
                                      startPos: p.barcodeStartX + 50 * p.moduleWidth,
                                      endPos: p.barcodeStartX + 92 * p.moduleWidth,
                                      margin: p.leftMargin, textY: textY, count: 6,
                                      height: height)
                }

            case "UPCA":
                if displayCode.count >= 12 {
                    let chars = Array(displayCode)
                    drawCharCenteredPNGJANUPC(img, font: font,
                                              ch: chars[0],
                                              cx: Int((p.posA + p.digitWidth * 0.3).rounded()),
                                              y: textY, fontSize: fontSize, ascent: ascent, color: foreColor)
                    drawJANUPCTextPNG(img, font: font, text: String(chars[1..<6]),
                                      startPos: p.barcodeStartX + 10 * p.moduleWidth,
                                      endPos: p.barcodeStartX + 31 * p.moduleWidth,
                                      margin: p.leftMargin, textY: textY, count: 5,
                                      height: height)
                    drawJANUPCTextPNG(img, font: font, text: String(chars[6..<11]),
                                      startPos: p.barcodeStartX + 34 * p.moduleWidth,
                                      endPos: p.barcodeStartX + 55 * p.moduleWidth,
                                      margin: p.leftMargin, textY: textY, count: 5,
                                      height: height)
                    let endX = p.barcodeStartX + p.barcodeWidth
                    drawCharCenteredPNGJANUPC(img, font: font,
                                              ch: chars[11],
                                              cx: Int((endX + p.digitWidth * 0.7).rounded()),
                                              y: textY, fontSize: fontSize, ascent: ascent, color: foreColor)
                }

            case "UPCE":
                if displayCode.count >= 8 {
                    let chars = Array(displayCode)
                    drawCharCenteredPNGJANUPC(img, font: font,
                                              ch: chars[0],
                                              cx: Int((p.posA + p.digitWidth * 0.3).rounded()),
                                              y: textY, fontSize: fontSize, ascent: ascent, color: foreColor)
                    let startB = p.barcodeStartX + 3 * p.moduleWidth + p.charWidth * 0.5
                    let endC = p.barcodeStartX + 45 * p.moduleWidth - p.charWidth * 0.5
                    drawJANUPCTextPNG(img, font: font, text: String(chars[1..<7]),
                                      startPos: startB, endPos: endC,
                                      margin: p.leftMargin, textY: textY, count: 6,
                                      height: height)
                    let endX = p.barcodeStartX + p.barcodeWidth
                    drawCharCenteredPNGJANUPC(img, font: font,
                                              ch: chars[7],
                                              cx: Int((endX + p.digitWidth * 0.7).rounded()),
                                              y: textY, fontSize: fontSize, ascent: ascent, color: foreColor)
                }

            default:
                break
            }
        }

        if isTrialMode() {
            drawSampleOverlayPNG(img, x: 0, y: 0, width: width, height: height)
        }

        try encodeImageBuffer(img)
    }

    /// Draws JAN/UPC text section in PNG mode.
    private func drawJANUPCTextPNG(_ img: PixelBuffer, font: TrueTypeFont, text: String,
                                    startPos: Double, endPos: Double, margin: Double,
                                    textY: Int, count: Int, height: Int) {
        let section = endPos - startPos
        let effSection = section - margin * 2
        guard effSection > 0, count > 0 else { return }

        var fontSize = Int((Double(height) * 0.2 * textFontScale * 0.8).rounded())
        if fontSize < 8 { fontSize = 8 }

        let pixelSize = Double(fontSize)
        let ascent = font.ascent(pixelSize: pixelSize)
        let chars = Array(text)

        if textEvenSpacing && count > 1 {
            let spacing = effSection / Double(count - 1)
            for i in 0..<min(count, chars.count) {
                let digitX = startPos + margin + Double(i) * spacing
                drawCharCenteredPNGJANUPC(img, font: font, ch: chars[i],
                                           cx: Int(digitX.rounded()), y: textY,
                                           fontSize: fontSize, ascent: ascent, color: foreColor)
            }
        } else {
            drawTextCenteredPNGJANUPC(img, font: font, text: text,
                                      cx: Int((startPos + section / 2).rounded()), y: textY,
                                      fontSize: fontSize, color: foreColor)
        }
    }

    /// Draws a single character centered at cx for JAN/UPC PNG rendering.
    private func drawCharCenteredPNGJANUPC(_ img: PixelBuffer, font: TrueTypeFont,
                                            ch: Character, cx: Int, y: Int,
                                            fontSize: Int, ascent: Int, color: RGBA) {
        let pixelSize = Double(fontSize)
        let gi = font.glyphIndex(for: UInt16(ch.asciiValue ?? 0))
        let adv = font.advanceWidth(glyphIndex: gi)
        let advPx = Double(adv) * pixelSize / 2048.0
        let startX = cx - Int(advPx / 2)

        if let glyph = font.rasterizeGlyph(glyphIndex: gi, pixelSize: pixelSize) {
            let gx = startX + glyph.bearingX
            let gy = y + ascent - glyph.bearingY
            for gy2 in 0..<glyph.height {
                for gx2 in 0..<glyph.width {
                    let alpha = glyph.bitmap[gy2 * glyph.width + gx2]
                    if alpha > 0 {
                        img.setPixel(gx + gx2, gy + gy2, RGBA(color.r, color.g, color.b, alpha))
                    }
                }
            }
        }
    }

    /// Draws centered text for JAN/UPC PNG rendering.
    private func drawTextCenteredPNGJANUPC(_ img: PixelBuffer, font: TrueTypeFont,
                                            text: String, cx: Int, y: Int,
                                            fontSize: Int, color: RGBA) {
        let pixelSize = Double(fontSize)
        let ascent = font.ascent(pixelSize: pixelSize)

        var totalWidth = 0.0
        for ch in text {
            let gi = font.glyphIndex(for: UInt16(ch.asciiValue ?? 0))
            let adv = font.advanceWidth(glyphIndex: gi)
            totalWidth += Double(adv) * pixelSize / 2048.0
        }
        let startX = cx - Int(totalWidth / 2)

        var curX = Double(startX)
        for ch in text {
            let gi = font.glyphIndex(for: UInt16(ch.asciiValue ?? 0))
            let adv = font.advanceWidth(glyphIndex: gi)
            let advPx = Double(adv) * pixelSize / 2048.0

            if let glyph = font.rasterizeGlyph(glyphIndex: gi, pixelSize: pixelSize) {
                let gx = Int(curX.rounded()) + glyph.bearingX
                let gy = y + ascent - glyph.bearingY
                for gy2 in 0..<glyph.height {
                    for gx2 in 0..<glyph.width {
                        let alpha = glyph.bitmap[gy2 * glyph.width + gx2]
                        if alpha > 0 {
                            img.setPixel(gx + gx2, gy + gy2, RGBA(color.r, color.g, color.b, alpha))
                        }
                    }
                }
            }
            curX += advPx
        }
    }
}
