import Foundation

// MARK: - JAN8 Constants

private let jan8LeftPatterns: [Int] = [0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b]
private let jan8RightPatterns: [Int] = [0x72, 0x66, 0x6c, 0x42, 0x5c, 0x4e, 0x50, 0x44, 0x48, 0x74]

// MARK: - Check Digit

/// Computes the JAN8 check digit (Modulus 10, weight 3-1).
public func calculateCheckDigitJAN8(_ src: String) -> String {
    let chars = Array(src)
    var total = 0
    for i in 0..<7 {
        let d = Int(chars[i].asciiValue!) - 48 // '0' = 48
        if i % 2 == 0 {
            total += d * 3
        } else {
            total += d
        }
    }
    let cd = (10 - (total % 10)) % 10
    return String(cd)
}

// MARK: - JAN8 Class

/// JAN8 (EAN-8) barcode encoder.
public class JAN8: BarcodeBase1D, Encoder1D {
    public var extendedGuard: Bool = true

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    /// Returns the bar/space width pattern for JAN8.
    public func encode(_ code: String) throws -> [Int] {
        guard !code.isEmpty else {
            throw BarcodeError.emptyString
        }
        for c in code {
            guard c >= "0" && c <= "9" else {
                throw BarcodeError.invalidInput("JAN8 barcode requires numeric digits only")
            }
        }

        var s = code
        if s.count == 8 {
            let prefix7 = String(s.prefix(7))
            let expected = calculateCheckDigitJAN8(prefix7)
            let lastIdx = s.index(s.startIndex, offsetBy: 7)
            guard String(s[lastIdx...]) == expected else {
                throw BarcodeError.invalidInput("invalid check digit")
            }
        } else if s.count == 7 {
            s = s + calculateCheckDigitJAN8(s)
        } else {
            throw BarcodeError.invalidInput("JAN8 barcode requires 7 or 8 digits")
        }

        let chars = Array(s)

        var result = [Int]()
        // Start guard (101)
        result.append(contentsOf: [1, 1, 1])

        // Left 4 digits (L-pattern)
        for i in 0..<4 {
            let digit = Int(chars[i].asciiValue!) - 48
            let pattern = jan8LeftPatterns[digit]
            result.append(contentsOf: bitsToRunlength(pattern: pattern, numBits: 7))
        }

        // Center guard (01010)
        result.append(contentsOf: [1, 1, 1, 1, 1])

        // Right 4 digits (R-pattern)
        for i in 0..<4 {
            let digit = Int(chars[4 + i].asciiValue!) - 48
            let pattern = jan8RightPatterns[digit]
            result.append(contentsOf: bitsToRunlength(pattern: pattern, numBits: 7))
        }

        // End guard (101)
        result.append(contentsOf: [1, 1, 1])

        return result
    }

    /// Draws the JAN8 barcode with extended guard bars.
    public func draw(code: String, width: Int, height: Int) throws {
        let pattern = try encode(code)

        var s = code
        if s.count == 7 {
            s = s + calculateCheckDigitJAN8(s)
        }

        if isSVGOutput() {
            if showText && extendedGuard {
                drawSVGJAN8(pattern, displayCode: s, width: width, height: height)
                return
            }
            let display = showText ? s : ""
            try drawSVGBars(pattern, width: width, height: height, text: display)
        } else {
            try drawPNGJANUPC(pattern, displayCode: s, width: width, height: height, clsName: "JAN8")
        }
    }
}
