import Foundation

// MARK: - JAN13 Constants

private let jan13ParityPatterns: [Int] = [0x00, 0x0b, 0x0d, 0x0e, 0x13, 0x19, 0x1c, 0x15, 0x16, 0x1a]

private let jan13LeftPatterns: [[Int]] = [
    [0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b], // L-pattern
    [0x27, 0x33, 0x1b, 0x21, 0x1d, 0x39, 0x05, 0x11, 0x09, 0x17], // G-pattern
]

private let jan13RightPatterns: [Int] = [0x72, 0x66, 0x6c, 0x42, 0x5c, 0x4e, 0x50, 0x44, 0x48, 0x74]

// MARK: - Check Digit

/// Computes the JAN13 check digit (Modulus 10, weight 1-3).
public func calculateCheckDigitJAN13(_ src: String) -> String {
    let chars = Array(src)
    var total = 0
    for i in 0..<12 {
        let d = Int(chars[i].asciiValue!) - 48 // '0' = 48
        if i % 2 == 0 {
            total += d
        } else {
            total += d * 3
        }
    }
    let cd = (10 - (total % 10)) % 10
    return String(cd)
}

// MARK: - JAN13 Class

/// JAN13 (EAN-13) barcode encoder.
public class JAN13: BarcodeBase1D, Encoder1D {
    public var extendedGuard: Bool = true

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    /// Returns the bar/space width pattern for JAN13.
    public func encode(_ code: String) throws -> [Int] {
        guard !code.isEmpty else {
            throw BarcodeError.emptyString
        }
        for c in code {
            guard c >= "0" && c <= "9" else {
                throw BarcodeError.invalidInput("JAN13 barcode requires numeric digits only")
            }
        }

        var s = code
        if s.count == 13 {
            let prefix12 = String(s.prefix(12))
            let expected = calculateCheckDigitJAN13(prefix12)
            let lastIdx = s.index(s.startIndex, offsetBy: 12)
            guard String(s[lastIdx...]) == expected else {
                throw BarcodeError.invalidInput("invalid check digit")
            }
        } else if s.count == 12 {
            s = s + calculateCheckDigitJAN13(s)
        } else {
            throw BarcodeError.invalidInput("JAN13 barcode requires 12 or 13 digits")
        }

        let chars = Array(s)
        let firstDigit = Int(chars[0].asciiValue!) - 48

        var result = [Int]()
        // Start guard (101)
        result.append(contentsOf: [1, 1, 1])

        // Left 6 digits (L or G pattern based on first digit parity)
        let chkOE = 0x20
        for i in 0..<6 {
            let digit = Int(chars[1 + i].asciiValue!) - 48
            let flgOE = (jan13ParityPatterns[firstDigit] & (chkOE >> i)) != 0 ? 1 : 0
            let pattern = jan13LeftPatterns[flgOE][digit]
            result.append(contentsOf: bitsToRunlength(pattern: pattern, numBits: 7))
        }

        // Center guard (01010)
        result.append(contentsOf: [1, 1, 1, 1, 1])

        // Right 6 digits (R-pattern)
        for i in 0..<6 {
            let digit = Int(chars[7 + i].asciiValue!) - 48
            let pattern = jan13RightPatterns[digit]
            result.append(contentsOf: bitsToRunlength(pattern: pattern, numBits: 7))
        }

        // End guard (101)
        result.append(contentsOf: [1, 1, 1])

        return result
    }

    /// Draws the JAN13 barcode with extended guard bars.
    public func draw(code: String, width: Int, height: Int) throws {
        let pattern = try encode(code)

        var s = code
        if s.count == 12 {
            s = s + calculateCheckDigitJAN13(s)
        }

        if isSVGOutput() {
            if showText && extendedGuard {
                drawSVGJAN13(pattern, displayCode: s, width: width, height: height)
                return
            }
            let display = showText ? s : ""
            try drawSVGBars(pattern, width: width, height: height, text: display)
        } else {
            try drawPNGJANUPC(pattern, displayCode: s, width: width, height: height, clsName: "JAN13")
        }
    }
}
