import Foundation

private let itfDigitPatterns: [Int] = [
    0x06, 0x11, 0x09, 0x18, 0x05, 0x14, 0x0c, 0x03, 0x12, 0x0a,
]

/// ITF (Interleaved 2 of 5) barcode encoder.
public class ITF: BarcodeBase1D, Encoder1D {
    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    public func encode(_ code: String) throws -> [Int] {
        if code.isEmpty {
            throw BarcodeError.emptyString
        }

        for ch in code {
            if ch < "0" || ch > "9" {
                throw BarcodeError.invalidInput("ITF barcode requires numeric digits only")
            }
        }

        var s = code
        if s.count % 2 != 0 {
            s = "0" + s
        }

        let wide = minLineWidth % 2 != 0 ? 3 : 2

        var result = [Int]()
        // Start pattern
        result.append(contentsOf: [1, 1, 1, 1])

        // Encode digit pairs
        let bytes = Array(s.utf8)
        var i = 0
        while i < bytes.count {
            let d1 = Int(bytes[i] - UInt8(ascii: "0"))
            let d2 = Int(bytes[i + 1] - UInt8(ascii: "0"))
            let p1 = itfDigitPatterns[d1]
            let p2 = itfDigitPatterns[d2]

            // Interleave the two 5-bit patterns
            var combined = 0x01
            for j in 0..<5 {
                let bit1 = (p1 & (0x10 >> j)) != 0 ? 1 : 0
                let bit2 = (p2 & (0x10 >> j)) != 0 ? 1 : 0
                combined = (combined << 1) | bit1
                combined = (combined << 1) | bit2
            }
            combined &= 0x3FF // 10 bits

            for j in 0..<10 {
                if (combined & (0x200 >> j)) != 0 {
                    result.append(wide)
                } else {
                    result.append(1)
                }
            }

            i += 2
        }

        // Stop pattern
        result.append(contentsOf: [wide, 1, 1])

        return result
    }

    public func draw(code: String, width: Int, height: Int) throws {
        try self.draw(code, width: width, height: height, encoder: self)
    }
}
