import Foundation

// MARK: - GS1-128 Static Data

/// GS1-128 uses the same patterns as Code128.
/// code128Patterns is defined in Code128.swift with internal access.

private let gs1128Start: [String] = ["211412", "211214", "211232"]
private let gs1128Stop = "2331112"

/// CODE-A character set: space(32) to _(95) + control chars (0-31)
private let gs1128CodeA: [UInt8] = {
    var a = [UInt8]()
    for i: UInt8 in 32...95 {
        a.append(i)
    }
    for i: UInt8 in 0..<32 {
        a.append(i)
    }
    return a
}()

/// CODE-B character set: space(32) to ~(126) + DEL(127)
private let gs1128CodeB: [UInt8] = {
    var b = [UInt8]()
    for i: UInt8 in 32...126 {
        b.append(i)
    }
    b.append(0x7f)
    return b
}()

/// CODE-C character set: "00" to "99"
private let gs1128CodeC: [String] = {
    var c = [String]()
    for i in 0..<100 {
        c.append(String(format: "%02d", i))
    }
    return c
}()

/// Characters that only exist in CODE-B (lowercase + special)
private let gs1128OnlyB = "`abcdefghijklmnopqrstuvwxyz{|}~"

/// Escape/control characters (0-31)
private let gs1128EscChars: [UInt8] = {
    var e = [UInt8]()
    for i: UInt8 in 0..<32 {
        e.append(i)
    }
    return e
}()

// MARK: - GS1-128 Helper Functions

private func gs1128IsInt4(_ bytes: [UInt8], _ pos: Int) -> Bool {
    if pos + 4 > bytes.count {
        return false
    }
    for i in pos..<(pos + 4) {
        if bytes[i] < UInt8(ascii: "0") || bytes[i] > UInt8(ascii: "9") {
            return false
        }
    }
    return true
}

private func gs1128GetABC(_ bytes: [UInt8]) -> Int {
    for c in bytes {
        if c < 32 {
            return 0 // CODE-A
        }
        if gs1128OnlyB.utf8.contains(c) {
            return 1 // CODE-B
        }
    }
    return 1 // default B
}

private func gs1128SafeSub(_ bytes: [UInt8], _ pos: Int, _ length: Int) -> [UInt8] {
    if pos >= bytes.count {
        return []
    }
    let end = min(pos + length, bytes.count)
    return Array(bytes[pos..<end])
}

private func gs1128SafeSubStr(_ bytes: [UInt8], _ pos: Int, _ length: Int) -> String {
    let sub = gs1128SafeSub(bytes, pos, length)
    return String(bytes: sub, encoding: .ascii) ?? ""
}

// MARK: - GS1128 Class

/// GS1-128 barcode encoder.
public class GS1128: BarcodeBase1D, Encoder1D {
    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
        self.fitWidth = true
    }

    public func encode(_ code: String) throws -> [Int] {
        guard !code.isEmpty else {
            throw BarcodeError.emptyString
        }

        let bytes = Array(code.utf8)
        let swABC = determineStartSet(bytes, code)

        var allPattern = [Int]()

        // Start code
        let startPtn = gs1128Start[swABC]
        for ch in startPtn {
            allPattern.append(Int(String(ch))!)
        }

        // Data encoding
        var swABC2 = swABC
        var i = 0
        while i < bytes.count {
            var chgFlg = false
            if swABC2 == 0 { // CODE-A
                if gs1128IsInt4(bytes, i) {
                    swABC2 = 2
                    chgFlg = true
                }
                if i < bytes.count && gs1128OnlyB.utf8.contains(bytes[i]) {
                    swABC2 = 1
                    chgFlg = true
                }
            } else if swABC2 == 1 { // CODE-B
                if gs1128IsInt4(bytes, i) {
                    swABC2 = 2
                    chgFlg = true
                }
                if i < bytes.count && gs1128EscChars.contains(bytes[i]) {
                    swABC2 = 0
                    chgFlg = true
                }
            } else { // CODE-C
                let c2 = gs1128SafeSubStr(bytes, i, 2)
                if c2.count < 2 {
                    swABC2 = gs1128GetABC(Array(bytes[i...]))
                    chgFlg = true
                } else if c2 != "{F" && c2 != "{A" {
                    var found = false
                    for cc in gs1128CodeC {
                        if c2 == cc {
                            found = true
                            break
                        }
                    }
                    if !found {
                        swABC2 = gs1128GetABC(Array(bytes[i...]))
                        chgFlg = true
                    }
                }
            }

            if chgFlg {
                var swCode = 99
                if swABC2 == 0 {
                    swCode = 101
                } else if swABC2 == 1 {
                    swCode = 100
                }
                let ptn = code128Patterns[swCode]
                for ch in ptn {
                    allPattern.append(Int(String(ch))!)
                }
            }

            // Encode character
            var idx = 0
            if i + 6 <= bytes.count && String(bytes: Array(bytes[i..<(i+6)]), encoding: .utf8) == "{FNC1}" {
                idx = 102
                i += 6
            } else if i + 4 <= bytes.count && String(bytes: Array(bytes[i..<(i+4)]), encoding: .utf8) == "{AI}" {
                i += 4
                continue
            } else if swABC2 == 0 { // CODE-A
                if let pos = gs1128CodeA.firstIndex(of: bytes[i]) {
                    idx = pos
                }
                i += 1
            } else if swABC2 == 1 { // CODE-B
                if let pos = gs1128CodeB.firstIndex(of: bytes[i]) {
                    idx = pos
                }
                i += 1
            } else { // CODE-C
                let c2 = gs1128SafeSubStr(bytes, i, 2)
                if c2.count < 2 {
                    if let pos = gs1128CodeB.firstIndex(of: bytes[i]) {
                        idx = pos
                    }
                    i += 1
                } else {
                    for (ci, cc) in gs1128CodeC.enumerated() {
                        if c2 == cc {
                            idx = ci
                            break
                        }
                    }
                    i += 2
                }
            }

            let ptn = code128Patterns[idx]
            for ch in ptn {
                allPattern.append(Int(String(ch))!)
            }
        }

        // Check digit
        let cdIdx = calcCheckDigit(bytes, code)
        let cdPtn = code128Patterns[cdIdx]
        for ch in cdPtn {
            allPattern.append(Int(String(ch))!)
        }

        // Stop code
        for ch in gs1128Stop {
            allPattern.append(Int(String(ch))!)
        }

        return allPattern
    }

    private func determineStartSet(_ bytes: [UInt8], _ code: String) -> Int {
        var cntNum = 0
        var cntF = 0
        var cntAI = 0

        while cntNum + cntF + cntAI < bytes.count - 1 {
            let pos = cntNum + cntF + cntAI
            if pos + 6 <= bytes.count &&
               String(bytes: Array(bytes[pos..<(pos+6)]), encoding: .utf8) == "{FNC1}" {
                cntF += 6
                continue
            }
            if pos + 4 <= bytes.count &&
               String(bytes: Array(bytes[pos..<(pos+4)]), encoding: .utf8) == "{AI}" {
                cntAI += 4
                continue
            }
            if pos >= bytes.count {
                break
            }
            if bytes[pos] < UInt8(ascii: "0") || bytes[pos] > UInt8(ascii: "9") {
                break
            }
            cntNum += 1
        }

        if cntNum < 3 {
            return gs1128GetABC(bytes)
        }
        return 2 // CODE-C
    }

    private func calcCheckDigit(_ bytes: [UInt8], _ code: String) -> Int {
        var cntNum = 0
        var cntF = 0
        var cntAI = 0

        while cntNum + cntF + cntAI < bytes.count - 1 {
            let pos = cntNum + cntF + cntAI
            if pos + 6 <= bytes.count &&
               String(bytes: Array(bytes[pos..<(pos+6)]), encoding: .utf8) == "{FNC1}" {
                cntF += 6
                continue
            }
            if pos + 4 <= bytes.count &&
               String(bytes: Array(bytes[pos..<(pos+4)]), encoding: .utf8) == "{AI}" {
                cntAI += 4
                continue
            }
            if pos >= bytes.count {
                break
            }
            if bytes[pos] < UInt8(ascii: "0") || bytes[pos] > UInt8(ascii: "9") {
                break
            }
            cntNum += 1
        }

        var swABC = 0
        var charSum = 0
        if cntNum < 3 {
            swABC = gs1128GetABC(bytes)
            if swABC == 1 {
                charSum = 104
            } else {
                charSum = 103
            }
        } else {
            swABC = 2
            charSum = 105
        }

        var cnt = 0
        var i = 0
        while i < bytes.count {
            if swABC == 0 { // CODE-A
                if gs1128IsInt4(bytes, i) {
                    swABC = 2
                    cnt += 1
                    charSum += 99 * cnt
                }
                if i < bytes.count && gs1128OnlyB.utf8.contains(bytes[i]) {
                    swABC = 1
                    cnt += 1
                    charSum += 100 * cnt
                }
            } else if swABC == 1 { // CODE-B
                if gs1128IsInt4(bytes, i) {
                    swABC = 2
                    cnt += 1
                    charSum += 99 * cnt
                }
                if i < bytes.count && gs1128EscChars.contains(bytes[i]) {
                    swABC = 0
                    cnt += 1
                    charSum += 101 * cnt
                }
            } else { // CODE-C
                let c2 = gs1128SafeSubStr(bytes, i, 2)
                if c2.count < 2 {
                    swABC = gs1128GetABC(Array(bytes[i...]))
                    cnt += 1
                    if swABC == 1 {
                        charSum += 100 * cnt
                    } else {
                        charSum += 101 * cnt
                    }
                } else if c2 != "{F" && c2 != "{A" {
                    var found = false
                    for cc in gs1128CodeC {
                        if c2 == cc {
                            found = true
                            break
                        }
                    }
                    if !found {
                        swABC = gs1128GetABC(Array(bytes[i...]))
                        cnt += 1
                        if swABC == 1 {
                            charSum += 100 * cnt
                        } else {
                            charSum += 101 * cnt
                        }
                    }
                }
            }

            var idx = -1
            if i + 6 <= bytes.count &&
               String(bytes: Array(bytes[i..<(i+6)]), encoding: .utf8) == "{FNC1}" {
                idx = 102
                i += 6
            } else if i + 4 <= bytes.count &&
                      String(bytes: Array(bytes[i..<(i+4)]), encoding: .utf8) == "{AI}" {
                idx = -1
                i += 4
            } else if swABC == 0 { // CODE-A
                let c = gs1128SafeSub(bytes, i, 1)
                if !c.isEmpty {
                    if let pos = gs1128CodeA.firstIndex(of: c[0]) {
                        idx = pos
                    }
                }
                i += 1
            } else if swABC == 1 { // CODE-B
                let c = gs1128SafeSub(bytes, i, 1)
                if !c.isEmpty {
                    if let pos = gs1128CodeB.firstIndex(of: c[0]) {
                        idx = pos
                    }
                }
                i += 1
            } else { // CODE-C
                let c2 = gs1128SafeSubStr(bytes, i, 2)
                if c2.count < 2 {
                    if let pos = gs1128CodeB.firstIndex(of: bytes[i]) {
                        idx = pos
                    }
                    i += 1
                } else {
                    for (ci, cc) in gs1128CodeC.enumerated() {
                        if c2 == cc {
                            idx = ci
                            break
                        }
                    }
                    i += 2
                }
            }

            if idx != -1 {
                cnt += 1
                charSum += idx * cnt
            }
        }

        return charSum % 103
    }

    /// Calculates Modulus 10 Weight 3-1 check digit.
    public static func calcCheckDigitMod10W3(_ digits: String) -> Int {
        let bytes = Array(digits.utf8)
        var total = 0
        var weight3 = true
        for i in stride(from: bytes.count - 1, through: 0, by: -1) {
            let d = Int(bytes[i] - UInt8(ascii: "0"))
            if weight3 {
                total += d * 3
            } else {
                total += d
            }
            weight3 = !weight3
        }
        let mod = total % 10
        if mod == 0 {
            return 0
        }
        return 10 - mod
    }

    /// Draws the GS1-128 barcode.
    /// GS1-128 has custom draw logic: display text strips {FNC1} and {AI} markers.
    public func draw(code: String, width: Int, height: Int) throws {
        let pattern = try encode(code)

        var display = ""
        if showText {
            display = code.replacingOccurrences(of: "{FNC1}", with: "")
            display = display.replacingOccurrences(of: "{AI}", with: "")
        }

        if isSVGOutput() {
            try drawSVGBars(pattern, width: width, height: height, text: display)
        } else {
            try renderBarsToPNG(pattern, width: width, height: height, text: display)
        }
    }
}
