import Foundation

// ISO 24724 Limited tables
private let leftRegThresholds = [0, 183064, 820064, 1000776, 1491021, 1979845, 1996939]
private let leftRegOffsets = [0, 183064, 820064, 1000776, 1491021, 1979845, 1996939]

private let tEvenLtd = [28, 728, 6454, 203, 2408, 1, 16632]
private let modulesOddLtd = [17, 13, 9, 15, 11, 19, 7]
private let modulesEvenLtd = [9, 13, 17, 11, 15, 7, 19]
private let widestOddLtd = [6, 5, 3, 5, 4, 8, 1]
private let widestEvenLtd = [3, 4, 6, 4, 5, 1, 8]

private let checksumWeightLtd = [
    1, 3, 9, 27, 81, 65, 17, 51, 64, 14,
    42, 37, 22, 66, 20, 60, 2, 6, 18, 54,
    73, 41, 34, 13, 39, 28, 84, 74,
]

private let finderPatternLtd = [
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 3, 2, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 3, 2, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 2, 1, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 3, 2, 1, 1,
    1, 1, 1, 1, 1, 2, 1, 1, 1, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 2, 1, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 3, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
    1, 1, 1, 2, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 3, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
    1, 2, 1, 1, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 3, 1, 1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 3, 2, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 2, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 3, 2, 1, 1,
    1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 3, 2, 1, 1,
    1, 1, 1, 1, 2, 1, 1, 1, 1, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 1, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 3, 2, 1, 1,
    1, 1, 2, 1, 1, 1, 1, 1, 1, 2, 3, 1, 1, 1,
    1, 1, 2, 1, 1, 1, 1, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 1, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 1, 2, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 2, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 2, 2, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 3, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 1, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 2, 1, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 1, 1, 1, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 1, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 2, 2, 1, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 1, 2, 1, 2, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 2, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 2, 2, 2, 2, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 3, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 2, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 2, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 2, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 2, 2, 2, 1, 2, 2, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 1, 2, 2, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 2, 2, 2, 2, 1, 2, 1, 1, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 3, 1, 2, 1, 2, 2, 3, 1, 1, 1,
    1, 1, 1, 1, 3, 1, 2, 2, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 2, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 1, 1, 1, 3, 2, 2, 1, 2, 1, 3, 1, 1, 1,
    1, 2, 1, 1, 3, 1, 2, 1, 2, 1, 3, 1, 1, 1,
]

private func determineGroupLtd(_ regValue: Int) -> Int {
    var group = 0
    if regValue > 183063 { group = 1 }
    if regValue > 820063 { group = 2 }
    if regValue > 1000775 { group = 3 }
    if regValue > 1491020 { group = 4 }
    if regValue > 1979844 { group = 5 }
    if regValue > 1996938 { group = 6 }
    return group
}

/// GS1 DataBar Limited barcode encoder.
public class GS1DataBarLimited: BarcodeBase1D {
    private(set) public var gtin14: String = ""
    private var totalWidths: [Int] = []

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
        self.showText = false
    }

    /// Returns the human-readable text "(01)GTIN-14".
    public func humanReadable() -> String { "(01)" + gtin14 }

    /// Encodes GS1 DataBar Limited.
    public func encode(_ code: String) throws -> [Int] {
        for c in code.unicodeScalars {
            if c < "0" || c > "9" {
                throw BarcodeError.invalidInput("GS1 DataBar Limited requires numeric digits only")
            }
        }
        if code.count < 8 || code.count > 13 {
            throw BarcodeError.invalidInput("input must be 8-13 digits")
        }

        let padded = padLeft(code, 13)
        let paddedChars = Array(padded.utf8)

        if paddedChars[0] != UInt8(ascii: "0") && paddedChars[0] != UInt8(ascii: "1") {
            throw BarcodeError.invalidInput("GS1 DataBar Limited requires leading digit 0 or 1 after padding")
        }

        let accum = parseInt64(padded)
        var leftReg = Int(accum / 2013571)
        var rightReg = Int(accum % 2013571)

        let leftGroup = determineGroupLtd(leftReg)
        let rightGroup = determineGroupLtd(rightReg)

        if leftGroup > 0 {
            leftReg -= leftRegOffsets[leftGroup]
        }
        if rightGroup > 0 {
            rightReg -= leftRegOffsets[rightGroup]
        }

        let leftOdd = leftReg / tEvenLtd[leftGroup]
        let leftEven = leftReg % tEvenLtd[leftGroup]
        let rightOdd = rightReg / tEvenLtd[rightGroup]
        let rightEven = rightReg % tEvenLtd[rightGroup]

        // Generate width patterns (14 elements each)
        var leftWidths = [Int](repeating: 0, count: 14)
        var rightWidths = [Int](repeating: 0, count: 14)

        var w = getWidths(leftOdd, modulesOddLtd[leftGroup], 7, widestOddLtd[leftGroup], 1)
        for i in 0..<7 {
            leftWidths[i * 2] = w[i]
        }
        w = getWidths(leftEven, modulesEvenLtd[leftGroup], 7, widestEvenLtd[leftGroup], 0)
        for i in 0..<7 {
            leftWidths[i * 2 + 1] = w[i]
        }
        w = getWidths(rightOdd, modulesOddLtd[rightGroup], 7, widestOddLtd[rightGroup], 1)
        for i in 0..<7 {
            rightWidths[i * 2] = w[i]
        }
        w = getWidths(rightEven, modulesEvenLtd[rightGroup], 7, widestEvenLtd[rightGroup], 0)
        for i in 0..<7 {
            rightWidths[i * 2 + 1] = w[i]
        }

        // Checksum (modulo 89)
        var checksum = 0
        for i in 0..<14 {
            checksum += checksumWeightLtd[i] * leftWidths[i]
            checksum += checksumWeightLtd[i + 14] * rightWidths[i]
        }
        checksum %= 89

        let checkStart = checksum * 14
        let checkElements = Array(finderPatternLtd[checkStart..<(checkStart + 14)])

        // Build total widths (46 elements)
        var tw = [Int](repeating: 0, count: 46)
        tw[0] = 1
        tw[1] = 1
        tw[44] = 1
        tw[45] = 1

        for i in 0..<14 {
            tw[i + 2] = leftWidths[i]
        }
        for i in 0..<14 {
            tw[i + 16] = checkElements[i]
        }
        for i in 0..<14 {
            tw[i + 30] = rightWidths[i]
        }

        let cd = calculateGTINCheckDigit(padded)
        let cdChar = Character(UnicodeScalar(UInt8(ascii: "0") + UInt8(cd)))
        gtin14 = padded + String(cdChar)
        totalWidths = tw

        return tw
    }

    /// Draws the GS1 DataBar Limited barcode.
    public func draw(code: String, width: Int, height: Int) throws {
        let pattern = try encode(code)
        // DataBar Limited is single-row, starts with space
        if isSVGOutput() {
            try drawSVGBars(pattern, width: width, height: height, text: "")
        } else {
            try renderBarsToPNG(pattern, width: width, height: height, text: "")
        }
    }
}
