import Foundation

/// ExpandedSymbolType specifies the GS1 DataBar Expanded variant.
public enum ExpandedSymbolType: Int {
    case unstacked  = 0
    case stacked    = 1
}

// Encoding modes
private let encNumeric    = 0
private let encAlpha      = 1
private let encISO        = 2
private let encInvalid    = 3
private let encAnyEnc     = 4
private let encAlphaOrISO = 5

// ISO 24724 Tables for GS1 DataBar Expanded
private let gSumExp = [0, 348, 1388, 2948, 3988]
private let tEvenExp = [4, 20, 52, 104, 204]
private let modulesOddExp = [12, 10, 8, 6, 4]
private let modulesEvenExp = [5, 7, 9, 11, 13]
private let widestOddExp = [7, 5, 4, 3, 1]
private let widestEvenExp = [2, 4, 5, 6, 8]

private let checksumWeightExp = [
    1, 3, 9, 27, 81, 32, 96, 77, 20, 60, 180, 118, 143, 7, 21, 63, 189,
    145, 13, 39, 117, 140, 209, 205, 193, 157, 49, 147, 19, 57, 171, 91,
    62, 186, 136, 197, 169, 85, 44, 132, 185, 133, 188, 142, 4, 12, 36,
    108, 113, 128, 173, 97, 80, 29, 87, 50, 150, 28, 84, 41, 123, 158, 52,
    156, 46, 138, 203, 187, 139, 206, 196, 166, 76, 17, 51, 153, 37, 111,
    122, 155, 43, 129, 176, 106, 107, 110, 119, 146, 16, 48, 144, 10, 30,
    90, 59, 177, 109, 116, 137, 200, 178, 112, 125, 164, 70, 210, 208, 202,
    184, 130, 179, 115, 134, 191, 151, 31, 93, 68, 204, 190, 148, 22, 66,
    198, 172, 94, 71, 2, 6, 18, 54, 162, 64, 192, 154, 40, 120, 149, 25,
    75, 14, 42, 126, 167, 79, 26, 78, 23, 69, 207, 199, 175, 103, 98, 83,
    38, 114, 131, 182, 124, 161, 61, 183, 127, 170, 88, 53, 159, 55, 165,
    73, 8, 24, 72, 5, 15, 45, 135, 194, 160, 58, 174, 100, 89,
]

private let finderPatternExp = [
    1, 8, 4, 1, 1, 1, 1, 4, 8, 1, 3, 6, 4, 1, 1, 1, 1, 4, 6, 3, 3, 4, 6, 1,
    1, 1, 1, 6, 4, 3, 3, 2, 8, 1, 1, 1, 1, 8, 2, 3, 2, 6, 5, 1, 1, 1, 1, 5,
    6, 2, 2, 2, 9, 1, 1, 1, 1, 9, 2, 2,
]

private let finderSequence = [
    1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 4, 3, 0, 0, 0, 0, 0, 0, 0, 0, 1, 6,
    3, 8, 0, 0, 0, 0, 0, 0, 0, 1, 10, 3, 8, 5, 0, 0, 0, 0, 0, 0, 1, 10, 3,
    8, 7, 12, 0, 0, 0, 0, 0, 1, 10, 3, 8, 9, 12, 11, 0, 0, 0, 0, 1, 2, 3,
    4, 5, 6, 7, 8, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 10, 9, 0, 0, 1, 2, 3, 4,
    5, 6, 7, 10, 11, 12, 0, 1, 2, 3, 4, 5, 8, 7, 10, 9, 12, 11,
]

private let weightRows = [
    0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 6,
    3, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 10, 3, 4,
    13, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13,
    14, 7, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13, 14,
    11, 12, 21, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 18, 3, 4, 13, 14,
    15, 16, 21, 22, 19, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7,
    8, 9, 10, 11, 12, 13, 14, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
    9, 10, 11, 12, 17, 18, 15, 16, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
    9, 10, 11, 12, 17, 18, 19, 20, 21, 22, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8,
    13, 14, 11, 12, 17, 18, 15, 16, 21, 22, 19, 20,
]

/// ISO/IEC 646 special character encoding table
private let isoChars: [UInt8: String] = [
    UInt8(ascii: "!"): "11101000", UInt8(ascii: "\""): "11101001", UInt8(ascii: "%"): "11101010",
    UInt8(ascii: "&"): "11101011", UInt8(ascii: "'"): "11101100", UInt8(ascii: "("): "11101101",
    UInt8(ascii: ")"): "11101110", UInt8(ascii: "*"): "11101111", UInt8(ascii: "+"): "11110000",
    UInt8(ascii: ","): "11110001", UInt8(ascii: "-"): "11110010", UInt8(ascii: "."): "11110011",
    UInt8(ascii: "/"): "11110100", UInt8(ascii: ":"): "11110101", UInt8(ascii: ";"): "11110110",
    UInt8(ascii: "<"): "11110111", UInt8(ascii: "="): "11111000", UInt8(ascii: ">"): "11111001",
    UInt8(ascii: "?"): "11111010", UInt8(ascii: "_"): "11111011", UInt8(ascii: " "): "11111100",
]

private func intToBits(_ value: Int, _ numBits: Int) -> String {
    var result = [UInt8](repeating: UInt8(ascii: "0"), count: numBits)
    for j in 0..<numBits {
        if value & (1 << (numBits - 1 - j)) != 0 {
            result[j] = UInt8(ascii: "1")
        }
    }
    return String(bytes: result, encoding: .ascii)!
}

private func calculateRemainder(_ binaryStringLength: Int) -> Int {
    var remainder = 12 - (binaryStringLength % 12)
    if remainder == 12 { remainder = 0 }
    if binaryStringLength < 36 {
        remainder = 36 - binaryStringLength
    }
    return remainder
}

private func bin2patFromBlack(_ binary: String) -> String {
    if binary.isEmpty { return "" }
    var pattern = bin2pat(binary)
    if binary.first == "0" {
        pattern = "0" + pattern
    }
    return pattern
}

/// GS1 DataBar Expanded barcode encoder.
public class GS1DataBarExpanded: BarcodeBase1D {
    public var symbolType: ExpandedSymbolType
    public var numColumns: Int
    private(set) public var humanReadableText: String = ""
    private var elements: [Int] = []
    private(set) public var patterns: [String] = []
    private(set) public var rowHeights: [Int] = []
    private(set) public var rowCount: Int = 0

    // Internal state
    private var binaryString: String = ""
    private var generalField: [UInt8] = []
    private var generalFieldType: [Int] = []
    private var encMethodResult: Int = 0

    public init(outputFormat: String, symbolType: ExpandedSymbolType = .unstacked, numColumns: Int = 2) {
        self.symbolType = symbolType
        self.numColumns = numColumns
        super.init(outputFormat: outputFormat)
        self.showText = false
    }

    /// Returns the human-readable text.
    public func humanReadable() -> String { humanReadableText }

    // MARK: - General Field Rules

    /// Applies ISO 24724 general field encoding rules.
    /// Returns true if last block is numeric and odd size.
    private func applyGeneralFieldRules() -> Bool {
        let n = generalField.count
        if n == 0 { return false }

        // Build blocks
        var blockLen = [1]
        var blockType = [generalFieldType[0]]
        for i in 1..<n {
            if generalFieldType[i] == generalFieldType[i - 1] {
                blockLen[blockLen.count - 1] += 1
            } else {
                blockLen.append(1)
                blockType.append(generalFieldType[i])
            }
        }
        var blockCount = blockLen.count

        // Apply rules
        for i in 0..<blockCount {
            let cur = blockType[i]
            let nxt = (i < blockCount - 1) ? blockType[i + 1] : -1

            if cur == encISO && i != blockCount - 1 {
                if nxt == encAnyEnc && blockLen[i + 1] >= 4 {
                    blockType[i + 1] = encNumeric
                }
                if nxt == encAnyEnc && blockLen[i + 1] < 4 {
                    blockType[i + 1] = encISO
                }
                if nxt == encAlphaOrISO && blockLen[i + 1] >= 5 {
                    blockType[i + 1] = encAlpha
                }
                if nxt == encAlphaOrISO && blockLen[i + 1] < 5 {
                    blockType[i + 1] = encISO
                }
            }

            if cur == encAlphaOrISO {
                blockType[i] = encAlpha
            }

            if cur == encAlpha && i != blockCount - 1 {
                if nxt == encAnyEnc && blockLen[i + 1] >= 6 {
                    blockType[i + 1] = encNumeric
                }
                if nxt == encAnyEnc && blockLen[i + 1] < 6 {
                    if i == blockCount - 2 && blockLen[i + 1] >= 4 {
                        blockType[i + 1] = encNumeric
                    } else {
                        blockType[i + 1] = encAlpha
                    }
                }
            }

            if cur == encAnyEnc {
                blockType[i] = encNumeric
            }
        }

        // Merge adjacent same-type blocks
        if blockCount > 1 {
            var i = 1
            while i < blockLen.count {
                if blockType[i - 1] == blockType[i] {
                    blockLen[i - 1] += blockLen[i]
                    blockLen.remove(at: i)
                    blockType.remove(at: i)
                    if i > 1 { i -= 1 }
                } else {
                    i += 1
                }
            }
            blockCount = blockLen.count
        }

        // Odd-size numeric blocks (not the last)
        for i in 0..<(blockCount - 1) {
            if blockType[i] == encNumeric && (blockLen[i] & 1) != 0 {
                blockLen[i] -= 1
                blockLen[i + 1] += 1
            }
        }

        // Expand back to per-character types
        var j = 0
        for i in 0..<blockCount {
            for _ in 0..<blockLen[i] {
                generalFieldType[j] = blockType[i]
                j += 1
            }
        }

        return blockType[blockCount - 1] == encNumeric && (blockLen[blockCount - 1] & 1) != 0
    }

    // MARK: - Binary String Calculation

    /// Builds the binary data string.
    private func calculateBinaryString(_ source: [UInt8], _ originalCode: String) throws {
        let aiTag: [UInt8] = Array("{AI}".utf8)
        var lastMode = encNumeric

        // Determine encoding method
        var encodingMethod = 2
        if source.count >= 16 && source[0] == UInt8(ascii: "0") && source[1] == UInt8(ascii: "1") {
            encodingMethod = 1
            humanReadableText = "(01)" + String(originalCode.dropFirst(2))
            humanReadableText = replaceAITags(humanReadableText)
        } else {
            encodingMethod = 2
            humanReadableText = originalCode
            humanReadableText = replaceAITags(humanReadableText)
        }

        // Check for compressed methods > 2
        if source.count >= 20 && encodingMethod == 1 &&
            source[2] == UInt8(ascii: "9") && source[16] == UInt8(ascii: "3") {
            checkCompressedMethods(source, encodingMethod)
            encodingMethod = encMethodResult
        }

        // Method header and read position
        let readPosn = applyMethodHeader(encodingMethod, source.count)

        // Verify compressed data is numeric
        for i in 0..<readPosn {
            let c = source[i]
            if !(c >= UInt8(ascii: "0") && c <= UInt8(ascii: "9")) &&
                c != UInt8(ascii: "[") && c != UInt8(ascii: "]") {
                throw BarcodeError.invalidInput("invalid characters in input data")
            }
        }

        // Encode compressed data field
        encodeCompressed(encodingMethod, source)

        // General purpose data field
        generalField = readPosn < source.count ? Array(source[readPosn...]) : []
        generalFieldType = [Int](repeating: encISO, count: generalField.count)

        if !generalField.isEmpty {
            lastMode = encodeGeneralField(lastMode)
        }

        // Length check
        if binaryString.count > 252 {
            throw BarcodeError.invalidInput("input too long")
        }

        // Padding
        let remainder = calculateRemainder(binaryString.count)
        var padstring = ""
        var padI = remainder
        if !generalField.isEmpty && lastMode == encNumeric {
            padstring = "0000"
            padI = remainder - 4
        }
        while padI > 0 {
            padstring += "00100"
            padI -= 5
        }
        if remainder > 0 && remainder <= padstring.count {
            let idx = padstring.index(padstring.startIndex, offsetBy: remainder)
            binaryString += String(padstring[..<idx])
        } else if remainder > 0 {
            binaryString += padstring
        }

        // Patch variable length symbol bit field
        let bs = binaryString
        let bsChars = Array(bs.utf8)
        let dataCharCount = bsChars.count / 12 + 1
        let patch0: String = (dataCharCount & 1 != 0) ? "1" : "0"
        let patch1: String = (bsChars.count > 156) ? "1" : "0"
        let patch = patch0 + patch1

        switch encodingMethod {
        case 1:
            let idx2 = bs.index(bs.startIndex, offsetBy: 2)
            let idx4 = bs.index(bs.startIndex, offsetBy: 4)
            binaryString = String(bs[..<idx2]) + patch + String(bs[idx4...])
        case 2:
            let idx3 = bs.index(bs.startIndex, offsetBy: 3)
            let idx5 = bs.index(bs.startIndex, offsetBy: 5)
            binaryString = String(bs[..<idx3]) + patch + String(bs[idx5...])
        case 5, 6:
            let idx6 = bs.index(bs.startIndex, offsetBy: 6)
            let idx8 = bs.index(bs.startIndex, offsetBy: 8)
            binaryString = String(bs[..<idx6]) + patch + String(bs[idx8...])
        default:
            break
        }
    }

    private func replaceAITags(_ input: String) -> String {
        let aiTag = "{AI}"
        var result = input
        while true {
            guard let range = result.range(of: aiTag) else { break }
            let aiStart = range.upperBound
            let aiEnd = result.index(aiStart, offsetBy: 2, limitedBy: result.endIndex) ?? result.endIndex
            let aiCode = String(result[aiStart..<aiEnd])
            let remaining = String(result[aiEnd...])
            result = String(result[..<range.lowerBound]) + "(" + aiCode + ")" + remaining
        }
        return result
    }

    private func checkCompressedMethods(_ source: [UInt8], _ baseMethod: Int) {
        encMethodResult = baseMethod

        if source.count >= 26 && source[17] == UInt8(ascii: "1") && source[18] == UInt8(ascii: "0") {
            var weight = 0.0
            for i in 0..<6 {
                weight = weight * 10 + Double(source[20 + i] - UInt8(ascii: "0"))
            }
            if weight < 99999.0 {
                if source[19] == UInt8(ascii: "3") && source.count == 26 {
                    if weight / 1000.0 <= 32.767 {
                        encMethodResult = 3
                        humanReadableText = "(01)" + String(bytes: Array(source[2..<16]), encoding: .ascii)! +
                            "(3103)" + String(bytes: Array(source[20...]), encoding: .ascii)!
                    }
                }
                if source.count == 34 {
                    setWeightDateMethod(source, weight, "310", [7, 9, 11, 13])
                }
            }
        }

        if source.count >= 26 && source[17] == UInt8(ascii: "2") && source[18] == UInt8(ascii: "0") {
            var weight = 0.0
            for i in 0..<6 {
                weight = weight * 10 + Double(source[20 + i] - UInt8(ascii: "0"))
            }
            if weight < 99999.0 {
                if (source[19] == UInt8(ascii: "2") || source[19] == UInt8(ascii: "3")) && source.count == 26 {
                    if source[19] == UInt8(ascii: "3") {
                        if weight / 1000.0 <= 22.767 {
                            encMethodResult = 4
                            humanReadableText = "(01)" + String(bytes: Array(source[2..<16]), encoding: .ascii)! +
                                "(3203)" + String(bytes: Array(source[20...]), encoding: .ascii)!
                        }
                    } else {
                        if weight / 100.0 <= 99.99 {
                            encMethodResult = 4
                            humanReadableText = "(01)" + String(bytes: Array(source[2..<16]), encoding: .ascii)! +
                                "(3202)" + String(bytes: Array(source[20...]), encoding: .ascii)!
                        }
                    }
                }
                if source.count == 34 {
                    setWeightDateMethod(source, weight, "320", [8, 10, 12, 14])
                }
            }
        }

        if source[17] == UInt8(ascii: "9") {
            if source[18] == UInt8(ascii: "2") && source[19] >= UInt8(ascii: "0") && source[19] <= UInt8(ascii: "3") {
                encMethodResult = 5
                humanReadableText = "(01)" + String(bytes: Array(source[2..<16]), encoding: .ascii)! +
                    "(392" + String(UnicodeScalar(source[19])) + ")" +
                    String(bytes: Array(source[20...]), encoding: .ascii)!
            }
            if source[18] == UInt8(ascii: "3") && source[19] >= UInt8(ascii: "0") && source[19] <= UInt8(ascii: "3") {
                encMethodResult = 6
                humanReadableText = "(01)" + String(bytes: Array(source[2..<16]), encoding: .ascii)! +
                    "(393" + String(UnicodeScalar(source[19])) + ")" +
                    String(bytes: Array(source[20...]), encoding: .ascii)!
            }
        }
    }

    private func setWeightDateMethod(_ source: [UInt8], _ weight: Double, _ aiPrefix: String, _ methods: [Int]) {
        guard source.count == 34 else { return }
        let dateAIMap: [UInt8: Int] = [
            UInt8(ascii: "1"): methods[0],
            UInt8(ascii: "3"): methods[1],
            UInt8(ascii: "5"): methods[2],
            UInt8(ascii: "7"): methods[3],
        ]
        let dateAIs: [Int: String] = [
            methods[0]: "11", methods[1]: "13", methods[2]: "15", methods[3]: "17",
        ]
        if source[26] == UInt8(ascii: "1") {
            if let m = dateAIMap[source[27]] {
                encMethodResult = m
                humanReadableText = "(01)" + String(bytes: Array(source[2..<16]), encoding: .ascii)! +
                    "(" + aiPrefix + String(UnicodeScalar(source[19])) + ")" +
                    String(bytes: Array(source[20..<26]), encoding: .ascii)! +
                    "(" + (dateAIs[m] ?? "") + ")" +
                    String(bytes: Array(source[28...]), encoding: .ascii)!
            }
        }
    }

    private func applyMethodHeader(_ encodingMethod: Int, _ sourceLen: Int) -> Int {
        struct HeaderInfo {
            let header: String
            let rp: Int
        }
        let headers: [Int: HeaderInfo] = [
            1: HeaderInfo(header: "1XX", rp: 16),
            2: HeaderInfo(header: "00XX", rp: 0),
            3: HeaderInfo(header: "0100", rp: -1),
            4: HeaderInfo(header: "0101", rp: -1),
            5: HeaderInfo(header: "01100XX", rp: 20),
            6: HeaderInfo(header: "01101XX", rp: 23),
            7: HeaderInfo(header: "0111000", rp: -1),
            8: HeaderInfo(header: "0111001", rp: -1),
            9: HeaderInfo(header: "0111010", rp: -1),
            10: HeaderInfo(header: "0111011", rp: -1),
            11: HeaderInfo(header: "0111100", rp: -1),
            12: HeaderInfo(header: "0111101", rp: -1),
            13: HeaderInfo(header: "0111110", rp: -1),
            14: HeaderInfo(header: "0111111", rp: -1),
        ]
        let h = headers[encodingMethod]!
        binaryString += h.header
        if h.rp < 0 {
            return sourceLen
        }
        return h.rp
    }

    private func encodeCompressed(_ encodingMethod: Int, _ source: [UInt8]) {
        switch encodingMethod {
        case 1:
            let groupVal = Int(source[2] - UInt8(ascii: "0"))
            binaryString += intToBits(groupVal, 4)
            for i in 1...4 {
                let gv = 100 * Int(source[i * 3] - UInt8(ascii: "0")) +
                          10 * Int(source[i * 3 + 1] - UInt8(ascii: "0")) +
                               Int(source[i * 3 + 2] - UInt8(ascii: "0"))
                binaryString += intToBits(gv, 10)
            }

        case 3, 4:
            for i in 1...4 {
                let gv = 100 * Int(source[i * 3] - UInt8(ascii: "0")) +
                          10 * Int(source[i * 3 + 1] - UInt8(ascii: "0")) +
                               Int(source[i * 3 + 2] - UInt8(ascii: "0"))
                binaryString += intToBits(gv, 10)
            }
            var gv = 0
            for i in 0..<6 {
                gv = gv * 10 + Int(source[20 + i] - UInt8(ascii: "0"))
            }
            if encodingMethod == 4 && source[19] == UInt8(ascii: "3") {
                gv += 10000
            }
            binaryString += intToBits(gv, 15)

        case 5, 6:
            for i in 1...4 {
                let gv = 100 * Int(source[i * 3] - UInt8(ascii: "0")) +
                          10 * Int(source[i * 3 + 1] - UInt8(ascii: "0")) +
                               Int(source[i * 3 + 2] - UInt8(ascii: "0"))
                binaryString += intToBits(gv, 10)
            }
            let decMap: [UInt8: String] = [
                UInt8(ascii: "0"): "00", UInt8(ascii: "1"): "01",
                UInt8(ascii: "2"): "10", UInt8(ascii: "3"): "11",
            ]
            binaryString += decMap[source[19]]!
            if encodingMethod == 6 {
                var gv = 0
                for i in 0..<3 {
                    gv = gv * 10 + Int(source[20 + i] - UInt8(ascii: "0"))
                }
                binaryString += intToBits(gv, 10)
            }

        case 7...14:
            for i in 1...4 {
                let gv = 100 * Int(source[i * 3] - UInt8(ascii: "0")) +
                          10 * Int(source[i * 3 + 1] - UInt8(ascii: "0")) +
                               Int(source[i * 3 + 2] - UInt8(ascii: "0"))
                binaryString += intToBits(gv, 10)
            }
            var gv = Int(source[19] - UInt8(ascii: "0"))
            for i in 0..<5 {
                gv = gv * 10 + Int(source[21 + i] - UInt8(ascii: "0"))
            }
            binaryString += intToBits(gv, 20)
            if source.count == 34 {
                gv = (((10 * Int(source[28] - UInt8(ascii: "0")) + Int(source[29] - UInt8(ascii: "0"))) * 384) +
                      ((10 * Int(source[30] - UInt8(ascii: "0")) + Int(source[31] - UInt8(ascii: "0"))) - 1) * 32 +
                      10 * Int(source[32] - UInt8(ascii: "0")) + Int(source[33] - UInt8(ascii: "0")))
            } else {
                gv = 38400
            }
            binaryString += intToBits(gv, 16)

        default:
            break
        }
    }

    private func encodeGeneralField(_ lastModeIn: Int) -> Int {
        var lastMode = lastModeIn
        let gf = generalField
        let n = gf.count

        // Classify each character
        var hasInvalid = false
        for i in 0..<n {
            let c = gf[i]
            if c < UInt8(ascii: " ") || c > UInt8(ascii: "z") {
                generalFieldType[i] = encInvalid
                hasInvalid = true
            } else {
                generalFieldType[i] = encISO
            }
            if c == UInt8(ascii: "#") || c == UInt8(ascii: "$") || c == UInt8(ascii: "@") ||
                c == UInt8(ascii: "\\") || c == UInt8(ascii: "^") || c == UInt8(ascii: "`") {
                generalFieldType[i] = encInvalid
                hasInvalid = true
            }
            if (c >= UInt8(ascii: "A") && c <= UInt8(ascii: "Z")) ||
                c == UInt8(ascii: "*") || c == UInt8(ascii: ",") ||
                c == UInt8(ascii: "-") || c == UInt8(ascii: ".") || c == UInt8(ascii: "/") {
                generalFieldType[i] = encAlphaOrISO
            }
            if (c >= UInt8(ascii: "0") && c <= UInt8(ascii: "9")) || c == UInt8(ascii: "[") {
                generalFieldType[i] = encAnyEnc
            }
        }

        if hasInvalid {
            return lastMode
        }

        // Propagate ISOIEC/ALPHA_OR_ISO to following FNC1
        for i in 0..<(n - 1) {
            if generalFieldType[i] == encISO && gf[i + 1] == UInt8(ascii: "[") {
                generalFieldType[i + 1] = encISO
            }
        }
        for i in 0..<(n - 1) {
            if generalFieldType[i] == encAlphaOrISO && gf[i + 1] == UInt8(ascii: "[") {
                generalFieldType[i + 1] = encAlphaOrISO
            }
        }

        // Apply rules
        let latch = applyGeneralFieldRules()

        // Set initial mode
        if generalFieldType[0] == encAlpha {
            binaryString += "0000"
            lastMode = encAlpha
        }
        if generalFieldType[0] == encISO {
            binaryString += "0000"
            binaryString += "00100"
            lastMode = encISO
        }

        // Encode characters
        var i = 0
        while true {
            let mode = generalFieldType[i]

            if mode == encNumeric {
                if lastMode != encNumeric {
                    binaryString += "000"
                }
                let d1 = gf[i] != UInt8(ascii: "[") ? Int(gf[i] - UInt8(ascii: "0")) : 10
                let d2 = gf[i + 1] != UInt8(ascii: "[") ? Int(gf[i + 1] - UInt8(ascii: "0")) : 10
                let value = 11 * d1 + d2 + 8
                binaryString += intToBits(value, 7)
                i += 2
                lastMode = encNumeric

            } else if mode == encAlpha {
                if i != 0 {
                    if lastMode == encNumeric {
                        binaryString += "0000"
                    }
                    if lastMode == encISO {
                        binaryString += "00100"
                    }
                }
                let c = gf[i]
                if c >= UInt8(ascii: "0") && c <= UInt8(ascii: "9") {
                    binaryString += intToBits(Int(c) - 43, 5)
                }
                if c >= UInt8(ascii: "A") && c <= UInt8(ascii: "Z") {
                    binaryString += intToBits(Int(c) - 33, 6)
                }
                lastMode = encAlpha
                if c == UInt8(ascii: "[") {
                    binaryString += "01111"
                    lastMode = encNumeric
                }
                if c == UInt8(ascii: "*") { binaryString += "111010" }
                if c == UInt8(ascii: ",") { binaryString += "111011" }
                if c == UInt8(ascii: "-") { binaryString += "111100" }
                if c == UInt8(ascii: ".") { binaryString += "111101" }
                if c == UInt8(ascii: "/") { binaryString += "111110" }
                i += 1

            } else if mode == encISO {
                if i != 0 {
                    if lastMode == encNumeric {
                        binaryString += "0000"
                        binaryString += "00100"
                    }
                    if lastMode == encAlpha {
                        binaryString += "00100"
                    }
                }
                let c = gf[i]
                if c >= UInt8(ascii: "0") && c <= UInt8(ascii: "9") {
                    binaryString += intToBits(Int(c) - 43, 5)
                }
                if c >= UInt8(ascii: "A") && c <= UInt8(ascii: "Z") {
                    binaryString += intToBits(Int(c) - 1, 7)
                }
                if c >= UInt8(ascii: "a") && c <= UInt8(ascii: "z") {
                    binaryString += intToBits(Int(c) - 7, 7)
                }
                lastMode = encISO
                if c == UInt8(ascii: "[") {
                    binaryString += "01111"
                    lastMode = encNumeric
                }
                if let v = isoChars[c] {
                    binaryString += v
                }
                i += 1
            } else {
                i += 1
            }

            var currentLength = i
            if latch { currentLength += 1 }
            if currentLength >= n { break }
        }

        // Handle single remaining digit (latch case)
        let remainder = calculateRemainder(binaryString.count)
        if latch {
            if lastMode == encNumeric {
                if remainder >= 4 && remainder <= 6 {
                    let value = Int(gf[i] - UInt8(ascii: "0")) + 1
                    binaryString += intToBits(value, 4)
                } else {
                    let d1 = Int(gf[i] - UInt8(ascii: "0"))
                    let d2 = 10
                    let value = 11 * d1 + d2 + 8
                    binaryString += intToBits(value, 7)
                }
            } else {
                let value = Int(gf[i]) - 43
                binaryString += intToBits(value, 5)
            }
        }

        return lastMode
    }

    // MARK: - Encode

    /// Encodes GS1 DataBar Expanded.
    public func encode(_ code: String) throws -> [Int] {
        let content = code.replacingOccurrences(of: "{AI}", with: "")
        if content.isEmpty {
            throw BarcodeError.invalidInput("input data is empty")
        }

        let sourceBytes = Array(content.utf8)

        binaryString = "0" // Linkage flag
        try calculateBinaryString(sourceBytes, code)

        // Data characters from binary string
        let dataChars = binaryString.count / 12

        // Calculate vs values (12-bit segments)
        let bsBytes = Array(binaryString.utf8)
        var vs = [Int](repeating: 0, count: dataChars)
        for i in 0..<dataChars {
            var v = 0
            for j in 0..<12 {
                if bsBytes[i * 12 + j] == UInt8(ascii: "1") {
                    v += (2048 >> j)
                }
            }
            vs[i] = v
        }

        // Determine groups and calculate widths
        var group = [Int](repeating: 0, count: dataChars)
        var charWidths = [[Int]](repeating: [Int](repeating: 0, count: 8), count: dataChars)

        for i in 0..<dataChars {
            if vs[i] <= 347 {
                group[i] = 1
            } else if vs[i] <= 1387 {
                group[i] = 2
            } else if vs[i] <= 2947 {
                group[i] = 3
            } else if vs[i] <= 3987 {
                group[i] = 4
            } else {
                group[i] = 5
            }

            let gi = group[i] - 1
            let vo = (vs[i] - gSumExp[gi]) / tEvenExp[gi]
            let ve = (vs[i] - gSumExp[gi]) % tEvenExp[gi]

            let w = getWidths(vo, modulesOddExp[gi], 4, widestOddExp[gi], 0)
            charWidths[i][0] = w[0]
            charWidths[i][2] = w[1]
            charWidths[i][4] = w[2]
            charWidths[i][6] = w[3]

            let w2 = getWidths(ve, modulesEvenExp[gi], 4, widestEvenExp[gi], 1)
            charWidths[i][1] = w2[0]
            charWidths[i][3] = w2[1]
            charWidths[i][5] = w2[2]
            charWidths[i][7] = w2[3]
        }

        // Checksum (modulo 211)
        var checksum = 0
        for i in 0..<dataChars {
            let row = weightRows[(((dataChars - 2) / 2) * 21) + i]
            for j in 0..<8 {
                checksum += charWidths[i][j] * checksumWeightExp[row * 8 + j]
            }
        }

        let checkChar = 211 * ((dataChars + 1) - 4) + (checksum % 211)

        // Check character widths
        let cg: Int
        if checkChar <= 347 {
            cg = 1
        } else if checkChar <= 1387 {
            cg = 2
        } else if checkChar <= 2947 {
            cg = 3
        } else if checkChar <= 3987 {
            cg = 4
        } else {
            cg = 5
        }

        let cOdd = (checkChar - gSumExp[cg - 1]) / tEvenExp[cg - 1]
        let cEven = (checkChar - gSumExp[cg - 1]) % tEvenExp[cg - 1]

        var checkWidths = [Int](repeating: 0, count: 8)
        let wc = getWidths(cOdd, modulesOddExp[cg - 1], 4, widestOddExp[cg - 1], 0)
        checkWidths[0] = wc[0]; checkWidths[2] = wc[1]
        checkWidths[4] = wc[2]; checkWidths[6] = wc[3]
        let wc2 = getWidths(cEven, modulesEvenExp[cg - 1], 4, widestEvenExp[cg - 1], 1)
        checkWidths[1] = wc2[0]; checkWidths[3] = wc2[1]
        checkWidths[5] = wc2[2]; checkWidths[7] = wc2[3]

        // Build elements array
        let patternWidth = ((((dataChars + 1) / 2) + ((dataChars + 1) & 1)) * 5 +
                            (dataChars + 1) * 8 + 4)
        var elems = [Int](repeating: 0, count: patternWidth)
        elems[0] = 1
        elems[1] = 1
        elems[patternWidth - 2] = 1
        elems[patternWidth - 1] = 1

        // Finder patterns
        let finderCount = (dataChars + 1) / 2 + ((dataChars + 1) & 1)
        for i in 0..<finderCount {
            let k = ((((dataChars + 1) - 2) / 2 + ((dataChars + 1) & 1)) - 1) * 11 + i
            for j in 0..<5 {
                elems[21 * i + j + 10] = finderPatternExp[(finderSequence[k] - 1) * 5 + j]
            }
        }

        // Check character at position 2
        for i in 0..<8 {
            elems[i + 2] = checkWidths[i]
        }

        // Odd-index data characters (forward)
        var idx = 1
        while idx < dataChars {
            for j in 0..<8 {
                elems[((idx - 1) / 2) * 21 + 23 + j] = charWidths[idx][j]
            }
            idx += 2
        }

        // Even-index data characters (reversed)
        idx = 0
        while idx < dataChars {
            for j in 0..<8 {
                elems[(idx / 2) * 21 + 15 + j] = charWidths[idx][7 - j]
            }
            idx += 2
        }

        elements = elems

        // Generate patterns for rendering
        generatePatterns(elems, patternWidth, dataChars)

        return elems
    }

    private func generatePatterns(_ elements: [Int], _ patternWidth: Int, _ dataChars: Int) {
        if symbolType == .unstacked {
            rowCount = 1
            rowHeights = [-1]
            var pattern = "0"
            for i in 0..<patternWidth {
                pattern += String(UnicodeScalar(UInt8(elements[i]) + UInt8(ascii: "0")))
            }
            patterns = [pattern]
        } else {
            generateStacked(elements, patternWidth, dataChars)
        }
    }

    private func generateStacked(_ elements: [Int], _ patternWidth: Int, _ dataChars: Int) {
        let codeblocks = (dataChars + 1) / 2 + ((dataChars + 1) % 2)
        var blocksPerRow = numColumns
        if blocksPerRow < 1 || blocksPerRow > 10 {
            blocksPerRow = 2
        }

        var stackRows = codeblocks / blocksPerRow
        if codeblocks % blocksPerRow > 0 {
            stackRows += 1
        }

        let totalRowCount = stackRows * 4 - 3
        rowHeights = [Int](repeating: 0, count: totalRowCount)
        patterns = [String](repeating: "", count: totalRowCount)
        var symbolRow = 0
        var currentBlock = 0

        for currentRow in 1...stackRows {
            var subElements = [Int](repeating: 0, count: 235)
            var specialCaseRow = false

            subElements[0] = 1
            subElements[1] = 1
            var elementsInSub = 2

            var reader = 0
            var leftToRight = false

            while true {
                if (blocksPerRow & 1) != 0 ||
                    (currentRow & 1) != 0 ||
                    (currentRow == stackRows &&
                     codeblocks != currentRow * blocksPerRow &&
                     (((currentRow * blocksPerRow) - codeblocks) & 1) != 0) {
                    leftToRight = true
                    let baseI = 2 + currentBlock * 21
                    for j in 0..<21 {
                        if baseI + j < patternWidth {
                            subElements[j + reader * 21 + 2] = elements[baseI + j]
                            elementsInSub += 1
                        }
                    }
                } else {
                    leftToRight = false
                    if currentRow * blocksPerRow < codeblocks {
                        let baseI = 2 + ((currentRow * blocksPerRow) - reader - 1) * 21
                        for j in 0..<21 {
                            if baseI + j < patternWidth {
                                subElements[(20 - j) + reader * 21 + 2] = elements[baseI + j]
                                elementsInSub += 1
                            }
                        }
                    } else {
                        let kOff = (currentRow * blocksPerRow) - codeblocks
                        let lOff = (currentRow * blocksPerRow) - reader - 1
                        let baseI = 2 + (lOff - kOff) * 21
                        for j in 0..<21 {
                            if baseI + j < patternWidth {
                                subElements[(20 - j) + reader * 21 + 2] = elements[baseI + j]
                                elementsInSub += 1
                            }
                        }
                    }
                }

                reader += 1
                currentBlock += 1
                if reader >= blocksPerRow || currentBlock >= codeblocks {
                    break
                }
            }

            // Row stop
            subElements[elementsInSub] = 1
            subElements[elementsInSub + 1] = 1
            elementsInSub += 2

            // Build pattern
            rowHeights[symbolRow] = -1
            var black = true

            if (currentRow & 1) != 0 {
                patterns[symbolRow] = "0"
                black = false
            } else {
                if currentRow == stackRows &&
                    codeblocks != currentRow * blocksPerRow &&
                    (((currentRow * blocksPerRow) - codeblocks) & 1) != 0 {
                    specialCaseRow = true
                    subElements[0] = 2
                    patterns[symbolRow] = "0"
                    black = false
                } else {
                    patterns[symbolRow] = ""
                }
            }

            // Build separator binary and pattern string
            var writer = 0
            var sepBin = [UInt8]()
            for i in 0..<elementsInSub {
                patterns[symbolRow] += String(UnicodeScalar(UInt8(subElements[i]) + UInt8(ascii: "0")))
                for _ in 0..<subElements[i] {
                    if black {
                        sepBin.append(UInt8(ascii: "0"))
                    } else {
                        sepBin.append(UInt8(ascii: "1"))
                    }
                }
                black = !black
                writer += subElements[i]
            }

            // Fix separator binary
            if writer > 4 {
                for idx in 0..<min(4, sepBin.count) {
                    sepBin[idx] = UInt8(ascii: "0")
                }
                if sepBin.count > writer {
                    sepBin = Array(sepBin[0..<writer])
                }
            }

            for j in 0..<reader {
                var k = 49 * j + 18
                if specialCaseRow {
                    k = 49 * j + 19
                }
                if leftToRight {
                    for ii in 0..<15 {
                        let idx2 = ii + k
                        if idx2 - 1 >= 0 && idx2 < sepBin.count &&
                            sepBin[idx2 - 1] == UInt8(ascii: "1") && sepBin[idx2] == UInt8(ascii: "1") {
                            sepBin[idx2] = UInt8(ascii: "0")
                        }
                    }
                } else {
                    for ii in stride(from: 14, through: 0, by: -1) {
                        let idx2 = ii + k
                        if idx2 >= 0 && idx2 + 1 < sepBin.count &&
                            sepBin[idx2 + 1] == UInt8(ascii: "1") && sepBin[idx2] == UInt8(ascii: "1") {
                            sepBin[idx2] = UInt8(ascii: "0")
                        }
                    }
                }
            }

            let sepBinStr = String(bytes: sepBin, encoding: .ascii) ?? ""
            let sepPattern = bin2patFromBlack(sepBinStr)

            // Separator rows
            if currentRow != 1 {
                var midSep = "05"
                var cnt = 5
                while cnt < 49 * blocksPerRow {
                    midSep += "11"
                    cnt += 2
                }
                patterns[symbolRow - 2] = midSep
                rowHeights[symbolRow - 2] = 1
                patterns[symbolRow - 1] = sepPattern
                rowHeights[symbolRow - 1] = 1
            }

            if currentRow != stackRows {
                patterns[symbolRow + 1] = sepPattern
                rowHeights[symbolRow + 1] = 1
            }

            symbolRow += 4
        }

        rowCount = totalRowCount
    }

    /// Draws the GS1 DataBar Expanded barcode.
    public func draw(code: String, width: Int, height: Int) throws {
        let _ = try encode(code)
        if isSVGOutput() {
            try drawMultiRowSVG(self, patterns, rowHeights, width, height)
        } else {
            try drawMultiRowPNG(self, patterns, rowHeights, width, height)
        }
    }
}
