import Foundation

/// SymbolType14 specifies the GS1 DataBar 14 variant.
public enum SymbolType14: Int {
    case omnidirectional       = 0
    case stacked               = 1
    case stackedOmnidirectional = 2
}

// ISO 24724 Table 3, 4
private let gSumTable14 = [0, 161, 961, 2015, 2715, 0, 336, 1036, 1516]
private let tTable14 = [1, 10, 34, 70, 126, 4, 20, 48, 81]
private let modulesOdd14 = [12, 10, 8, 6, 4, 5, 7, 9, 11]
private let modulesEven14 = [4, 6, 8, 10, 12, 10, 8, 6, 4]
private let widestOdd14 = [8, 6, 4, 3, 1, 2, 4, 6, 8]
private let widestEven14 = [1, 3, 5, 6, 8, 7, 5, 3, 1]

private let checksumWeight14 = [
    1, 3, 9, 27, 2, 6, 18, 54,
    4, 12, 36, 29, 8, 24, 72, 58,
    16, 48, 65, 37, 32, 17, 51, 74,
    64, 34, 23, 69, 49, 68, 46, 59,
]

private let finderPattern14 = [
    3, 8, 2, 1, 1,
    3, 5, 5, 1, 1,
    3, 3, 7, 1, 1,
    3, 1, 9, 1, 1,
    2, 7, 4, 1, 1,
    2, 5, 6, 1, 1,
    2, 3, 8, 1, 1,
    1, 5, 7, 1, 1,
    1, 3, 9, 1, 1,
]

/// GS1 DataBar 14 (RSS-14) barcode encoder.
public class GS1DataBar14: BarcodeBase1D {
    public var symbolType: SymbolType14
    private(set) public var gtin14: String = ""
    private var totalWidths: [Int] = []
    private(set) public var patterns: [String] = []
    private(set) public var rowHeights: [Int] = []
    private(set) public var rowCount: Int = 0

    public init(outputFormat: String, symbolType: SymbolType14 = .omnidirectional) {
        self.symbolType = symbolType
        super.init(outputFormat: outputFormat)
        self.showText = false
    }

    /// Returns the human-readable text "(01)GTIN-14".
    public func humanReadable() -> String { "(01)" + gtin14 }

    /// Encodes GS1 DataBar 14.
    public func encode(_ code: String) throws -> [Int] {
        for c in code.unicodeScalars {
            if c < "0" || c > "9" {
                throw BarcodeError.invalidInput("GS1 DataBar requires numeric digits only")
            }
        }
        if code.count > 13 {
            throw BarcodeError.invalidInput("input too long (max 13 digits)")
        }

        // Left-pad to 13 digits
        let padded = padLeft(code, 13)

        // Convert to numeric value
        let accum = parseInt64(padded)

        // Split into left_reg and right_reg
        let leftReg = Int(accum / 4537077)
        let rightReg = Int(accum % 4537077)

        // 4 data characters
        var dataChars = [Int](repeating: 0, count: 4)
        dataChars[0] = leftReg / 1597
        dataChars[1] = leftReg % 1597
        dataChars[2] = rightReg / 1597
        dataChars[3] = rightReg % 1597

        // Data groups
        var dataGroup = [Int](repeating: 0, count: 4)
        for idx in [0, 2] {
            dataGroup[idx] = 0
            if dataChars[idx] >= 161 { dataGroup[idx] = 1 }
            if dataChars[idx] >= 961 { dataGroup[idx] = 2 }
            if dataChars[idx] >= 2015 { dataGroup[idx] = 3 }
            if dataChars[idx] >= 2715 { dataGroup[idx] = 4 }
        }
        for idx in [1, 3] {
            dataGroup[idx] = 5
            if dataChars[idx] >= 336 { dataGroup[idx] = 6 }
            if dataChars[idx] >= 1036 { dataGroup[idx] = 7 }
            if dataChars[idx] >= 1516 { dataGroup[idx] = 8 }
        }

        // Calculate odd/even values
        var vOdd = [Int](repeating: 0, count: 4)
        var vEven = [Int](repeating: 0, count: 4)
        vOdd[0] = (dataChars[0] - gSumTable14[dataGroup[0]]) / tTable14[dataGroup[0]]
        vEven[0] = (dataChars[0] - gSumTable14[dataGroup[0]]) % tTable14[dataGroup[0]]
        vOdd[1] = (dataChars[1] - gSumTable14[dataGroup[1]]) % tTable14[dataGroup[1]]
        vEven[1] = (dataChars[1] - gSumTable14[dataGroup[1]]) / tTable14[dataGroup[1]]
        vOdd[3] = (dataChars[3] - gSumTable14[dataGroup[3]]) % tTable14[dataGroup[3]]
        vEven[3] = (dataChars[3] - gSumTable14[dataGroup[3]]) / tTable14[dataGroup[3]]
        vOdd[2] = (dataChars[2] - gSumTable14[dataGroup[2]]) / tTable14[dataGroup[2]]
        vEven[2] = (dataChars[2] - gSumTable14[dataGroup[2]]) % tTable14[dataGroup[2]]

        // Width patterns
        var dataWidths = [[Int]](repeating: [Int](repeating: 0, count: 4), count: 8)
        for i in 0..<4 {
            if i == 0 || i == 2 {
                let w = getWidths(vOdd[i], modulesOdd14[dataGroup[i]], 4, widestOdd14[dataGroup[i]], 1)
                dataWidths[0][i] = w[0]
                dataWidths[2][i] = w[1]
                dataWidths[4][i] = w[2]
                dataWidths[6][i] = w[3]
                let w2 = getWidths(vEven[i], modulesEven14[dataGroup[i]], 4, widestEven14[dataGroup[i]], 0)
                dataWidths[1][i] = w2[0]
                dataWidths[3][i] = w2[1]
                dataWidths[5][i] = w2[2]
                dataWidths[7][i] = w2[3]
            } else {
                let w = getWidths(vOdd[i], modulesOdd14[dataGroup[i]], 4, widestOdd14[dataGroup[i]], 0)
                dataWidths[0][i] = w[0]
                dataWidths[2][i] = w[1]
                dataWidths[4][i] = w[2]
                dataWidths[6][i] = w[3]
                let w2 = getWidths(vEven[i], modulesEven14[dataGroup[i]], 4, widestEven14[dataGroup[i]], 1)
                dataWidths[1][i] = w2[0]
                dataWidths[3][i] = w2[1]
                dataWidths[5][i] = w2[2]
                dataWidths[7][i] = w2[3]
            }
        }

        // Checksum (modulo 79)
        var checksum = 0
        for i in 0..<8 {
            checksum += checksumWeight14[i] * dataWidths[i][0]
            checksum += checksumWeight14[i + 8] * dataWidths[i][1]
            checksum += checksumWeight14[i + 16] * dataWidths[i][2]
            checksum += checksumWeight14[i + 24] * dataWidths[i][3]
        }
        checksum %= 79

        // Finder pattern selection
        if checksum >= 8 { checksum += 1 }
        if checksum >= 72 { checksum += 1 }
        let cLeft = checksum / 9
        let cRight = checksum % 9

        // Build total widths (46 elements)
        var tw = [Int](repeating: 0, count: 46)
        tw[0] = 1
        tw[1] = 1
        tw[44] = 1
        tw[45] = 1

        for i in 0..<8 {
            tw[i + 2] = dataWidths[i][0]
            tw[i + 15] = dataWidths[7 - i][1]
            tw[i + 23] = dataWidths[i][3]
            tw[i + 36] = dataWidths[7 - i][2]
        }

        for i in 0..<5 {
            tw[i + 10] = finderPattern14[i + 5 * cLeft]
            tw[i + 31] = finderPattern14[(4 - i) + 5 * cRight]
        }

        // GTIN-14 check digit
        let cd = calculateGTINCheckDigit(padded)
        let cdChar = Character(UnicodeScalar(UInt8(ascii: "0") + UInt8(cd)))
        gtin14 = padded + String(cdChar)
        totalWidths = tw

        // Generate grid and patterns
        generateGrid(tw)

        return tw
    }

    private func generateGrid(_ totalWidths: [Int]) {
        var grid = [[Bool]](repeating: [Bool](repeating: false, count: 100), count: 5)
        var currentRowCount = 0
        var symbolWidth = 0

        switch symbolType {
        case .omnidirectional:
            var writer = 0
            var latch = false
            for i in 0..<46 {
                for _ in 0..<totalWidths[i] {
                    if latch {
                        grid[currentRowCount][writer] = true
                    }
                    writer += 1
                }
                latch = !latch
            }
            if writer > symbolWidth { symbolWidth = writer }
            currentRowCount += 1

        case .stacked:
            // Upper row (row 0)
            var writer = 0
            var latch = false
            for i in 0..<23 {
                for _ in 0..<totalWidths[i] {
                    grid[currentRowCount][writer] = latch
                    writer += 1
                }
                latch = !latch
            }
            grid[currentRowCount][writer] = true
            grid[currentRowCount][writer + 1] = false

            // Lower row (row 2)
            currentRowCount += 2
            grid[currentRowCount][0] = true
            grid[currentRowCount][1] = false
            writer = 0
            latch = true
            for i in 23..<46 {
                for _ in 0..<totalWidths[i] {
                    grid[currentRowCount][writer + 2] = latch
                    writer += 1
                }
                latch = !latch
            }

            // Separator (row 1)
            for i in 4..<46 {
                if grid[currentRowCount - 2][i] == grid[currentRowCount][i] {
                    if !grid[currentRowCount - 2][i] {
                        grid[currentRowCount - 1][i] = true
                    }
                } else {
                    if !grid[currentRowCount - 1][i - 1] {
                        grid[currentRowCount - 1][i] = true
                    }
                }
            }

            currentRowCount += 1
            if 50 > symbolWidth { symbolWidth = 50 }

        case .stackedOmnidirectional:
            // Upper row (row 0)
            var writer = 0
            var latch = false
            for i in 0..<23 {
                for _ in 0..<totalWidths[i] {
                    grid[currentRowCount][writer] = latch
                    writer += 1
                }
                latch = !latch
            }
            grid[currentRowCount][writer] = true
            grid[currentRowCount][writer + 1] = false

            // Lower row (row 4)
            currentRowCount += 4
            grid[currentRowCount][0] = true
            grid[currentRowCount][1] = false
            writer = 0
            latch = true
            for i in 23..<46 {
                for _ in 0..<totalWidths[i] {
                    grid[currentRowCount][writer + 2] = latch
                    writer += 1
                }
                latch = !latch
            }

            // Middle separator (row 2)
            var i = 5
            while i < 46 {
                grid[currentRowCount - 2][i] = true
                i += 2
            }

            // Upper separator (row 1)
            for i in 4..<46 {
                if !grid[currentRowCount - 4][i] {
                    grid[currentRowCount - 3][i] = true
                }
            }
            var latchVal = true
            for i in 17..<33 {
                if !grid[currentRowCount - 4][i] {
                    if latchVal {
                        grid[currentRowCount - 3][i] = true
                        latchVal = false
                    } else {
                        grid[currentRowCount - 3][i] = false
                        latchVal = true
                    }
                } else {
                    grid[currentRowCount - 3][i] = false
                    latchVal = true
                }
            }

            // Lower separator (row 3)
            for i in 4..<46 {
                if !grid[currentRowCount][i] {
                    grid[currentRowCount - 1][i] = true
                }
            }
            latchVal = true
            for i in 16..<32 {
                if !grid[currentRowCount][i] {
                    if latchVal {
                        grid[currentRowCount - 1][i] = true
                        latchVal = false
                    } else {
                        grid[currentRowCount - 1][i] = false
                        latchVal = true
                    }
                } else {
                    grid[currentRowCount - 1][i] = false
                    latchVal = true
                }
            }

            if 50 > symbolWidth { symbolWidth = 50 }
            currentRowCount += 1
        }

        // Convert grid to pattern strings
        patterns = []
        rowHeights = []

        for i in 0..<currentRowCount {
            var binary = ""
            for j in 0..<symbolWidth {
                binary += grid[i][j] ? "1" : "0"
            }

            var pattern = bin2pat(binary)
            if !binary.isEmpty && binary.first == "0" {
                pattern = "0" + pattern
            }

            patterns.append(pattern)
        }

        // Row heights
        switch symbolType {
        case .omnidirectional:
            rowHeights = [-1]
        case .stacked:
            rowHeights = [5, 1, 7]
        case .stackedOmnidirectional:
            rowHeights = [-1, 1, 1, 1, -1]
        }

        rowCount = currentRowCount
    }

    /// Draws the GS1 DataBar 14 barcode.
    public func draw(code: String, width: Int, height: Int) throws {
        let _ = try encode(code)
        if isSVGOutput() {
            try drawMultiRowSVG(self, patterns, rowHeights, width, height)
        } else {
            try drawMultiRowPNG(self, patterns, rowHeights, width, height)
        }
    }
}
