import Foundation

/// Minimal TrueType font parser for rendering barcode text.
/// Parses cmap, head, hhea, hmtx, loca, glyf tables to extract glyph outlines.
class TrueTypeFont {
    let data: [UInt8]
    private var tables: [String: (offset: Int, length: Int)] = [:]

    // head table
    private var unitsPerEm: Int = 0
    private var indexToLocFormat: Int = 0

    // hhea table
    private var numHMetrics: Int = 0

    // maxp table
    private var numGlyphs: Int = 0

    // Table offsets
    private var cmapOffset: Int = 0
    private var hmtxOffset: Int = 0
    private var locaOffset: Int = 0
    private var glyfOffset: Int = 0

    // cmap format 4 data
    private var cmapSegCount: Int = 0
    private var cmapEndCodes: [UInt16] = []
    private var cmapStartCodes: [UInt16] = []
    private var cmapIdDeltas: [Int16] = []
    private var cmapIdRangeOffsets: [UInt16] = []
    private var cmapIdRangeBase: Int = 0

    init?(_ fontData: [UInt8]) {
        self.data = fontData
        guard fontData.count > 12 else { return nil }
        if !parseTables() { return nil }
        if !parseHead() { return nil }
        if !parseHhea() { return nil }
        if !parseMaxp() { return nil }
        if !parseCmap() { return nil }

        guard let hmtx = tables["hmtx"] else { return nil }
        hmtxOffset = hmtx.offset
        guard let loca = tables["loca"] else { return nil }
        locaOffset = loca.offset
        guard let glyf = tables["glyf"] else { return nil }
        glyfOffset = glyf.offset
    }

    private func u16(_ offset: Int) -> UInt16 {
        guard offset + 1 < data.count else { return 0 }
        return UInt16(data[offset]) << 8 | UInt16(data[offset + 1])
    }

    private func i16(_ offset: Int) -> Int16 {
        Int16(bitPattern: u16(offset))
    }

    private func u32(_ offset: Int) -> UInt32 {
        guard offset + 3 < data.count else { return 0 }
        return UInt32(data[offset]) << 24 | UInt32(data[offset+1]) << 16 |
               UInt32(data[offset+2]) << 8 | UInt32(data[offset+3])
    }

    private func tag(_ offset: Int) -> String {
        guard offset + 3 < data.count else { return "" }
        return String(bytes: data[offset..<offset+4], encoding: .ascii) ?? ""
    }

    // MARK: - Table Directory

    private func parseTables() -> Bool {
        let numTables = Int(u16(4))
        for i in 0..<numTables {
            let entry = 12 + i * 16
            let name = tag(entry)
            let offset = Int(u32(entry + 8))
            let length = Int(u32(entry + 12))
            tables[name] = (offset, length)
        }
        return tables["head"] != nil && tables["cmap"] != nil
    }

    private func parseHead() -> Bool {
        guard let head = tables["head"] else { return false }
        let off = head.offset
        unitsPerEm = Int(u16(off + 18))
        indexToLocFormat = Int(i16(off + 50))
        return unitsPerEm > 0
    }

    private func parseHhea() -> Bool {
        guard let hhea = tables["hhea"] else { return false }
        numHMetrics = Int(u16(hhea.offset + 34))
        return true
    }

    private func parseMaxp() -> Bool {
        guard let maxp = tables["maxp"] else { return false }
        numGlyphs = Int(u16(maxp.offset + 4))
        return true
    }

    // MARK: - cmap

    private func parseCmap() -> Bool {
        guard let cmap = tables["cmap"] else { return false }
        let base = cmap.offset
        let numSubtables = Int(u16(base + 2))

        // Find format 4 subtable (platform 3, encoding 1 = Windows Unicode BMP)
        var subtableOffset = 0
        for i in 0..<numSubtables {
            let entry = base + 4 + i * 8
            let platformId = u16(entry)
            let encodingId = u16(entry + 2)
            if platformId == 3 && encodingId == 1 {
                subtableOffset = base + Int(u32(entry + 4))
                break
            }
            // Also accept platform 0 (Unicode)
            if platformId == 0 {
                subtableOffset = base + Int(u32(entry + 4))
            }
        }

        guard subtableOffset > 0 else { return false }
        let format = u16(subtableOffset)
        guard format == 4 else { return false }

        cmapSegCount = Int(u16(subtableOffset + 6)) / 2
        let endCodesOff = subtableOffset + 14
        let startCodesOff = endCodesOff + cmapSegCount * 2 + 2
        let idDeltaOff = startCodesOff + cmapSegCount * 2
        let idRangeOff = idDeltaOff + cmapSegCount * 2

        cmapEndCodes = (0..<cmapSegCount).map { u16(endCodesOff + $0 * 2) }
        cmapStartCodes = (0..<cmapSegCount).map { u16(startCodesOff + $0 * 2) }
        cmapIdDeltas = (0..<cmapSegCount).map { i16(idDeltaOff + $0 * 2) }
        cmapIdRangeOffsets = (0..<cmapSegCount).map { u16(idRangeOff + $0 * 2) }
        cmapIdRangeBase = idRangeOff

        cmapOffset = subtableOffset
        return true
    }

    // MARK: - Glyph Lookup

    /// Maps a Unicode code point to a glyph index.
    func glyphIndex(for codePoint: UInt16) -> Int {
        for i in 0..<cmapSegCount {
            if codePoint <= cmapEndCodes[i] && codePoint >= cmapStartCodes[i] {
                if cmapIdRangeOffsets[i] == 0 {
                    return Int(UInt16(truncatingIfNeeded: Int(codePoint) + Int(cmapIdDeltas[i])))
                } else {
                    let rangeOffset = Int(cmapIdRangeOffsets[i])
                    let glyphOff = cmapIdRangeBase + i * 2 + rangeOffset + Int(codePoint - cmapStartCodes[i]) * 2
                    let idx = u16(glyphOff)
                    if idx == 0 { return 0 }
                    return Int(UInt16(truncatingIfNeeded: Int(idx) + Int(cmapIdDeltas[i])))
                }
            }
        }
        return 0
    }

    /// Returns the advance width for a glyph in font units.
    func advanceWidth(glyphIndex gi: Int) -> Int {
        if gi < numHMetrics {
            return Int(u16(hmtxOffset + gi * 4))
        } else if numHMetrics > 0 {
            return Int(u16(hmtxOffset + (numHMetrics - 1) * 4))
        }
        return 0
    }

    /// Returns the glyph outline offset and length from loca/glyf.
    private func glyphLocation(_ gi: Int) -> (offset: Int, length: Int)? {
        guard gi < numGlyphs else { return nil }
        let off: Int
        let nextOff: Int
        if indexToLocFormat == 0 {
            off = Int(u16(locaOffset + gi * 2)) * 2
            nextOff = Int(u16(locaOffset + (gi + 1) * 2)) * 2
        } else {
            off = Int(u32(locaOffset + gi * 4))
            nextOff = Int(u32(locaOffset + (gi + 1) * 4))
        }
        let length = nextOff - off
        if length <= 0 { return nil }
        return (glyfOffset + off, length)
    }

    // MARK: - Glyph Rasterization

    /// Rasterizes a glyph at the given pixel size. Returns (bitmap, width, height, bearingX, bearingY).
    func rasterizeGlyph(glyphIndex gi: Int, pixelSize: Double)
        -> (bitmap: [UInt8], width: Int, height: Int, bearingX: Int, bearingY: Int)?
    {
        guard let loc = glyphLocation(gi) else { return nil }
        let scale = pixelSize / Double(unitsPerEm)

        let numContours = Int(i16(loc.offset))
        guard numContours > 0 else { return nil } // Skip composite glyphs for simplicity

        let xMin = Double(i16(loc.offset + 2)) * scale
        let yMin = Double(i16(loc.offset + 4)) * scale
        let xMax = Double(i16(loc.offset + 6)) * scale
        let yMax = Double(i16(loc.offset + 8)) * scale

        let w = Int(ceil(xMax - xMin)) + 2
        let h = Int(ceil(yMax - yMin)) + 2
        if w <= 0 || h <= 0 || w > 500 || h > 500 { return nil }

        // Parse contour endpoints
        var endPtsOfContours = [Int]()
        for i in 0..<numContours {
            endPtsOfContours.append(Int(u16(loc.offset + 10 + i * 2)))
        }
        let numPoints = (endPtsOfContours.last ?? -1) + 1
        guard numPoints > 0 else { return nil }

        // Skip instructions
        let instrLength = Int(u16(loc.offset + 10 + numContours * 2))
        var flagOffset = loc.offset + 10 + numContours * 2 + 2 + instrLength

        // Parse flags
        var flags = [UInt8]()
        flags.reserveCapacity(numPoints)
        var idx = 0
        while idx < numPoints && flagOffset < data.count {
            let flag = data[flagOffset]
            flagOffset += 1
            flags.append(flag)
            idx += 1
            if (flag & 0x08) != 0 && flagOffset < data.count {
                let repeatCount = Int(data[flagOffset])
                flagOffset += 1
                for _ in 0..<repeatCount {
                    if idx < numPoints {
                        flags.append(flag)
                        idx += 1
                    }
                }
            }
        }

        // Parse X coordinates
        var xCoords = [Double]()
        xCoords.reserveCapacity(numPoints)
        var xCur = 0
        for i in 0..<numPoints {
            guard i < flags.count else { break }
            let flag = flags[i]
            if (flag & 0x02) != 0 { // x-Short
                guard flagOffset < data.count else { break }
                let val = Int(data[flagOffset])
                flagOffset += 1
                xCur += (flag & 0x10) != 0 ? val : -val
            } else if (flag & 0x10) == 0 {
                xCur += Int(i16(flagOffset))
                flagOffset += 2
            }
            xCoords.append(Double(xCur) * scale - xMin)
        }

        // Parse Y coordinates
        var yCoords = [Double]()
        yCoords.reserveCapacity(numPoints)
        var yCur = 0
        for i in 0..<numPoints {
            guard i < flags.count else { break }
            let flag = flags[i]
            if (flag & 0x04) != 0 { // y-Short
                guard flagOffset < data.count else { break }
                let val = Int(data[flagOffset])
                flagOffset += 1
                yCur += (flag & 0x20) != 0 ? val : -val
            } else if (flag & 0x20) == 0 {
                yCur += Int(i16(flagOffset))
                flagOffset += 2
            }
            yCoords.append(yMax - Double(yCur) * scale) // Flip Y
        }

        // Rasterize using scanline fill
        var bitmap = [UInt8](repeating: 0, count: w * h)

        var contourStart = 0
        for c in 0..<numContours {
            let contourEnd = endPtsOfContours[c]
            // Build edges from contour points
            var points = [(x: Double, y: Double)]()
            for i in contourStart...contourEnd {
                if i < xCoords.count && i < yCoords.count {
                    points.append((xCoords[i], yCoords[i]))
                }
            }

            if points.count >= 2 {
                // Expand quadratic bezier curves to line segments
                var expanded = [(x: Double, y: Double)]()
                var pi = 0
                while pi < points.count {
                    let curr = points[pi]
                    let nextIdx = (pi + 1) % points.count
                    let isOnCurve = pi < flags.count - contourStart ?
                        (flags[contourStart + pi] & 0x01) != 0 : true

                    if isOnCurve {
                        expanded.append(curr)
                    }
                    pi += 1
                }
                if expanded.isEmpty { expanded = points }

                // Scanline fill
                for i in 0..<expanded.count {
                    let p0 = expanded[i]
                    let p1 = expanded[(i + 1) % expanded.count]
                    rasterizeLine(&bitmap, w: w, h: h, x0: p0.x, y0: p0.y, x1: p1.x, y1: p1.y)
                }
            }

            contourStart = contourEnd + 1
        }

        // Scanline fill using winding number
        fillBitmapScanline(&bitmap, w: w, h: h)

        let bearingX = Int(round(xMin))
        let bearingY = Int(round(yMax))

        return (bitmap, w, h, bearingX, bearingY)
    }

    private func rasterizeLine(_ bitmap: inout [UInt8], w: Int, h: Int,
                                x0: Double, y0: Double, x1: Double, y1: Double) {
        let dy = y1 - y0
        if abs(dy) < 0.001 { return }

        let minY = max(0, Int(floor(min(y0, y1))))
        let maxY = min(h - 1, Int(ceil(max(y0, y1))))

        for y in minY...maxY {
            let fy = Double(y) + 0.5
            if (fy < min(y0, y1)) || (fy >= max(y0, y1)) { continue }
            let t = (fy - y0) / dy
            let x = x0 + t * (x1 - x0)
            let ix = max(0, min(w - 1, Int(round(x))))
            let idx = y * w + ix
            if idx >= 0 && idx < bitmap.count {
                // Mark edge crossing (toggle)
                bitmap[idx] = bitmap[idx] == 0 ? 1 : 0
            }
        }
    }

    private func fillBitmapScanline(_ bitmap: inout [UInt8], w: Int, h: Int) {
        for y in 0..<h {
            var inside = false
            for x in 0..<w {
                let idx = y * w + x
                if bitmap[idx] != 0 {
                    inside = !inside
                }
                bitmap[idx] = inside ? 255 : 0
            }
        }
    }

    /// Returns the ascent in pixels for the given pixel size.
    func ascent(pixelSize: Double) -> Int {
        guard let hhea = tables["hhea"] else { return Int(pixelSize * 0.8) }
        let asc = Int(i16(hhea.offset + 4))
        return Int(round(Double(asc) * pixelSize / Double(unitsPerEm)))
    }

    /// Returns the descent in pixels (negative) for the given pixel size.
    func descent(pixelSize: Double) -> Int {
        guard let hhea = tables["hhea"] else { return Int(-pixelSize * 0.2) }
        let desc = Int(i16(hhea.offset + 6))
        return Int(round(Double(desc) * pixelSize / Double(unitsPerEm)))
    }
}

// MARK: - Font Data Access

/// Returns the embedded Roboto-Regular.ttf bytes.
func getRobotoFontData() -> [UInt8]? {
    #if canImport(Foundation)
    guard let url = Bundle.module.url(forResource: "Roboto-Regular", withExtension: "ttf") else {
        return nil
    }
    guard let data = try? Data(contentsOf: url) else {
        return nil
    }
    return [UInt8](data)
    #else
    return nil
    #endif
}

/// Cached font instance.
private var _cachedFont: TrueTypeFont?
private var _fontLoaded = false

func getDefaultFont() -> TrueTypeFont? {
    if _fontLoaded { return _cachedFont }
    _fontLoaded = true
    if let data = getRobotoFontData() {
        _cachedFont = TrueTypeFont(data)
    }
    return _cachedFont
}
