import Foundation

// MARK: - DataMatrix Code Size Constants

public let DX_SZ_RECT_AUTO  = -3
public let DX_SZ_AUTO       = -2
public let DX_SZ_SHAPE_AUTO = -1
public let DX_SZ_10x10      = 0
public let DX_SZ_12x12      = 1
public let DX_SZ_14x14      = 2
public let DX_SZ_16x16      = 3
public let DX_SZ_18x18      = 4
public let DX_SZ_20x20      = 5
public let DX_SZ_22x22      = 6
public let DX_SZ_24x24      = 7
public let DX_SZ_26x26      = 8
public let DX_SZ_32x32      = 9
public let DX_SZ_36x36      = 10
public let DX_SZ_40x40      = 11
public let DX_SZ_44x44      = 12
public let DX_SZ_48x48      = 13
public let DX_SZ_52x52      = 14
public let DX_SZ_64x64      = 15
public let DX_SZ_72x72      = 16
public let DX_SZ_80x80      = 17
public let DX_SZ_88x88      = 18
public let DX_SZ_96x96      = 19
public let DX_SZ_104x104    = 20
public let DX_SZ_120x120    = 21
public let DX_SZ_132x132    = 22
public let DX_SZ_144x144    = 23
public let DX_SZ_8x18       = 24
public let DX_SZ_8x32       = 25
public let DX_SZ_12x26      = 26
public let DX_SZ_12x36      = 27
public let DX_SZ_16x36      = 28
public let DX_SZ_16x48      = 29

// MARK: - DataMatrix Encoding Scheme Constants

public let DX_SCHEME_AUTO_FAST = -2
public let DX_SCHEME_AUTO_BEST = -1
public let DX_SCHEME_ASCII     = 0
public let DX_SCHEME_C40       = 1
public let DX_SCHEME_TEXT      = 2
public let DX_SCHEME_X12       = 3
public let DX_SCHEME_EDIFACT   = 4
public let DX_SCHEME_BASE256   = 5
public let DX_SCHEME_ASCII_GS1 = 6

// MARK: - Internal Constants

private let dxAttrSymbolRows          = 0
private let dxAttrSymbolCols          = 1
private let dxAttrDataRegionRows      = 2
private let dxAttrDataRegionCols      = 3
private let dxAttrHorizDataRegions    = 4
private let dxAttrVertDataRegions     = 5
private let dxAttrMappingMatrixRows   = 6
private let dxAttrMappingMatrixCols   = 7
private let dxAttrInterleavedBlocks   = 8
private let dxAttrBlockErrorWords     = 9
private let dxAttrBlockMaxCorrectable = 10
private let dxAttrSymbolDataWords     = 11
private let dxAttrSymbolErrorWords    = 12
private let dxAttrSymbolMaxCorrectable = 13

private let dxMaskBit1 = 0x80
private let dxMaskBit2 = 0x40
private let dxMaskBit3 = 0x20
private let dxMaskBit4 = 0x10
private let dxMaskBit5 = 0x08
private let dxMaskBit6 = 0x04
private let dxMaskBit7 = 0x02
private let dxMaskBit8 = 0x01

private let dxChannelValid           = 0x00
private let dxChannelUnsupportedChar = 0x01
private let dxChannelCannotUnlatch   = 0x02

private let dxUnlatchExplicit = 0
private let dxUnlatchImplicit = 1

private let dxModuleOff      = 0x00
private let dxModuleOnRed    = 0x01
private let dxModuleOnGreen  = 0x02
private let dxModuleOnBlue   = 0x04
private let dxModuleOnRGB    = 0x07
private let dxModuleAssigned = 0x10
private let dxModuleVisited  = 0x20
private let dxModuleData     = 0x40

private let dxCharAsciiPad        = 129
private let dxCharAsciiUpperShift = 235
private let dxCharTripletShift1   = 0
private let dxCharTripletShift2   = 1
private let dxCharTripletShift3   = 2
private let dxCharFNC1            = 232
private let dxCharTripletUnlatch  = 254
private let dxCharEdifactUnlatch  = 31
private let dxCharC40Latch        = 230
private let dxCharTextLatch       = 239
private let dxCharX12Latch        = 238
private let dxCharEdifactLatch    = 240
private let dxCharBase256Latch    = 231

private let dxSzSquareCount = 24
private let dxSzRectCount   = 6
private let dxUndefined     = -1

// MARK: - Symbol Attribute Tables (30 entries: 24 square + 6 rectangular)

private let dxSymbolRows: [Int] = [
    10, 12, 14, 16, 18, 20, 22, 24, 26,
    32, 36, 40, 44, 48, 52,
    64, 72, 80, 88, 96, 104,
    120, 132, 144,
    8, 8, 12, 12, 16, 16,
]

private let dxSymbolCols: [Int] = [
    10, 12, 14, 16, 18, 20, 22, 24, 26,
    32, 36, 40, 44, 48, 52,
    64, 72, 80, 88, 96, 104,
    120, 132, 144,
    18, 32, 26, 36, 36, 48,
]

private let dxDataRegionRows: [Int] = [
    8, 10, 12, 14, 16, 18, 20, 22, 24,
    14, 16, 18, 20, 22, 24,
    14, 16, 18, 20, 22, 24,
    18, 20, 22,
    6, 6, 10, 10, 14, 14,
]

private let dxDataRegionCols: [Int] = [
    8, 10, 12, 14, 16, 18, 20, 22, 24,
    14, 16, 18, 20, 22, 24,
    14, 16, 18, 20, 22, 24,
    18, 20, 22,
    16, 14, 24, 16, 16, 22,
]

private let dxHorizDataRegions: [Int] = [
    1, 1, 1, 1, 1, 1, 1, 1, 1,
    2, 2, 2, 2, 2, 2,
    4, 4, 4, 4, 4, 4,
    6, 6, 6,
    1, 2, 1, 2, 2, 2,
]

private let dxInterleavedBlocks: [Int] = [
    1, 1, 1, 1, 1, 1, 1, 1, 1,
    1, 1, 1, 1, 1, 2,
    2, 4, 4, 4, 4, 6,
    6, 8, 10,
    1, 1, 1, 1, 1, 1,
]

private let dxSymbolDataWords: [Int] = [
    3, 5, 8, 12, 18, 22, 30, 36, 44,
    62, 86, 114, 144, 174, 204,
    280, 368, 456, 576, 696, 816,
    1050, 1304, 1558,
    5, 10, 16, 22, 32, 49,
]

private let dxBlockErrorWords: [Int] = [
    5, 7, 10, 12, 14, 18, 20, 24, 28,
    36, 42, 48, 56, 68, 42,
    56, 36, 48, 56, 68, 56,
    68, 62, 62,
    7, 11, 14, 18, 24, 28,
]

private let dxBlockMaxCorrectable: [Int] = [
    2, 3, 5, 6, 7, 9, 10, 12, 14,
    18, 21, 24, 28, 34, 21,
    28, 18, 24, 28, 34, 28,
    34, 31, 31,
    3, 5, 7, 9, 12, 14,
]

// MARK: - GF(256) Tables

private let dxALogVal: [Int] = [
    1, 2, 4, 8, 16, 32, 64, 128, 45, 90, 180, 69, 138, 57, 114, 228,
    229, 231, 227, 235, 251, 219, 155, 27, 54, 108, 216, 157, 23, 46, 92, 184,
    93, 186, 89, 178, 73, 146, 9, 18, 36, 72, 144, 13, 26, 52, 104, 208,
    141, 55, 110, 220, 149, 7, 14, 28, 56, 112, 224, 237, 247, 195, 171, 123,
    246, 193, 175, 115, 230, 225, 239, 243, 203, 187, 91, 182, 65, 130, 41, 82,
    164, 101, 202, 185, 95, 190, 81, 162, 105, 210, 137, 63, 126, 252, 213, 135,
    35, 70, 140, 53, 106, 212, 133, 39, 78, 156, 21, 42, 84, 168, 125, 250,
    217, 159, 19, 38, 76, 152, 29, 58, 116, 232, 253, 215, 131, 43, 86, 172,
    117, 234, 249, 223, 147, 11, 22, 44, 88, 176, 77, 154, 25, 50, 100, 200,
    189, 87, 174, 113, 226, 233, 255, 211, 139, 59, 118, 236, 245, 199, 163, 107,
    214, 129, 47, 94, 188, 85, 170, 121, 242, 201, 191, 83, 166, 97, 194, 169,
    127, 254, 209, 143, 51, 102, 204, 181, 71, 142, 49, 98, 196, 165, 103, 206,
    177, 79, 158, 17, 34, 68, 136, 61, 122, 244, 197, 167, 99, 198, 161, 111,
    222, 145, 15, 30, 60, 120, 240, 205, 183, 67, 134, 33, 66, 132, 37, 74,
    148, 5, 10, 20, 40, 80, 160, 109, 218, 153, 31, 62, 124, 248, 221, 151,
    3, 6, 12, 24, 48, 96, 192, 173, 119, 238, 241, 207, 179, 75, 150, 1,
]

private let dxLogVal: [Int] = [
    -255, 255, 1, 240, 2, 225, 241, 53, 3, 38, 226, 133, 242, 43, 54, 210,
    4, 195, 39, 114, 227, 106, 134, 28, 243, 140, 44, 23, 55, 118, 211, 234,
    5, 219, 196, 96, 40, 222, 115, 103, 228, 78, 107, 125, 135, 8, 29, 162,
    244, 186, 141, 180, 45, 99, 24, 49, 56, 13, 119, 153, 212, 199, 235, 91,
    6, 76, 220, 217, 197, 11, 97, 184, 41, 36, 223, 253, 116, 138, 104, 193,
    229, 86, 79, 171, 108, 165, 126, 145, 136, 34, 9, 74, 30, 32, 163, 84,
    245, 173, 187, 204, 142, 81, 181, 190, 46, 88, 100, 159, 25, 231, 50, 207,
    57, 147, 14, 67, 120, 128, 154, 248, 213, 167, 200, 63, 236, 110, 92, 176,
    7, 161, 77, 124, 221, 102, 218, 95, 198, 90, 12, 152, 98, 48, 185, 179,
    42, 209, 37, 132, 224, 52, 254, 239, 117, 233, 139, 22, 105, 27, 194, 113,
    230, 206, 87, 158, 80, 189, 172, 203, 109, 175, 166, 62, 127, 247, 146, 66,
    137, 192, 35, 252, 10, 183, 75, 216, 31, 83, 33, 73, 164, 144, 85, 170,
    246, 65, 174, 61, 188, 202, 205, 157, 143, 169, 82, 72, 182, 215, 191, 251,
    47, 178, 89, 151, 101, 94, 160, 123, 26, 112, 232, 21, 51, 238, 208, 131,
    58, 69, 148, 18, 15, 16, 68, 17, 121, 149, 129, 19, 155, 59, 249, 70,
    214, 250, 168, 71, 201, 156, 64, 60, 237, 130, 111, 20, 93, 122, 177, 150,
]

// MARK: - GF(256) Helper Functions

private func dxGFProduct(_ a: Int, _ b: Int) -> Int {
    if a == 0 || b == 0 { return 0 }
    return dxALogVal[(dxLogVal[a] + dxLogVal[b]) % 255]
}

private func dxGFSum(_ a: Int, _ b: Int) -> Int {
    return a ^ b
}

private func dxGFDoublify(_ a: Int, _ b: Int) -> Int {
    if a == 0 { return 0 }
    if b == 0 { return a }
    return dxALogVal[(dxLogVal[a] + b) % 255]
}

// MARK: - Symbol Attribute Lookup

private func dxGetSymbolAttribute(_ attribute: Int, _ sizeIdx: Int) -> Int {
    if sizeIdx < 0 || sizeIdx >= dxSzSquareCount + dxSzRectCount {
        return dxUndefined
    }
    switch attribute {
    case dxAttrSymbolRows:
        return dxSymbolRows[sizeIdx]
    case dxAttrSymbolCols:
        return dxSymbolCols[sizeIdx]
    case dxAttrDataRegionRows:
        return dxDataRegionRows[sizeIdx]
    case dxAttrDataRegionCols:
        return dxDataRegionCols[sizeIdx]
    case dxAttrHorizDataRegions:
        return dxHorizDataRegions[sizeIdx]
    case dxAttrVertDataRegions:
        if sizeIdx < dxSzSquareCount {
            return dxHorizDataRegions[sizeIdx]
        }
        return 1
    case dxAttrMappingMatrixRows:
        return dxDataRegionRows[sizeIdx] * dxGetSymbolAttribute(dxAttrVertDataRegions, sizeIdx)
    case dxAttrMappingMatrixCols:
        return dxDataRegionCols[sizeIdx] * dxHorizDataRegions[sizeIdx]
    case dxAttrInterleavedBlocks:
        return dxInterleavedBlocks[sizeIdx]
    case dxAttrBlockErrorWords:
        return dxBlockErrorWords[sizeIdx]
    case dxAttrBlockMaxCorrectable:
        return dxBlockMaxCorrectable[sizeIdx]
    case dxAttrSymbolDataWords:
        return dxSymbolDataWords[sizeIdx]
    case dxAttrSymbolErrorWords:
        return dxBlockErrorWords[sizeIdx] * dxInterleavedBlocks[sizeIdx]
    case dxAttrSymbolMaxCorrectable:
        return dxBlockMaxCorrectable[sizeIdx] * dxInterleavedBlocks[sizeIdx]
    default:
        return dxUndefined
    }
}

private func dxGetBlockDataSize(_ sizeIdx: Int, _ blockIdx: Int) -> Int {
    let sdw = dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx)
    let ilb = dxGetSymbolAttribute(dxAttrInterleavedBlocks, sizeIdx)
    if sdw < 1 || ilb < 1 {
        return dxUndefined
    }
    let count = sdw / ilb
    if sizeIdx == DX_SZ_144x144 && blockIdx < 8 {
        return count + 1
    }
    return count
}

// MARK: - Reed-Solomon ECC Generation

private func dxGenReedSolECC(_ code: inout [Int], _ sizeIdx: Int) {
    let sdw = dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx)
    let sew = dxGetSymbolAttribute(dxAttrSymbolErrorWords, sizeIdx)
    let stw = sdw + sew
    let bew = dxGetSymbolAttribute(dxAttrBlockErrorWords, sizeIdx)
    let step = dxGetSymbolAttribute(dxAttrInterleavedBlocks, sizeIdx)

    if bew != sew / step {
        return
    }

    var g = [Int](repeating: 0x01, count: 69)

    for i in 1...bew {
        for j in stride(from: i - 1, through: 0, by: -1) {
            g[j] = dxGFDoublify(g[j], i)
            if j > 0 {
                g[j] = dxGFSum(g[j], g[j - 1])
            }
        }
    }

    for block in 0..<step {
        var b = [Int](repeating: 0, count: 68)

        var i = block
        while i < sdw {
            let val = dxGFSum(b[bew - 1], code[i])
            for j in stride(from: bew - 1, through: 1, by: -1) {
                b[j] = dxGFSum(b[j - 1], dxGFProduct(g[j], val))
            }
            b[0] = dxGFProduct(g[0], val)
            i += step
        }

        let blockDataWords = dxGetBlockDataSize(sizeIdx, block)
        var bIndex = bew
        var idx = block + (step * blockDataWords)
        while idx < stw {
            bIndex -= 1
            code[idx] = b[bIndex]
            idx += step
        }
    }
}

// MARK: - Randomize Helpers

private func dxRandomize253(_ codewordValue: Int, _ codewordPosition: Int) -> Int {
    let pseudoRandom = ((149 * codewordPosition) % 253) + 1
    var tmp = codewordValue + pseudoRandom
    if tmp > 254 {
        tmp -= 254
    }
    return tmp & 0xFF
}

private func dxRandomize255(_ codewordValue: Int, _ codewordPosition: Int) -> Int {
    let pseudoRandom = ((149 * codewordPosition) % 255) + 1
    let tmp = codewordValue + pseudoRandom
    if tmp <= 255 {
        return tmp & 0xFF
    }
    return (tmp - 256) & 0xFF
}

private func dxUnRandomize255(_ value: Int, _ idx: Int) -> Int {
    let pseudoRandom = ((149 * idx) % 255) + 1
    var tmp = value - pseudoRandom
    if tmp < 0 {
        tmp += 256
    }
    return tmp & 0xFF
}

// MARK: - Triplet/Quadruplet Helpers

private func dxGetTripletValues(_ cw1: Int, _ cw2: Int) -> [Int] {
    let compact = (cw1 << 8) | cw2
    return [
        (compact - 1) / 1600,
        ((compact - 1) / 40) % 40,
        (compact - 1) % 40,
    ]
}

private func dxGetQuadrupletValues(_ cw1: Int, _ cw2: Int, _ cw3: Int) -> [Int] {
    return [
        cw1 >> 2,
        ((cw1 & 0x03) << 4) | ((cw2 & 0xF0) >> 4),
        ((cw2 & 0x0F) << 2) | ((cw3 & 0xC0) >> 6),
        cw3 & 0x3F,
    ]
}

private func dxIsDigitByte(_ v: Int) -> Bool {
    return v >= 48 && v <= 57
}

// MARK: - DxMessage

private class DxMessage {
    var array: [Int]
    var code: [Int]
    var padCount: Int = 0

    init(sizeIdx: Int) {
        let mr = dxGetSymbolAttribute(dxAttrMappingMatrixRows, sizeIdx)
        let mc = dxGetSymbolAttribute(dxAttrMappingMatrixCols, sizeIdx)
        let cs = dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx)
             + dxGetSymbolAttribute(dxAttrSymbolErrorWords, sizeIdx)
        array = [Int](repeating: 0, count: mc * mr)
        code = [Int](repeating: 0, count: cs)
    }

    func symbolModuleStatus(_ sizeIdx: Int, _ symbolRow: Int, _ symbolCol: Int) -> Int {
        let drr = dxGetSymbolAttribute(dxAttrDataRegionRows, sizeIdx)
        let drc = dxGetSymbolAttribute(dxAttrDataRegionCols, sizeIdx)
        let sr  = dxGetSymbolAttribute(dxAttrSymbolRows, sizeIdx)
        let mc  = dxGetSymbolAttribute(dxAttrMappingMatrixCols, sizeIdx)

        let srr = sr - symbolRow - 1
        let mappingRow = srr - 1 - 2 * (srr / (drr + 2))
        let mappingCol = symbolCol - 1 - 2 * (symbolCol / (drc + 2))

        // Solid finder pattern (L-shape: bottom and left edges)
        if symbolRow % (drr + 2) == 0 || symbolCol % (drc + 2) == 0 {
            return dxModuleOnRGB
        }

        // Alternating clock pattern (top edge)
        if (symbolRow + 1) % (drr + 2) == 0 {
            if (symbolCol & 0x01) != 0 {
                return 0
            }
            return dxModuleOnRGB
        }

        // Alternating clock pattern (right edge)
        if (symbolCol + 1) % (drc + 2) == 0 {
            if (symbolRow & 0x01) != 0 {
                return 0
            }
            return dxModuleOnRGB
        }

        // Data module
        return array[mappingRow * mc + mappingCol] | dxModuleData
    }
}

// MARK: - DxChannel

private class DxChannel {
    var encScheme: Int
    var invalid: Int
    var inputIndex: Int
    var encodedLength: Int
    var currentLength: Int
    var firstCodeWord: Int
    var input: [Int]
    var encodedWords: [Int]

    init() {
        encScheme = DX_SCHEME_ASCII
        invalid = dxChannelValid
        inputIndex = 0
        encodedLength = 0
        currentLength = 0
        firstCodeWord = 0
        input = []
        encodedWords = [Int](repeating: 0, count: 1558)
    }

    func copy() -> DxChannel {
        let nc = DxChannel()
        nc.encScheme = encScheme
        nc.invalid = invalid
        nc.inputIndex = inputIndex
        nc.encodedLength = encodedLength
        nc.currentLength = currentLength
        nc.firstCodeWord = firstCodeWord
        nc.input = input // shared read-only
        nc.encodedWords = [Int](encodedWords)
        return nc
    }
}

// MARK: - DxEncode (Main Encoding Engine)

private class DxEncode {
    var scheme: Int
    var sizeIdxReq: Int
    var marginSize: Int
    var moduleSize: Int
    var flgGS1: Bool
    var message: DxMessage?
    var rawData: [[Bool]]?
    var sizeIdx: Int
    var symbolRows: Int
    var symbolCols: Int
    var mappingRows: Int
    var mappingCols: Int

    init() {
        scheme = DX_SCHEME_ASCII
        sizeIdxReq = DX_SZ_AUTO
        marginSize = 10
        moduleSize = 5
        flgGS1 = false
        message = nil
        rawData = nil
        sizeIdx = 0
        symbolRows = 0
        symbolCols = 0
        mappingRows = 0
        mappingCols = 0
    }

    // MARK: - encodeDataMatrixRaw

    func encodeDataMatrixRaw(_ inputBytes: [Int]) throws {
        var buf = [Int](repeating: 0, count: 4096)
        var sizeIdx = self.sizeIdxReq

        let (dataWordCount, newSizeIdx) = try encodeDataCodewords(&buf, inputBytes, sizeIdx)
        sizeIdx = newSizeIdx

        if dataWordCount <= 0 {
            throw BarcodeError.encodingFailed("failed to encode data: data may be empty or contain invalid characters")
        }
        if sizeIdx == DX_SZ_AUTO || sizeIdx == DX_SZ_RECT_AUTO {
            throw BarcodeError.encodingFailed("data too long for specified symbol size")
        }

        let paddedSize = dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx)
        let padCount = addPadChars(&buf, dataWordCount, paddedSize)

        self.sizeIdx = sizeIdx
        self.symbolRows = dxGetSymbolAttribute(dxAttrSymbolRows, sizeIdx)
        self.symbolCols = dxGetSymbolAttribute(dxAttrSymbolCols, sizeIdx)
        self.mappingRows = dxGetSymbolAttribute(dxAttrMappingMatrixRows, sizeIdx)
        self.mappingCols = dxGetSymbolAttribute(dxAttrMappingMatrixCols, sizeIdx)

        self.message = DxMessage(sizeIdx: sizeIdx)
        self.message!.padCount = padCount

        for i in 0..<paddedSize {
            self.message!.code[i] = buf[i]
        }

        dxGenReedSolECC(&self.message!.code, sizeIdx)
        modulePlacementECC200(&self.message!.array, self.message!.code, sizeIdx, dxModuleOnRGB)
        printPatternRaw()
    }

    // MARK: - Encoding Schemes

    func encodeDataCodewords(_ buf: inout [Int], _ inputBytes: [Int], _ sizeIdx: Int) throws -> (Int, Int) {
        var dwc: Int

        switch scheme {
        case DX_SCHEME_AUTO_BEST:
            dwc = encodeAutoBest(&buf, inputBytes)
        case DX_SCHEME_AUTO_FAST:
            dwc = 0
        default:
            dwc = encodeSingleScheme(&buf, inputBytes, scheme)
        }

        let newSizeIdx = findCorrectSymbolSize(dwc, sizeIdxReq)
        if newSizeIdx == DX_SZ_SHAPE_AUTO {
            return (0, newSizeIdx)
        }
        return (dwc, newSizeIdx)
    }

    func findCorrectSymbolSize(_ dataWords: Int, _ sizeIdxRequest: Int) -> Int {
        if dataWords <= 0 {
            return DX_SZ_SHAPE_AUTO
        }

        var sizeIdx = sizeIdxRequest

        if sizeIdxRequest == DX_SZ_AUTO || sizeIdxRequest == DX_SZ_RECT_AUTO {
            var idxBeg: Int
            var idxEnd: Int
            if sizeIdxRequest == DX_SZ_AUTO {
                idxBeg = 0
                idxEnd = dxSzSquareCount
            } else {
                idxBeg = dxSzSquareCount
                idxEnd = dxSzSquareCount + dxSzRectCount
            }

            sizeIdx = idxBeg
            for i in idxBeg..<idxEnd {
                sizeIdx = i
                if dxGetSymbolAttribute(dxAttrSymbolDataWords, i) >= dataWords {
                    break
                }
            }

            if sizeIdx == idxEnd {
                return DX_SZ_SHAPE_AUTO
            }
        }

        if dataWords > dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx) {
            return DX_SZ_SHAPE_AUTO
        }

        return sizeIdx
    }

    func encodeSingleScheme(_ buf: inout [Int], _ codewords: [Int], _ scheme: Int) -> Int {
        let channel = DxChannel()
        initChannel(channel, codewords)

        while channel.inputIndex < channel.input.count {
            let ok = encodeNextWord(channel, scheme)
            if !ok {
                return 0
            }
            if channel.invalid != dxChannelValid {
                return 0
            }
        }

        let size = channel.encodedLength / 12
        for i in 0..<size {
            buf[i] = channel.encodedWords[i]
        }
        return size
    }

    func encodeAutoBest(_ buf: inout [Int], _ codewords: [Int]) -> Int {
        var channels = [DxChannel?](repeating: nil, count: 6)

        for ts in DX_SCHEME_ASCII...DX_SCHEME_BASE256 {
            channels[ts] = DxChannel()
            initChannel(channels[ts]!, codewords)
            let ok = encodeNextWord(channels[ts]!, ts)
            // C++ bug: "if (err) return 0;" - returns 0 when encoding succeeds
            if ok {
                return 0
            }
        }

        // Encode remaining (unreachable due to above bug, ported for faithfulness)
        while channels[0]!.inputIndex < channels[0]!.input.count {
            var best = [DxChannel?](repeating: nil, count: 6)
            for ts in DX_SCHEME_ASCII...DX_SCHEME_BASE256 {
                best[ts] = findBestChannel(channels, ts)
            }
            channels = best
        }

        var winner = channels[DX_SCHEME_ASCII]!
        for ts in (DX_SCHEME_ASCII + 1)...DX_SCHEME_BASE256 {
            guard let ch = channels[ts] else { continue }
            if ch.invalid != dxChannelValid { continue }
            if ch.encodedLength < winner.encodedLength {
                winner = ch
            }
        }

        let ws = winner.encodedLength / 12
        for i in 0..<ws {
            buf[i] = winner.encodedWords[i]
        }
        return ws
    }

    func findBestChannel(_ channels: [DxChannel?], _ targetScheme: Int) -> DxChannel {
        var winner: DxChannel? = nil
        for s in DX_SCHEME_ASCII...DX_SCHEME_BASE256 {
            guard let channel = channels[s] else { continue }
            if channel.invalid != dxChannelValid { continue }
            if channel.inputIndex == channel.input.count { continue }
            encodeNextWord(channel, targetScheme)
            if (channel.invalid & dxChannelUnsupportedChar) != 0 {
                return channel.copy()
            }
            if (channel.invalid & dxChannelCannotUnlatch) != 0 {
                continue
            }
            if winner == nil || channel.currentLength < winner!.currentLength {
                winner = channel.copy()
            }
        }
        if let w = winner {
            return w
        }
        return DxChannel()
    }

    @discardableResult
    func encodeNextWord(_ channel: DxChannel, _ targetScheme: Int) -> Bool {
        if channel.encScheme != targetScheme {
            changeEncScheme(channel, targetScheme, dxUnlatchExplicit)
            if channel.invalid != dxChannelValid {
                return false
            }
        }

        if channel.encScheme != targetScheme {
            return false
        }

        switch channel.encScheme {
        case DX_SCHEME_ASCII:
            return encodeAsciiCodeword(channel)
        case DX_SCHEME_C40, DX_SCHEME_TEXT, DX_SCHEME_X12:
            return encodeTripletCodeword(channel)
        case DX_SCHEME_EDIFACT:
            return encodeEdifactCodeword(channel)
        case DX_SCHEME_BASE256:
            return encodeBase256Codeword(channel)
        default:
            return false
        }
    }

    // MARK: - ASCII Encoding

    func encodeAsciiCodeword(_ channel: DxChannel) -> Bool {
        let inputValue = channel.input[channel.inputIndex]

        // Digit pair compression
        if dxIsDigitByte(inputValue) && channel.currentLength >= channel.firstCodeWord + 12 {
            let prevIndex = (channel.currentLength - 12) / 12
            let prevValue = (channel.encodedWords[prevIndex] - 1) & 0xFF
            var prevPrevValue = 0
            if prevIndex > channel.firstCodeWord / 12 {
                prevPrevValue = channel.encodedWords[prevIndex - 1]
            }

            if prevPrevValue != 235 && dxIsDigitByte(prevValue) {
                channel.encodedWords[prevIndex] = (10 * (prevValue - 48) + (inputValue - 48) + 130) & 0xFF
                channel.inputIndex += 1
                return true
            }
        }

        // FNC1 in GS1 mode
        if flgGS1 && inputValue == dxCharFNC1 {
            pushInputWord(channel, dxCharFNC1)
            incrementProgress(channel, 12)
            channel.inputIndex += 1
            return true
        }

        var adjInputValue = inputValue

        // Upper shift for values >= 128
        if adjInputValue >= 128 {
            pushInputWord(channel, dxCharAsciiUpperShift)
            incrementProgress(channel, 12)
            adjInputValue -= 128
        }

        pushInputWord(channel, (adjInputValue + 1) & 0xFF)
        incrementProgress(channel, 12)
        channel.inputIndex += 1
        return true
    }

    // MARK: - Triplet Encoding (C40/Text/X12)

    func encodeTripletCodeword(_ channel: DxChannel) -> Bool {
        if channel.encScheme != DX_SCHEME_C40 && channel.encScheme != DX_SCHEME_TEXT &&
            channel.encScheme != DX_SCHEME_X12 {
            return false
        }

        if channel.currentLength > channel.encodedLength {
            return false
        }

        if channel.currentLength == channel.encodedLength {
            if channel.currentLength % 12 != 0 {
                return false
            }

            var ptrIndex = channel.inputIndex
            var tripletCount = 0
            var buffer = [Int](repeating: 0, count: 6)

            outerLoop: while true {
                while tripletCount < 3 && ptrIndex < channel.input.count {
                    let inputWord = channel.input[ptrIndex]
                    ptrIndex += 1
                    let outputWords = getC40TextX12Words(inputWord, channel.encScheme)
                    if outputWords.isEmpty {
                        channel.invalid = dxChannelUnsupportedChar
                        return false
                    }
                    for w in outputWords {
                        buffer[tripletCount] = w & 0xFF
                        tripletCount += 1
                    }
                }

                var triplet = [buffer[0], buffer[1], buffer[2]]

                if tripletCount >= 3 {
                    pushTriplet(channel, triplet)
                    buffer[0] = buffer[3]
                    buffer[1] = buffer[4]
                    buffer[2] = buffer[5]
                    tripletCount -= 3
                }

                if ptrIndex == channel.input.count {
                    while channel.currentLength < channel.encodedLength {
                        incrementProgress(channel, 8)
                        channel.inputIndex += 1
                    }

                    if channel.currentLength == channel.encodedLength + 8 {
                        channel.currentLength = channel.encodedLength
                        channel.inputIndex -= 1
                    }

                    let inputCount = channel.input.count - channel.inputIndex
                    triplet = [buffer[0], buffer[1], buffer[2]]
                    let ok = processEndOfSymbolTriplet(channel, triplet, tripletCount, inputCount)
                    if !ok {
                        return false
                    }
                    break outerLoop
                }

                if tripletCount == 0 {
                    break outerLoop
                }
            }
        }

        if channel.currentLength < channel.encodedLength {
            incrementProgress(channel, 8)
            channel.inputIndex += 1
        }

        return true
    }

    func getC40TextX12Words(_ inputWord: Int, _ encScheme: Int) -> [Int] {
        var result = [Int]()
        var adjInput = inputWord

        if adjInput > 127 {
            if encScheme == DX_SCHEME_X12 {
                return []
            }
            result.append(dxCharTripletShift2)
            result.append(30)
            adjInput -= 128
        }

        if encScheme == DX_SCHEME_X12 {
            if adjInput == 13 {
                result.append(0)
            } else if adjInput == 42 {
                result.append(1)
            } else if adjInput == 62 {
                result.append(2)
            } else if adjInput == 32 {
                result.append(3)
            } else if adjInput >= 45 && adjInput <= 57 {
                result.append(adjInput - 44)
            } else if adjInput >= 65 && adjInput <= 90 {
                result.append(adjInput - 51)
            }
        } else {
            if adjInput <= 31 {
                result.append(dxCharTripletShift1)
                result.append(adjInput)
            } else if adjInput == 32 {
                result.append(3)
            } else if adjInput <= 47 {
                result.append(dxCharTripletShift2)
                result.append(adjInput - 33)
            } else if adjInput <= 57 {
                result.append(adjInput - 44)
            } else if adjInput <= 64 {
                result.append(dxCharTripletShift2)
                result.append(adjInput - 43)
            } else if adjInput <= 90 && encScheme == DX_SCHEME_C40 {
                result.append(adjInput - 51)
            } else if adjInput <= 90 && encScheme == DX_SCHEME_TEXT {
                result.append(dxCharTripletShift3)
                result.append(adjInput - 64)
            } else if adjInput <= 95 {
                result.append(dxCharTripletShift2)
                result.append(adjInput - 69)
            } else if adjInput == 96 && encScheme == DX_SCHEME_TEXT {
                result.append(dxCharTripletShift3)
                result.append(0)
            } else if adjInput <= 122 && encScheme == DX_SCHEME_TEXT {
                result.append(adjInput - 83)
            } else if adjInput <= 127 {
                result.append(dxCharTripletShift3)
                result.append(adjInput - 96)
            }
        }

        return result
    }

    func processEndOfSymbolTriplet(_ channel: DxChannel, _ triplet: [Int], _ tripletCount: Int, _ inputCount: Int) -> Bool {
        if channel.currentLength % 12 != 0 {
            return false
        }

        let inputAdjust = tripletCount - inputCount
        var currentByte = channel.currentLength / 12
        var needed = currentByte + inputCount
        if inputCount == 3 {
            needed = currentByte + 2
        }
        var sizeIdx = findCorrectSymbolSize(needed, sizeIdxReq)
        if sizeIdx == DX_SZ_SHAPE_AUTO {
            return false
        }

        var remaining = dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx) - currentByte

        if inputCount == 1 && remaining == 1 {
            changeEncScheme(channel, DX_SCHEME_ASCII, dxUnlatchImplicit)
            let ok = encodeNextWord(channel, DX_SCHEME_ASCII)
            if !ok {
                return false
            }
            if channel.invalid != dxChannelValid || channel.inputIndex != channel.input.count {
                return false
            }
        } else if remaining == 2 {
            if tripletCount == 3 {
                pushTriplet(channel, triplet)
                incrementProgress(channel, 24)
                channel.encScheme = DX_SCHEME_ASCII
                channel.inputIndex += 3
                channel.inputIndex -= inputAdjust
            } else if tripletCount == 2 {
                let t = [triplet[0], triplet[1], 0]
                pushTriplet(channel, t)
                incrementProgress(channel, 24)
                channel.encScheme = DX_SCHEME_ASCII
                channel.inputIndex += 2
                channel.inputIndex -= inputAdjust
            } else if tripletCount == 1 {
                changeEncScheme(channel, DX_SCHEME_ASCII, dxUnlatchExplicit)
                let ok = encodeNextWord(channel, DX_SCHEME_ASCII)
                if !ok {
                    return false
                }
                if channel.invalid != dxChannelValid {
                    return false
                }
            }
        } else {
            currentByte = channel.currentLength / 12
            remaining = dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx) - currentByte
            if remaining > 0 {
                changeEncScheme(channel, DX_SCHEME_ASCII, dxUnlatchExplicit)
                while channel.inputIndex < channel.input.count {
                    let ok = encodeNextWord(channel, DX_SCHEME_ASCII)
                    if !ok {
                        return false
                    }
                    if channel.invalid != dxChannelValid {
                        return false
                    }
                }
            }
        }

        return channel.inputIndex == channel.input.count
    }

    func pushTriplet(_ channel: DxChannel, _ triplet: [Int]) {
        let tv = 1600 * triplet[0] + 40 * triplet[1] + triplet[2] + 1
        pushInputWord(channel, (tv >> 8) & 0xFF)
        pushInputWord(channel, tv & 0xFF)
    }

    // MARK: - Edifact Encoding

    func encodeEdifactCodeword(_ channel: DxChannel) -> Bool {
        if channel.encScheme != DX_SCHEME_EDIFACT {
            return false
        }

        let inputValue = channel.input[channel.inputIndex]
        if inputValue < 32 || inputValue > 94 {
            channel.invalid = dxChannelUnsupportedChar
            return false
        }

        pushInputWord(channel, inputValue & 0x3F)
        incrementProgress(channel, 9)
        channel.inputIndex += 1

        checkEndOfSymbolEdifact(channel)
        return true
    }

    func checkEndOfSymbolEdifact(_ channel: DxChannel) {
        let edifactValues = channel.input.count - channel.inputIndex
        if edifactValues > 4 {
            return
        }

        let currentByte = channel.currentLength / 12
        let sizeIdx = findCorrectSymbolSize(currentByte, DX_SZ_AUTO)
        let symbolCodewords = dxGetSymbolAttribute(dxAttrSymbolDataWords, sizeIdx) - currentByte

        if channel.currentLength % 12 == 0 && (symbolCodewords == 1 || symbolCodewords == 2) {
            let asciiCodewords = edifactValues
            if asciiCodewords <= symbolCodewords {
                changeEncScheme(channel, DX_SCHEME_ASCII, dxUnlatchImplicit)
                for _ in 0..<edifactValues {
                    let ok = encodeNextWord(channel, DX_SCHEME_ASCII)
                    if !ok {
                        return
                    }
                    if channel.invalid != dxChannelValid {
                        return
                    }
                }
            }
        } else if edifactValues == 0 {
            changeEncScheme(channel, DX_SCHEME_ASCII, dxUnlatchExplicit)
        }
    }

    // MARK: - Base256 Encoding

    func encodeBase256Codeword(_ channel: DxChannel) -> Bool {
        if channel.encScheme != DX_SCHEME_BASE256 {
            return false
        }

        let firstBytePtrIndex = channel.firstCodeWord / 12
        let headerByte0 = dxUnRandomize255(
            channel.encodedWords[firstBytePtrIndex],
            channel.firstCodeWord / 12 + 1)

        var newDataLength: Int
        if headerByte0 <= 249 {
            newDataLength = headerByte0
        } else {
            newDataLength = 250 * (headerByte0 - 249)
            newDataLength += dxUnRandomize255(
                channel.encodedWords[firstBytePtrIndex + 1],
                channel.firstCodeWord / 12 + 2)
        }

        newDataLength += 1

        var headerByteCount: Int
        var hb = [Int](repeating: 0, count: 2)
        if newDataLength <= 249 {
            headerByteCount = 1
            hb[0] = newDataLength & 0xFF
            hb[1] = 0
        } else {
            headerByteCount = 2
            hb[0] = (newDataLength / 250 + 249) & 0xFF
            hb[1] = (newDataLength % 250) & 0xFF
        }

        if newDataLength <= 0 || newDataLength > 1555 {
            return false
        }

        if newDataLength == 250 {
            var i = channel.currentLength / 12 - 1
            while i > channel.firstCodeWord / 12 {
                let valTmp = dxUnRandomize255(channel.encodedWords[i], i + 1)
                channel.encodedWords[i + 1] = dxRandomize255(valTmp, i + 2)
                i -= 1
            }
            incrementProgress(channel, 12)
            channel.encodedLength += 12
        }

        for i in 0..<headerByteCount {
            channel.encodedWords[firstBytePtrIndex + i] = dxRandomize255(
                hb[i], channel.firstCodeWord / 12 + i + 1)
        }

        pushInputWord(channel, dxRandomize255(
            channel.input[channel.inputIndex], channel.currentLength / 12 + 1))
        incrementProgress(channel, 12)
        channel.inputIndex += 1
        return true
    }

    // MARK: - Channel Utilities

    func pushInputWord(_ channel: DxChannel, _ codeword: Int) {
        if channel.encodedLength / 12 > 3 * 1558 {
            return
        }

        switch channel.encScheme {
        case DX_SCHEME_ASCII, DX_SCHEME_C40, DX_SCHEME_TEXT, DX_SCHEME_X12, DX_SCHEME_BASE256:
            channel.encodedWords[channel.currentLength / 12] = codeword & 0xFF
            channel.encodedLength += 12

        case DX_SCHEME_EDIFACT:
            let pos = channel.currentLength % 4
            let startByte = ((channel.currentLength + 9) / 12) - pos

            var q = dxGetQuadrupletValues(
                channel.encodedWords[startByte],
                channel.encodedWords[startByte + 1],
                channel.encodedWords[startByte + 2])
            q[pos] = codeword & 0xFF
            for i in (pos + 1)..<4 {
                q[i] = 0
            }

            if pos >= 2 {
                channel.encodedWords[startByte + 2] = ((q[2] & 0x03) << 6) | q[3]
            }
            if pos >= 1 {
                channel.encodedWords[startByte + 1] = ((q[1] & 0x0F) << 4) | (q[2] >> 2)
            }
            channel.encodedWords[startByte] = (q[0] << 2) | (q[1] >> 4)

            channel.encodedLength += 9

        default:
            break
        }
    }

    func incrementProgress(_ channel: DxChannel, _ encodedUnits: Int) {
        if channel.encScheme == DX_SCHEME_C40 || channel.encScheme == DX_SCHEME_TEXT {
            let pos = (channel.currentLength % 6) / 2
            let startByte = (channel.currentLength / 12) - (pos >> 1)
            let triplet = dxGetTripletValues(
                channel.encodedWords[startByte],
                channel.encodedWords[startByte + 1])
            if triplet[pos] <= 2 {
                channel.currentLength += 8
            }
        }

        channel.currentLength += encodedUnits
    }

    func changeEncScheme(_ channel: DxChannel, _ targetScheme: Int, _ unlatchType: Int) {
        if channel.encScheme == targetScheme {
            return
        }

        switch channel.encScheme {
        case DX_SCHEME_ASCII:
            if channel.currentLength % 12 != 0 && channel.currentLength % 20 != 0 {
                return
            }

        case DX_SCHEME_C40, DX_SCHEME_TEXT, DX_SCHEME_X12:
            if channel.currentLength % 12 != 0 && channel.currentLength % 20 != 0 {
                channel.invalid = dxChannelCannotUnlatch
                return
            }
            if channel.currentLength != channel.encodedLength {
                channel.invalid = dxChannelCannotUnlatch
                return
            }
            if unlatchType == dxUnlatchExplicit {
                pushInputWord(channel, dxCharTripletUnlatch)
                incrementProgress(channel, 12)
            }

        case DX_SCHEME_EDIFACT:
            if channel.currentLength % 3 != 0 {
                return
            }
            if unlatchType == dxUnlatchExplicit {
                pushInputWord(channel, dxCharEdifactUnlatch)
                incrementProgress(channel, 9)
            }
            let advance = (channel.currentLength % 4) * 3
            channel.currentLength += advance
            channel.encodedLength += advance

        default:
            // Base256: nothing on unlatch
            break
        }

        channel.encScheme = DX_SCHEME_ASCII

        switch targetScheme {
        case DX_SCHEME_ASCII:
            break // no latch needed
        case DX_SCHEME_C40:
            pushInputWord(channel, dxCharC40Latch)
            incrementProgress(channel, 12)
        case DX_SCHEME_TEXT:
            pushInputWord(channel, dxCharTextLatch)
            incrementProgress(channel, 12)
        case DX_SCHEME_X12:
            pushInputWord(channel, dxCharX12Latch)
            incrementProgress(channel, 12)
        case DX_SCHEME_EDIFACT:
            pushInputWord(channel, dxCharEdifactLatch)
            incrementProgress(channel, 12)
        case DX_SCHEME_BASE256:
            pushInputWord(channel, dxCharBase256Latch)
            incrementProgress(channel, 12)
            pushInputWord(channel, dxRandomize255(0, 2))
            incrementProgress(channel, 12)
        default:
            break
        }

        channel.encScheme = targetScheme
        channel.firstCodeWord = channel.currentLength - 12
    }

    func initChannel(_ channel: DxChannel, _ codewords: [Int]) {
        channel.encScheme = DX_SCHEME_ASCII
        channel.invalid = dxChannelValid
        channel.inputIndex = 0
        channel.input = [Int](codewords)
    }

    // MARK: - Padding

    func addPadChars(_ buf: inout [Int], _ dataWordCount: Int, _ paddedSize: Int) -> Int {
        var padCount = 0
        var dwc = dataWordCount
        if dwc < paddedSize {
            padCount += 1
            buf[dwc] = dxCharAsciiPad
            dwc += 1
        }

        while dwc < paddedSize {
            padCount += 1
            buf[dwc] = dxRandomize253(dxCharAsciiPad, dwc + 1)
            dwc += 1
        }

        return padCount
    }

    // MARK: - Module Placement ECC200

    func modulePlacementECC200(_ modules: inout [Int], _ codewords: [Int], _ sizeIdx: Int, _ moduleOnColor: Int) {
        let mr = dxGetSymbolAttribute(dxAttrMappingMatrixRows, sizeIdx)
        let mc = dxGetSymbolAttribute(dxAttrMappingMatrixCols, sizeIdx)

        var ch = 0
        var row = 4
        var col = 0

        while true {
            if row == mr && col == 0 {
                patternShapeSpecial1(&modules, mr, mc, codewords, ch, moduleOnColor)
                ch += 1
            } else if row == mr - 2 && col == 0 && mc % 4 != 0 {
                patternShapeSpecial2(&modules, mr, mc, codewords, ch, moduleOnColor)
                ch += 1
            } else if row == mr - 2 && col == 0 && mc % 8 == 4 {
                patternShapeSpecial3(&modules, mr, mc, codewords, ch, moduleOnColor)
                ch += 1
            } else if row == mr + 4 && col == 2 && mc % 8 == 0 {
                patternShapeSpecial4(&modules, mr, mc, codewords, ch, moduleOnColor)
                ch += 1
            }

            // Upward diagonal
            while true {
                if row < mr && col >= 0 && (modules[row * mc + col] & dxModuleVisited) == 0 {
                    patternShapeStandard(&modules, mr, mc, row, col, codewords, ch, moduleOnColor)
                    ch += 1
                }
                row -= 2
                col += 2
                if !(row >= 0 && col < mc) {
                    break
                }
            }
            row += 1
            col += 3

            // Downward diagonal
            while true {
                if row >= 0 && col < mc && (modules[row * mc + col] & dxModuleVisited) == 0 {
                    patternShapeStandard(&modules, mr, mc, row, col, codewords, ch, moduleOnColor)
                    ch += 1
                }
                row += 2
                col -= 2
                if !(row < mr && col >= 0) {
                    break
                }
            }
            row += 3
            col += 1

            if !(row < mr || col < mc) {
                break
            }
        }

        // Fill unvisited corner
        if (modules[mr * mc - 1] & dxModuleVisited) == 0 {
            modules[mr * mc - 1] |= moduleOnColor
            modules[mr * mc - mc - 2] |= moduleOnColor
        }
    }

    func patternShapeStandard(_ modules: inout [Int], _ mr: Int, _ mc: Int,
                              _ row: Int, _ col: Int, _ cw: [Int], _ ci: Int, _ color: Int) {
        placeModule(&modules, mr, mc, row - 2, col - 2, cw, ci, dxMaskBit1, color)
        placeModule(&modules, mr, mc, row - 2, col - 1, cw, ci, dxMaskBit2, color)
        placeModule(&modules, mr, mc, row - 1, col - 2, cw, ci, dxMaskBit3, color)
        placeModule(&modules, mr, mc, row - 1, col - 1, cw, ci, dxMaskBit4, color)
        placeModule(&modules, mr, mc, row - 1, col,     cw, ci, dxMaskBit5, color)
        placeModule(&modules, mr, mc, row,     col - 2, cw, ci, dxMaskBit6, color)
        placeModule(&modules, mr, mc, row,     col - 1, cw, ci, dxMaskBit7, color)
        placeModule(&modules, mr, mc, row,     col,     cw, ci, dxMaskBit8, color)
    }

    func patternShapeSpecial1(_ modules: inout [Int], _ mr: Int, _ mc: Int,
                              _ cw: [Int], _ ci: Int, _ color: Int) {
        placeModule(&modules, mr, mc, mr - 1, 0,      cw, ci, dxMaskBit1, color)
        placeModule(&modules, mr, mc, mr - 1, 1,      cw, ci, dxMaskBit2, color)
        placeModule(&modules, mr, mc, mr - 1, 2,      cw, ci, dxMaskBit3, color)
        placeModule(&modules, mr, mc, 0,      mc - 2, cw, ci, dxMaskBit4, color)
        placeModule(&modules, mr, mc, 0,      mc - 1, cw, ci, dxMaskBit5, color)
        placeModule(&modules, mr, mc, 1,      mc - 1, cw, ci, dxMaskBit6, color)
        placeModule(&modules, mr, mc, 2,      mc - 1, cw, ci, dxMaskBit7, color)
        placeModule(&modules, mr, mc, 3,      mc - 1, cw, ci, dxMaskBit8, color)
    }

    func patternShapeSpecial2(_ modules: inout [Int], _ mr: Int, _ mc: Int,
                              _ cw: [Int], _ ci: Int, _ color: Int) {
        placeModule(&modules, mr, mc, mr - 3, 0,      cw, ci, dxMaskBit1, color)
        placeModule(&modules, mr, mc, mr - 2, 0,      cw, ci, dxMaskBit2, color)
        placeModule(&modules, mr, mc, mr - 1, 0,      cw, ci, dxMaskBit3, color)
        placeModule(&modules, mr, mc, 0,      mc - 4, cw, ci, dxMaskBit4, color)
        placeModule(&modules, mr, mc, 0,      mc - 3, cw, ci, dxMaskBit5, color)
        placeModule(&modules, mr, mc, 0,      mc - 2, cw, ci, dxMaskBit6, color)
        placeModule(&modules, mr, mc, 0,      mc - 1, cw, ci, dxMaskBit7, color)
        placeModule(&modules, mr, mc, 1,      mc - 1, cw, ci, dxMaskBit8, color)
    }

    func patternShapeSpecial3(_ modules: inout [Int], _ mr: Int, _ mc: Int,
                              _ cw: [Int], _ ci: Int, _ color: Int) {
        placeModule(&modules, mr, mc, mr - 3, 0,      cw, ci, dxMaskBit1, color)
        placeModule(&modules, mr, mc, mr - 2, 0,      cw, ci, dxMaskBit2, color)
        placeModule(&modules, mr, mc, mr - 1, 0,      cw, ci, dxMaskBit3, color)
        placeModule(&modules, mr, mc, 0,      mc - 2, cw, ci, dxMaskBit4, color)
        placeModule(&modules, mr, mc, 0,      mc - 1, cw, ci, dxMaskBit5, color)
        placeModule(&modules, mr, mc, 1,      mc - 1, cw, ci, dxMaskBit6, color)
        placeModule(&modules, mr, mc, 2,      mc - 1, cw, ci, dxMaskBit7, color)
        placeModule(&modules, mr, mc, 3,      mc - 1, cw, ci, dxMaskBit8, color)
    }

    func patternShapeSpecial4(_ modules: inout [Int], _ mr: Int, _ mc: Int,
                              _ cw: [Int], _ ci: Int, _ color: Int) {
        placeModule(&modules, mr, mc, mr - 1, 0,      cw, ci, dxMaskBit1, color)
        placeModule(&modules, mr, mc, mr - 1, mc - 1, cw, ci, dxMaskBit2, color)
        placeModule(&modules, mr, mc, 0,      mc - 3, cw, ci, dxMaskBit3, color)
        placeModule(&modules, mr, mc, 0,      mc - 2, cw, ci, dxMaskBit4, color)
        placeModule(&modules, mr, mc, 0,      mc - 1, cw, ci, dxMaskBit5, color)
        placeModule(&modules, mr, mc, 1,      mc - 3, cw, ci, dxMaskBit6, color)
        placeModule(&modules, mr, mc, 1,      mc - 2, cw, ci, dxMaskBit7, color)
        placeModule(&modules, mr, mc, 1,      mc - 1, cw, ci, dxMaskBit8, color)
    }

    func placeModule(_ modules: inout [Int], _ mr: Int, _ mc: Int,
                     _ row: Int, _ col: Int, _ codeword: [Int], _ ci: Int, _ mask: Int, _ color: Int) {
        var r = row
        var c = col
        if r < 0 {
            r += mr
            c += 4 - ((mr + 4) % 8)
        }
        if c < 0 {
            c += mc
            r += 4 - ((mc + 4) % 8)
        }

        let index = r * mc + c
        if (modules[index] & dxModuleAssigned) != 0 {
            if (modules[index] & color) != 0 {
                modules[index] = modules[index] | mask
            } else {
                modules[index] = modules[index] & (~mask & 0xFF)
            }
        } else {
            if (codeword[ci] & mask) != 0 {
                modules[index] |= color
            }
            modules[index] |= dxModuleAssigned
        }

        modules[index] |= dxModuleVisited
    }

    // MARK: - Pattern Output

    func printPatternRaw() {
        let sc = symbolCols
        let sr = symbolRows
        let si = sizeIdx

        var data = [[Bool]]()
        for _ in 0..<sc {
            data.append([Bool](repeating: false, count: sr))
        }
        for symbolRow in 0..<sr {
            for symbolCol in 0..<sc {
                let status = message!.symbolModuleStatus(si, symbolRow, symbolCol)
                data[symbolCol][sr - symbolRow - 1] = (status & dxModuleOnBlue) != 0
            }
        }
        rawData = data
    }
}

// MARK: - Public DataMatrix Class

/// DataMatrix generates DataMatrix ECC200 barcodes.
public class DataMatrix: BarcodeBase2D {
    private var codeSize: Int = DX_SZ_AUTO
    private var encodeScheme: Int = DX_SCHEME_ASCII

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    /// Sets the DataMatrix symbol size (use DX_SZ_* constants).
    public func setCodeSize(_ size: Int) {
        codeSize = size
    }

    /// Returns the current code size setting.
    public func getCodeSize() -> Int {
        return codeSize
    }

    /// Sets the encoding scheme (use DX_SCHEME_* constants).
    public func setEncodeScheme(_ scheme: Int) {
        encodeScheme = scheme
    }

    /// Returns the current encoding scheme.
    public func getEncodeScheme() -> Int {
        return encodeScheme
    }

    /// Renders the DataMatrix barcode at the given size.
    /// For DataMatrix, both width and height are controlled by size (square output).
    public func draw(code: String, size: Int) throws {
        let data = stringToByteInts(code)
        try drawBytes(data, size, size)
    }

    /// Returns the DataMatrix pattern as [[Bool]] (indexed [col][row]).
    public func getPattern(code: String) throws -> [[Bool]] {
        let data = stringToByteInts(code)
        return try calDataMatrixBytes(data)
    }

    // MARK: - Private Implementation

    private func drawBytes(_ data: [Int], _ width: Int, _ height: Int) throws {
        let patt = try calDataMatrixBytes(data)

        let patternCols = patt.count
        guard patternCols > 0 else {
            throw BarcodeError.encodingFailed("empty pattern")
        }
        let patternRows = patt[0].count
        guard patternRows > 0 else {
            throw BarcodeError.encodingFailed("empty pattern")
        }

        // Convert [col][row] pattern to [[Bool]] in [row][col] format for drawSVG2D/drawPNG2D
        var matrix = [[Bool]]()
        for r in 0..<patternRows {
            var row = [Bool]()
            for c in 0..<patternCols {
                row.append(patt[c][r])
            }
            matrix.append(row)
        }

        if isSVGOutput() {
            try drawSVG2D(matrix, width: width, height: height)
        } else {
            try drawPNG2D(matrix, width: width, height: height)
        }
    }

    private func calDataMatrixBytes(_ byteData: [Int]) throws -> [[Bool]] {
        let enc = DxEncode()
        enc.moduleSize = 1
        enc.marginSize = 0
        enc.sizeIdxReq = codeSize
        enc.scheme = encodeScheme

        try enc.encodeDataMatrixRaw(byteData)
        guard let rawData = enc.rawData else {
            throw BarcodeError.encodingFailed("encoding produced no pattern data")
        }
        return rawData
    }

    private func stringToByteInts(_ s: String) -> [Int] {
        let bytes = Array(s.utf8)
        return bytes.map { Int($0) }
    }
}
