import Foundation

// MARK: - Multi-Row DataBar Rendering

/// Renders a multi-row pattern-based barcode as SVG.
/// patterns: run-length encoded strings for each row (digit = width, leading "0" means starts with space)
/// rowHeights: height in X units for each row (-1 = variable/fills remaining space)
func drawMultiRowSVG(_ b: BarcodeBase, _ patterns: [String], _ rowHeights: [Int],
                     _ width: Int, _ height: Int) throws {
    if patterns.isEmpty {
        throw BarcodeError.encodingFailed("no patterns to render")
    }

    // Calculate total fixed height and number of variable rows
    var totalFixed = 0
    var variableCount = 0
    for h in rowHeights {
        if h > 0 {
            totalFixed += h
        } else {
            variableCount += 1
        }
    }

    // Calculate row pixel heights
    var rowPxHeights = [Int](repeating: 0, count: rowHeights.count)
    if variableCount > 0 {
        var variableHeight = (height - totalFixed) / variableCount
        if variableHeight < 1 { variableHeight = 1 }
        for i in 0..<rowHeights.count {
            if rowHeights[i] > 0 {
                rowPxHeights[i] = rowHeights[i]
            } else {
                rowPxHeights[i] = variableHeight
            }
        }
    } else {
        let scale = Double(height) / Double(totalFixed)
        for i in 0..<rowHeights.count {
            rowPxHeights[i] = Int((Double(rowHeights[i]) * scale).rounded())
        }
    }

    b.svgBegin(width, height)
    b.svgRect(0, 0, Double(width), Double(height), b.backColor)

    var y = 0
    for ri in 0..<patterns.count {
        if ri >= rowPxHeights.count { break }
        let rh = rowPxHeights[ri]
        drawPatternRowSVG(b, patterns[ri], 0, y, width, rh)
        y += rh
    }

    if isTrialMode() {
        b.svgSampleOverlay(0, 0, width, height)
    }
    b.svgStream += "</svg>\n"
}

/// Draws a single row from a run-length pattern string as SVG.
func drawPatternRowSVG(_ b: BarcodeBase, _ pattern: String, _ x: Int, _ y: Int,
                        _ width: Int, _ height: Int) {
    if pattern.isEmpty { return }

    let patChars = Array(pattern.utf8)

    // Parse pattern: leading "0" means starts with space
    var startIdx = 0
    var isBar = true
    if patChars[0] == UInt8(ascii: "0") {
        isBar = false
        startIdx = 1
    }

    // Calculate total modules
    var totalModules = 0
    for i in startIdx..<patChars.count {
        totalModules += Int(patChars[i] - UInt8(ascii: "0"))
    }
    if totalModules <= 0 { return }

    let unitW = Double(width) / Double(totalModules)
    var accum = 0.0
    var bar = isBar

    for i in startIdx..<patChars.count {
        let modules = Int(patChars[i] - UInt8(ascii: "0"))
        let x1 = (Double(x) + accum * unitW).rounded()
        accum += Double(modules)
        let x2 = (Double(x) + accum * unitW).rounded()
        if bar && x2 > x1 {
            b.svgRect(x1, Double(y), x2 - x1, Double(height), b.foreColor)
        }
        bar = !bar
    }
}

/// Renders a multi-row pattern-based barcode as PNG.
func drawMultiRowPNG(_ b: BarcodeBase, _ patterns: [String], _ rowHeights: [Int],
                     _ width: Int, _ height: Int) throws {
    if patterns.isEmpty {
        throw BarcodeError.encodingFailed("no patterns to render")
    }

    // Calculate total fixed height and number of variable rows
    var totalFixed = 0
    var variableCount = 0
    for h in rowHeights {
        if h > 0 {
            totalFixed += h
        } else {
            variableCount += 1
        }
    }

    // Calculate row pixel heights
    var rowPxHeights = [Int](repeating: 0, count: rowHeights.count)
    if variableCount > 0 {
        var variableHeight = (height - totalFixed) / variableCount
        if variableHeight < 1 { variableHeight = 1 }
        for i in 0..<rowHeights.count {
            if rowHeights[i] > 0 {
                rowPxHeights[i] = rowHeights[i]
            } else {
                rowPxHeights[i] = variableHeight
            }
        }
    } else {
        let scale = Double(height) / Double(totalFixed)
        for i in 0..<rowHeights.count {
            rowPxHeights[i] = Int((Double(rowHeights[i]) * scale).rounded())
        }
    }

    let img = PixelBuffer(width: width, height: height)
    img.fill(b.backColor)

    var y = 0
    for ri in 0..<patterns.count {
        if ri >= rowPxHeights.count { break }
        let rh = rowPxHeights[ri]
        drawPatternRowPNG(img, patterns[ri], 0, y, width, rh, b.foreColor)
        y += rh
    }

    if isTrialMode() {
        drawSampleOverlayPNG(img, x: 0, y: 0, width: width, height: height)
    }

    try b.encodeImageBuffer(img)
}

/// Draws a single row from a run-length pattern string as PNG.
func drawPatternRowPNG(_ img: PixelBuffer, _ pattern: String, _ x: Int, _ y: Int,
                        _ width: Int, _ height: Int, _ foreColor: RGBA) {
    if pattern.isEmpty { return }

    let patChars = Array(pattern.utf8)

    var startIdx = 0
    var isBar = true
    if patChars[0] == UInt8(ascii: "0") {
        isBar = false
        startIdx = 1
    }

    var totalModules = 0
    for i in startIdx..<patChars.count {
        totalModules += Int(patChars[i] - UInt8(ascii: "0"))
    }
    if totalModules <= 0 { return }

    let unitW = Double(width) / Double(totalModules)
    var accum = 0.0
    var bar = isBar

    for i in startIdx..<patChars.count {
        let modules = Int(patChars[i] - UInt8(ascii: "0"))
        let x1 = Int((Double(x) + accum * unitW).rounded())
        accum += Double(modules)
        let x2 = Int((Double(x) + accum * unitW).rounded())
        if bar && x2 > x1 {
            img.fillRect(x1, y, x2, y + height, foreColor)
        }
        bar = !bar
    }
}
