import Foundation

private let code93Chars = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ-. $/+%<>?!"
private let code93Patterns: [String] = [
    "131112", "111213", "111312", "111411", "121113", "121212", "121311",
    "111114", "131211", "141111", "211113", "211212", "211311", "221112",
    "221211", "231111", "112113", "112212", "112311", "122112", "132111",
    "111123", "111222", "111321", "121122", "131121", "212112", "212211",
    "211122", "211221", "221121", "222111", "112122", "112221", "122121",
    "123111", "121131", "311112", "311211", "321111", "112131", "113121",
    "211131", "121221", "312111", "311121", "122211",
]
private let code93Start = "111141"
private let code93Stop  = "1111411"

/// Code93 barcode encoder.
public class Code93: BarcodeBase1D, Encoder1D {
    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    private func modulus47(_ code: String, weight: Int) throws -> Int {
        var total = 0
        let chars = Array(code)
        for i in 0..<chars.count {
            guard let value = code93Chars.firstIndex(of: chars[i]) else {
                throw BarcodeError.invalidCharacter(symbology: "CODE93", character: chars[i])
            }
            let idx = code93Chars.distance(from: code93Chars.startIndex, to: value)
            var posWeight = chars.count - i
            if posWeight > weight {
                posWeight = ((posWeight - 1) % weight) + 1
            }
            total += idx * posWeight
        }
        return total % 47
    }

    public func encode(_ code: String) throws -> [Int] {
        if code.isEmpty {
            throw BarcodeError.emptyString
        }

        let upper = code.uppercased()

        for ch in upper {
            if code93Chars.firstIndex(of: ch) == nil {
                throw BarcodeError.invalidCharacter(symbology: "CODE93", character: ch)
            }
        }

        let checkC = try modulus47(upper, weight: 20)
        let codeWithC = upper + String(Array(code93Chars)[checkC])

        let checkK = try modulus47(codeWithC, weight: 15)

        var result = [Int]()

        // Start pattern
        for ch in code93Start {
            result.append(Int(String(ch))!)
        }

        // Data
        let charsArr = Array(code93Chars)
        for ch in upper {
            let index = code93Chars.distance(from: code93Chars.startIndex, to: code93Chars.firstIndex(of: ch)!)
            for p in code93Patterns[index] {
                result.append(Int(String(p))!)
            }
        }

        // Check digit C
        for p in code93Patterns[checkC] {
            result.append(Int(String(p))!)
        }

        // Check digit K
        for p in code93Patterns[checkK] {
            result.append(Int(String(p))!)
        }

        // Stop pattern
        for ch in code93Stop {
            result.append(Int(String(ch))!)
        }

        return result
    }

    public func draw(code: String, width: Int, height: Int) throws {
        try self.draw(code, width: width, height: height, encoder: self)
    }
}
