import Foundation

private let code39Chars = "1234567890ABCDEFGHIJKLMNOPQRSTUVWXYZ-. *$/+%"
private let code39Ptn: [Int] = [
    0x121, 0x061, 0x160, 0x031, 0x130, 0x070, 0x025, 0x124, 0x064, 0x034,
    0x109, 0x049, 0x148, 0x019, 0x118, 0x058, 0x00d, 0x10c, 0x04c, 0x01c,
    0x103, 0x043, 0x142, 0x013, 0x112, 0x052, 0x007, 0x106, 0x046, 0x016,
    0x181, 0x0c1, 0x1c0, 0x091, 0x190, 0x0d0, 0x085, 0x184, 0x0c4, 0x094,
    0x0a8, 0x0a2, 0x08a, 0x02a,
]

/// Code39 barcode encoder.
public class Code39: BarcodeBase1D, Encoder1D {
    public var showStartStop: Bool = true

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    public func encode(_ code: String) throws -> [Int] {
        if code.isEmpty {
            throw BarcodeError.emptyString
        }

        var validCode = ""
        for ch in code {
            let upper = String(ch).uppercased()
            if upper == "*" {
                throw BarcodeError.invalidCharacter(symbology: "CODE39", character: ch)
            }
            guard let pos = code39Chars.firstIndex(of: Character(upper)),
                  code39Chars.distance(from: code39Chars.startIndex, to: pos) != 39 else {
                throw BarcodeError.invalidCharacter(symbology: "CODE39", character: ch)
            }
            validCode += upper
        }

        let fullCode = "*" + validCode + "*"

        var result = [Int]()
        for (i, ch) in fullCode.enumerated() {
            if i > 0 {
                result.append(1) // inter-character gap
            }
            guard let charPos = code39Chars.firstIndex(of: ch) else { continue }
            let charIndex = code39Chars.distance(from: code39Chars.startIndex, to: charPos)
            let bitPattern = code39Ptn[charIndex]

            // 9-bit pattern: MSB first, 1=wide(3), 0=narrow(1)
            var mask = 0x100
            for _ in 0..<9 {
                if (bitPattern & mask) != 0 {
                    result.append(3)
                } else {
                    result.append(1)
                }
                mask >>= 1
            }
        }
        return result
    }

    /// Draws the Code39 barcode.
    public func draw(code: String, width: Int, height: Int) throws {
        try self.draw(code, width: width, height: height, encoder: self)
    }
}
