import Foundation

/// Code128 mode constants.
public let Code128Auto  = 0
public let Code128CodeA = 1
public let Code128CodeB = 2
public let Code128CodeC = 3

let code128Patterns: [String] = [
    "212222", "222122", "222221", "121223", "121322", "131222", "122213",
    "122312", "132212", "221213", "221312", "231212", "112232", "122132",
    "122231", "113222", "123122", "123221", "223211", "221132", "221231",
    "213212", "223112", "312131", "311222", "321122", "321221", "312212",
    "322112", "322211", "212123", "212321", "232121", "111323", "131123",
    "131321", "112313", "132113", "132311", "211313", "231113", "231311",
    "112133", "112331", "132131", "113123", "113321", "133121", "313121",
    "211331", "231131", "213113", "213311", "213131", "311123", "311321",
    "331121", "312113", "312311", "332111", "314111", "221411", "431111",
    "111224", "111422", "121124", "121421", "141122", "141221", "112214",
    "112412", "122114", "122411", "142112", "142211", "241211", "221114",
    "413111", "241112", "134111", "111242", "121142", "121241", "114212",
    "124112", "124211", "411212", "421112", "421211", "212141", "214121",
    "412121", "111143", "111341", "131141", "114113", "114311", "411113",
    "411311", "113141", "114131", "311141", "411131",
    "211412", "211214", "211232", "2331112", // 103-106: Start A/B/C, Stop
]

private let dpInf = 999999999

// MARK: - DP Encoding Types

private let dpSetA = 0
private let dpSetB = 1
private let dpSetC = 2

private struct DPResult {
    let cost: Int
    let enc: [Int]
}

// MARK: - Character Index Helpers

private func code128AIndex(_ c: UInt8) -> Int {
    if c <= 95 {
        return Int(c)
    }
    return -1
}

private func code128BIndex(_ c: UInt8) -> Int {
    if c < 32 || c > 126 {
        return -1
    }
    return Int(c) - 32
}

private func code128CIndex(_ bytes: [UInt8], _ i: Int) -> Int {
    if i + 1 >= bytes.count {
        return -1
    }
    let c1 = bytes[i]
    let c2 = bytes[i + 1]
    if c1 < UInt8(ascii: "0") || c1 > UInt8(ascii: "9") ||
       c2 < UInt8(ascii: "0") || c2 > UInt8(ascii: "9") {
        return -1
    }
    return Int(c1 - UInt8(ascii: "0")) * 10 + Int(c2 - UInt8(ascii: "0"))
}

// MARK: - DP Solver

private func solveDP(_ bytes: [UInt8], _ i: Int, _ iset: Int,
                     _ memo: inout [Int: DPResult]) throws -> DPResult {
    if i >= bytes.count {
        return DPResult(cost: 0, enc: [])
    }
    let key = (i << 2) + iset
    if let r = memo[key] {
        return r
    }

    var bestCost = dpInf
    var bestEnc = [Int]()
    let c = bytes[i]

    // Stay in current set
    if iset == dpSetA {
        let idx = code128AIndex(c)
        if idx >= 0 {
            let nxt = try solveDP(bytes, i + 1, iset, &memo)
            let cost = 1 + nxt.cost
            if cost < bestCost {
                bestCost = cost
                bestEnc = [idx] + nxt.enc
            }
        }
    } else if iset == dpSetB {
        let idx = code128BIndex(c)
        if idx >= 0 {
            let nxt = try solveDP(bytes, i + 1, iset, &memo)
            let cost = 1 + nxt.cost
            if cost < bestCost {
                bestCost = cost
                bestEnc = [idx] + nxt.enc
            }
        }
    } else { // dpSetC
        let idx = code128CIndex(bytes, i)
        if idx >= 0 {
            let nxt = try solveDP(bytes, i + 2, iset, &memo)
            let cost = 1 + nxt.cost
            if cost < bestCost {
                bestCost = cost
                bestEnc = [idx] + nxt.enc
            }
        }
    }

    // Switch to different set
    let switchVals: [[Int]] = [
        [0, 100, 99],   // from A: to A(unused), to B=100, to C=99
        [101, 0, 99],   // from B: to A=101, to B(unused), to C=99
        [101, 100, 0],  // from C: to A=101, to B=100, to C(unused)
    ]

    for st in 0..<3 {
        if st == iset {
            continue
        }
        let sv = switchVals[iset][st]

        if st == dpSetA {
            let idx = code128AIndex(c)
            if idx >= 0 {
                let nxt = try solveDP(bytes, i + 1, st, &memo)
                let cost = 2 + nxt.cost
                if cost < bestCost {
                    bestCost = cost
                    bestEnc = [sv, idx] + nxt.enc
                }
            }
        } else if st == dpSetB {
            let idx = code128BIndex(c)
            if idx >= 0 {
                let nxt = try solveDP(bytes, i + 1, st, &memo)
                let cost = 2 + nxt.cost
                if cost < bestCost {
                    bestCost = cost
                    bestEnc = [sv, idx] + nxt.enc
                }
            }
        } else { // dpSetC
            let idx = code128CIndex(bytes, i)
            if idx >= 0 {
                let nxt = try solveDP(bytes, i + 2, st, &memo)
                let cost = 2 + nxt.cost
                if cost < bestCost {
                    bestCost = cost
                    bestEnc = [sv, idx] + nxt.enc
                }
            }
        }
    }

    if bestCost == dpInf {
        throw BarcodeError.encodingFailed("DP encode error at pos=\(i)")
    }

    let r = DPResult(cost: bestCost, enc: bestEnc)
    memo[key] = r
    return r
}

// MARK: - Pattern Expansion

private func expandCode128Pattern(_ vals: [Int]) throws -> [Int] {
    var result = [Int]()
    for v in vals {
        guard v >= 0 && v <= 106 else {
            throw BarcodeError.encodingFailed("invalid code value \(v)")
        }
        let ptn = code128Patterns[v]
        for ch in ptn {
            result.append(Int(String(ch))!)
        }
    }
    return result
}

// MARK: - Code128 Class

/// Code128 barcode encoder with AUTO/A/B/C mode support.
public class Code128: BarcodeBase1D, Encoder1D {
    public var codeMode: Int = Code128Auto

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    public func encode(_ code: String) throws -> [Int] {
        guard !code.isEmpty else {
            throw BarcodeError.emptyString
        }
        if codeMode == Code128Auto {
            return try encodeAuto(code)
        }
        return try encodeSingleSet(code, fixedSet: codeMode)
    }

    private func encodeSingleSet(_ s: String, fixedSet: Int) throws -> [Int] {
        let startVal = 102 + fixedSet // A=103, B=104, C=105
        var vals = [startVal]
        let bytes = Array(s.utf8)

        var i = 0
        while i < bytes.count {
            let ch = bytes[i]
            switch fixedSet {
            case Code128CodeC:
                let cc = code128CIndex(bytes, i)
                guard cc >= 0 else {
                    throw BarcodeError.encodingFailed("not 2-digit at pos=\(i)")
                }
                vals.append(cc)
                i += 2
            case Code128CodeA:
                let idx = code128AIndex(ch)
                guard idx >= 0 else {
                    throw BarcodeError.invalidInput("invalid char for CODE_A")
                }
                vals.append(idx)
                i += 1
            default: // Code128CodeB
                let idx = code128BIndex(ch)
                guard idx >= 0 else {
                    throw BarcodeError.invalidInput("invalid char for CODE_B")
                }
                vals.append(idx)
                i += 1
            }
        }

        // Checksum (Modulus 103)
        var checkSum = vals[0]
        for j in 1..<vals.count {
            checkSum += vals[j] * j
        }
        checkSum %= 103
        vals.append(checkSum)
        vals.append(106) // Stop

        return try expandCode128Pattern(vals)
    }

    private func encodeAuto(_ s: String) throws -> [Int] {
        let bytes = Array(s.utf8)

        struct Candidate {
            let startVal: Int
            let cost: Int
            let enc: [Int]
        }

        var candidates = [Candidate]()
        for iset in 0..<3 {
            let startVal = 103 + iset
            var memo = [Int: DPResult]()
            if let r = try? solveDP(bytes, 0, iset, &memo) {
                candidates.append(Candidate(startVal: startVal, cost: r.cost, enc: r.enc))
            }
        }

        guard !candidates.isEmpty else {
            throw BarcodeError.encodingFailed("no valid start code")
        }

        // Pick lowest cost
        var best = candidates[0]
        for c in candidates.dropFirst() {
            if 1 + c.cost < 1 + best.cost {
                best = c
            }
        }

        var encVals = [best.startVal] + best.enc

        // Checksum
        var checkSum = encVals[0]
        for j in 1..<encVals.count {
            checkSum += encVals[j] * j
        }
        let ck = checkSum % 103
        encVals.append(ck)
        encVals.append(106) // Stop

        return try expandCode128Pattern(encVals)
    }

    /// Draws the Code128 barcode.
    public func draw(code: String, width: Int, height: Int) throws {
        try self.draw(code, width: width, height: height, encoder: self)
    }
}
