import Foundation

/// Protocol for 1D barcode encoders.
public protocol Encoder1D {
    func encode(_ code: String) throws -> [Int]
}

/// BarcodeBase1D is the base for 1D barcode symbologies.
open class BarcodeBase1D: BarcodeBase {
    public var showText: Bool = true
    public var textEvenSpacing: Bool = true
    var textFontScale: Double = 1.0
    var textHorizontalSpacingScale: Double = 1.0
    var textVerticalOffsetScale: Double = 1.0
    var minLineWidth: Int = 1

    public override init(outputFormat: String) {
        super.init(outputFormat: outputFormat)
    }

    // MARK: - Text Properties

    public func setTextFontScale(_ s: Double) { textFontScale = s }
    public func getTextFontScale() -> Double { textFontScale }

    public func setTextVerticalOffsetScale(_ s: Double) { textVerticalOffsetScale = s }
    public func getTextVerticalOffsetScale() -> Double { textVerticalOffsetScale }

    public func setTextHorizontalSpacingScale(_ s: Double) { textHorizontalSpacingScale = s }
    public func getTextHorizontalSpacingScale() -> Double { textHorizontalSpacingScale }

    public func setMinLineWidth(_ w: Int) { minLineWidth = w }
    public func getMinLineWidth() -> Int { minLineWidth }

    // MARK: - Draw

    /// Renders a 1D barcode using the given encoder.
    public func draw(_ code: String, width: Int, height: Int, encoder: Encoder1D) throws {
        let pattern = try encoder.encode(code)
        if pattern.isEmpty {
            throw BarcodeError.encodingFailed("empty pattern")
        }
        if isSVGOutput() {
            try drawSVG1D(pattern, code: code, width: width, height: height)
        } else {
            try drawPNG1D(pattern, code: code, width: width, height: height)
        }
    }

    // MARK: - SVG Rendering

    func drawSVG1D(_ pattern: [Int], code: String, width: Int, height: Int) throws {
        let display = showText ? code : ""
        try drawSVGBars(pattern, width: width, height: height, text: display)
    }

    func drawSVGBars(_ pattern: [Int], width: Int, height: Int, text: String) throws {
        let totalUnits = pattern.reduce(0, +)
        if totalUnits <= 0 {
            throw BarcodeError.encodingFailed("pattern has zero total units")
        }

        let fontSize = max(8, Int(Double(height) * 0.15 * textFontScale))

        var textH = 0
        var displayText = text
        if !displayText.isEmpty {
            textH = Int(Double(fontSize) * 1.4)
        }
        var barH = height - textH
        if barH <= 0 {
            barH = height
            displayText = ""
        }

        svgBegin(width, height)
        svgRect(0, 0, Double(width), Double(height), backColor)

        let unitW = Double(width) / Double(totalUnits)
        var accum = 0.0
        var isBar = true
        for units in pattern {
            let x1 = (accum * unitW).rounded()
            accum += Double(units)
            let x2 = (accum * unitW).rounded()
            if isBar && x2 > x1 {
                svgRect(x1, 0, x2 - x1, Double(barH), foreColor)
            }
            isBar = !isBar
        }

        if !displayText.isEmpty {
            let textY = Double(barH) + 2
            if textEvenSpacing {
                svgTextEvenDistribution(displayText, x: 0, y: textY, width: Double(width),
                                       fontSize: fontSize, hScale: textHorizontalSpacingScale)
            } else {
                svgText(Double(width) / 2, textY, displayText, fontSize, foreColor, anchor: "middle")
            }
        }

        svgEnd()
    }

    /// Renders text with even character distribution in SVG.
    func svgTextEvenDistribution(_ text: String, x: Double, y: Double, width: Double,
                                  fontSize: Int, hScale: Double) {
        if text.isEmpty { return }
        let numChars = text.count

        var marginRatio = 0.05
        if width < 100 {
            marginRatio = 0.03
        } else if width > 400 {
            marginRatio = 0.07
        }

        let horizontalMargin = width * marginRatio * hScale
        let textWidth = width - 2 * horizontalMargin

        var charSpacing = textWidth
        if numChars > 1 {
            charSpacing = textWidth / Double(numChars - 1)
        }

        let minSpacing = Double(fontSize) / 2
        if charSpacing < minSpacing {
            charSpacing = minSpacing
        }

        var totalTextWidth = 0.0
        if numChars > 1 {
            totalTextWidth = charSpacing * Double(numChars - 1)
        }
        let startX = x + (width - totalTextWidth) / 2

        let chars = Array(text)
        for i in 0..<numChars {
            let charX = startX + Double(i) * charSpacing
            svgText(charX, y, String(chars[i]), fontSize, foreColor, anchor: "middle")
        }
    }
}
