import Foundation

/// Output format constants.
public let FORMAT_PNG  = "png"
public let FORMAT_JPEG = "jpeg"
public let FORMAT_SVG  = "svg"

/// BarcodeBase holds common state for all barcode types.
open class BarcodeBase {
    var foreColor: RGBA = .black
    var backColor: RGBA = .white
    var format: String
    var fitWidth: Bool = false
    var pxAdjBlack: Int = 0
    var pxAdjWhite: Int = 0
    var svgStream: String = ""
    var svgWidth: Int = 0
    var svgHeight: Int = 0
    var imageBuffer: [UInt8]?

    public init(outputFormat: String) {
        self.format = outputFormat
    }

    // MARK: - Color

    public func setForegroundColor(_ r: UInt8, _ g: UInt8, _ b: UInt8, _ a: UInt8 = 255) {
        foreColor = RGBA(r, g, b, a)
    }

    public func setBackgroundColor(_ r: UInt8, _ g: UInt8, _ b: UInt8, _ a: UInt8 = 255) {
        backColor = RGBA(r, g, b, a)
    }

    public func getForeColor() -> RGBA { foreColor }
    public func getBackColor() -> RGBA { backColor }

    // MARK: - Pixel Adjustment

    public func setPxAdjustBlack(_ adj: Int) { pxAdjBlack = adj }
    public func setPxAdjustWhite(_ adj: Int) { pxAdjWhite = adj }

    // MARK: - Fit Width

    public func setFitWidth(_ fit: Bool) { fitWidth = fit }
    public func getFitWidth() -> Bool { fitWidth }

    // MARK: - Output Format

    public func setOutputFormat(_ fmt: String) { format = fmt }
    public func getOutputFormat() -> String { format }
    func isSVGOutput() -> Bool { format == FORMAT_SVG }

    // MARK: - Output

    /// Returns the rendered image bytes (PNG/JPEG).
    public func getImageMemory() -> [UInt8]? {
        imageBuffer
    }

    /// Returns a data-URI string for PNG/JPEG output.
    public func getImageBase64() throws -> String {
        if isSVGOutput() {
            throw BarcodeError.wrongOutputMode("getImageBase64() is not available in SVG mode; use getSVG()")
        }
        guard let data = imageBuffer, !data.isEmpty else {
            return ""
        }
        let encoded = Data(data).base64EncodedString()
        let mime = format == FORMAT_JPEG ? "image/jpeg" : "image/png"
        return "data:\(mime);base64,\(encoded)"
    }

    /// Returns the SVG string.
    public func getSVG() throws -> String {
        if !isSVGOutput() {
            throw BarcodeError.wrongOutputMode("getSVG() requires SVG output mode")
        }
        return svgStream
    }

    // MARK: - SVG Helpers

    func svgBegin(_ width: Int, _ height: Int) {
        svgWidth = width
        svgHeight = height
        svgStream = ""
        svgStream += "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
        svgStream += "<svg xmlns=\"http://www.w3.org/2000/svg\" version=\"1.1\" "
        svgStream += "width=\"\(width)\" height=\"\(height)\" viewBox=\"0 0 \(width) \(height)\">\n"
    }

    func svgEnd() {
        if isTrialMode() {
            svgSampleOverlay(0, 0, svgWidth, svgHeight)
        }
        svgStream += "</svg>\n"
    }

    func svgSampleOverlay(_ x: Int, _ y: Int, _ width: Int, _ height: Int) {
        var fontSize = Int(Double(height) * 0.12)
        if fontSize < 8 { fontSize = 8 }
        if fontSize > 40 { fontSize = 40 }
        var margin = Int(Double(height) * 0.01)
        if margin < 2 { margin = 2 }
        let text = getTrialText()
        svgStream += "  <text x=\"\(x + margin)\" y=\"\(y + margin)\" "
        svgStream += "font-family=\"Arial, sans-serif\" font-size=\"\(fontSize)\" "
        svgStream += "font-weight=\"bold\" font-style=\"italic\" fill=\"\(colorToRGB(.red))\" "
        svgStream += "dominant-baseline=\"hanging\">\(text)</text>\n"
    }

    func svgRect(_ x: Double, _ y: Double, _ w: Double, _ h: Double, _ c: RGBA) {
        var opacity = ""
        if c.a < 255 {
            opacity = "\" fill-opacity=\"\(String(format: "%.2f", Double(c.a) / 255.0))"
        }
        svgStream += "  <rect x=\"\(String(format: "%.2f", x))\" y=\"\(String(format: "%.2f", y))\" "
        svgStream += "width=\"\(String(format: "%.2f", w))\" height=\"\(String(format: "%.2f", h))\" "
        svgStream += "fill=\"\(colorToRGB(c))\(opacity)\"/>\n"
    }

    func svgText(_ x: Double, _ y: Double, _ text: String, _ fontSize: Int, _ c: RGBA, anchor: String = "middle") {
        svgStream += "  <text x=\"\(String(format: "%.2f", x))\" y=\"\(String(format: "%.2f", y))\" "
        svgStream += "font-family=\"Arial, sans-serif\" font-size=\"\(fontSize)\" "
        svgStream += "fill=\"\(colorToRGB(c))\" text-anchor=\"\(anchor)\" "
        svgStream += "dominant-baseline=\"hanging\">\(text)</text>\n"
    }
}
