// Barcode.Swift - All-in-One
//
// Full 18-type barcode REST API with PNG, SVG output using Vapor + Leaf.
//
//   swift run
//   -> http://localhost:5761

import Vapor
import Leaf
import BarcodePao

// MARK: - Barcode Type Definition

struct BarcodeTypeDef: Content {
    let id: String
    let label: String
    let group: String
    let dim: String       // "1d", "2d", "postal"
    let defaultCode: String
    let w: Int
    let h: Int

    enum CodingKeys: String, CodingKey {
        case id, label, group, dim
        case defaultCode = "default"
        case w, h
    }
}

let barcodeTypes: [BarcodeTypeDef] = [
    // 2D
    BarcodeTypeDef(id: "QR",           label: "QR Code",              group: "2D Barcode", dim: "2d",     defaultCode: "https://www.pao.ac/", w: 200, h: 200),
    BarcodeTypeDef(id: "DataMatrix",   label: "DataMatrix",           group: "2D Barcode", dim: "2d",     defaultCode: "Hello DataMatrix",     w: 200, h: 200),
    BarcodeTypeDef(id: "PDF417",       label: "PDF417",               group: "2D Barcode", dim: "2d",     defaultCode: "Hello PDF417",         w: 200, h: 100),
    // Special
    BarcodeTypeDef(id: "GS1_128",      label: "GS1-128",             group: "Special",    dim: "1d",     defaultCode: "[01]04912345123459",   w: 400, h: 100),
    BarcodeTypeDef(id: "YubinCustomer",label: "Yubin Customer",       group: "Special",    dim: "postal", defaultCode: "1060032",              w: 400, h: 60),
    // 1D
    BarcodeTypeDef(id: "Code128",      label: "Code 128",             group: "1D Barcode", dim: "1d",     defaultCode: "Hello-2026",           w: 400, h: 100),
    BarcodeTypeDef(id: "Code39",       label: "Code 39",              group: "1D Barcode", dim: "1d",     defaultCode: "HELLO-123",            w: 400, h: 100),
    BarcodeTypeDef(id: "Code93",       label: "Code 93",              group: "1D Barcode", dim: "1d",     defaultCode: "CODE93",               w: 400, h: 100),
    BarcodeTypeDef(id: "NW7",          label: "NW-7 / Codabar",       group: "1D Barcode", dim: "1d",     defaultCode: "A123456B",             w: 400, h: 100),
    BarcodeTypeDef(id: "ITF",          label: "ITF",                   group: "1D Barcode", dim: "1d",     defaultCode: "123456",               w: 400, h: 100),
    BarcodeTypeDef(id: "Matrix2of5",   label: "Matrix 2 of 5",        group: "1D Barcode", dim: "1d",     defaultCode: "1234",                 w: 400, h: 100),
    BarcodeTypeDef(id: "NEC2of5",      label: "NEC 2 of 5",           group: "1D Barcode", dim: "1d",     defaultCode: "1234",                 w: 400, h: 100),
    // GS1 DataBar
    BarcodeTypeDef(id: "GS1DataBar14",       label: "GS1 DataBar 14",       group: "GS1 DataBar", dim: "1d", defaultCode: "0123456789012",            w: 300, h: 80),
    BarcodeTypeDef(id: "GS1DataBarLimited",  label: "GS1 DataBar Limited",  group: "GS1 DataBar", dim: "1d", defaultCode: "0123456789012",            w: 300, h: 80),
    BarcodeTypeDef(id: "GS1DataBarExpanded", label: "GS1 DataBar Expanded", group: "GS1 DataBar", dim: "1d", defaultCode: "[01]90012345678908",       w: 400, h: 80),
    // JAN / UPC
    BarcodeTypeDef(id: "JAN13",  label: "JAN-13 / EAN-13", group: "JAN / UPC", dim: "1d", defaultCode: "490123456789",  w: 300, h: 100),
    BarcodeTypeDef(id: "JAN8",   label: "JAN-8 / EAN-8",   group: "JAN / UPC", dim: "1d", defaultCode: "1234567",       w: 250, h: 100),
    BarcodeTypeDef(id: "UPCA",   label: "UPC-A",            group: "JAN / UPC", dim: "1d", defaultCode: "01234567890",   w: 300, h: 100),
    BarcodeTypeDef(id: "UPCE",   label: "UPC-E",            group: "JAN / UPC", dim: "1d", defaultCode: "0123456",       w: 250, h: 100),
]

let barcodeMap: [String: BarcodeTypeDef] = {
    var m = [String: BarcodeTypeDef]()
    for t in barcodeTypes { m[t.id] = t }
    return m
}()

// MARK: - Request / Response Models

struct DrawRequest: Content {
    var type: String?
    var code: String?
    var format: String?         // "png", "jpeg", "svg"
    var width: Int?
    var height: Int?

    // 1D options
    var showText: String?       // "1" or "0"
    var evenSpacing: String?    // "1" or "0"

    // Color options
    var foreColor: String?      // hex without #
    var backColor: String?      // hex without #
    var transparentBg: String?  // "1" or "0"

    // QR options
    var qrErrorLevel: Int?
    var qrVersion: Int?

    // DataMatrix options
    var datamatrixSize: Int?

    // PDF417 options
    var pdf417ErrorLevel: Int?
    var pdf417Cols: Int?
    var pdf417Rows: Int?

    // GS1 DataBar 14 options
    var databar14Type: String?

    // GS1 DataBar Expanded options
    var databarExpandedType: String?
    var databarExpandedCols: Int?

    // JAN/UPC options
    var extendedGuard: String?  // "1" or "0"

    enum CodingKeys: String, CodingKey {
        case type, code, format, width, height
        case showText = "show_text"
        case evenSpacing = "even_spacing"
        case foreColor = "fore_color"
        case backColor = "back_color"
        case transparentBg = "transparent_bg"
        case qrErrorLevel = "qr_error_level"
        case qrVersion = "qr_version"
        case datamatrixSize = "datamatrix_size"
        case pdf417ErrorLevel = "pdf417_error_level"
        case pdf417Cols = "pdf417_cols"
        case pdf417Rows = "pdf417_rows"
        case databar14Type = "databar14_type"
        case databarExpandedType = "databar_expanded_type"
        case databarExpandedCols = "databar_expanded_cols"
        case extendedGuard = "extended_guard"
    }
}

struct DrawResponse: Content {
    var ok: Bool
    var base64: String?
    var svg: String?
    var error: String?
}

// MARK: - Color Parsing

func parseHexColor(_ hex: String) -> (UInt8, UInt8, UInt8) {
    guard hex.count == 6 else { return (0, 0, 0) }
    let chars = Array(hex)
    let r = UInt8(strtoul(String(chars[0...1]), nil, 16))
    let g = UInt8(strtoul(String(chars[2...3]), nil, 16))
    let b = UInt8(strtoul(String(chars[4...5]), nil, 16))
    return (r, g, b)
}

// MARK: - Barcode Creation

func applyColors1D(_ bc: BarcodeBase1D, foreColor: String?, backColor: String?, transparent: Bool) {
    if let fc = foreColor, !fc.isEmpty, fc != "000000" {
        let (r, g, b) = parseHexColor(fc)
        bc.setForegroundColor(r, g, b, 255)
    }
    if transparent {
        bc.setBackgroundColor(0, 0, 0, 0)
    } else if let bgc = backColor, !bgc.isEmpty, bgc != "FFFFFF" {
        let (r, g, b) = parseHexColor(bgc)
        bc.setBackgroundColor(r, g, b, 255)
    }
}

func applyColors2D(_ bc: BarcodeBase2D, foreColor: String?, backColor: String?, transparent: Bool) {
    if let fc = foreColor, !fc.isEmpty, fc != "000000" {
        let (r, g, b) = parseHexColor(fc)
        bc.setForegroundColor(r, g, b, 255)
    }
    if transparent {
        bc.setBackgroundColor(0, 0, 0, 0)
    } else if let bgc = backColor, !bgc.isEmpty, bgc != "FFFFFF" {
        let (r, g, b) = parseHexColor(bgc)
        bc.setBackgroundColor(r, g, b, 255)
    }
}

func applyCommon1D(_ bc: BarcodeBase1D, req: DrawRequest) {
    if req.showText == "0" {
        bc.showText = false
    }
    if req.evenSpacing == "0" {
        bc.textEvenSpacing = false
    }
}

/// Creates the appropriate barcode object, configures it, draws, and returns it.
func createAndDraw(typeID: String, code: String, outputFormat: String,
                   width: Int, height: Int, req: DrawRequest) throws -> BarcodeBase {
    let transparent = req.transparentBg == "1"
    let foreColor = req.foreColor
    let backColor = req.backColor

    switch typeID {
    case "QR":
        let bc = QRCode(outputFormat: outputFormat)
        bc.errorCorrectionLevel = req.qrErrorLevel ?? QR_ECC_M
        if let ver = req.qrVersion, ver > 0 {
            bc.version = ver
        }
        applyColors2D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, size: width)
        return bc

    case "DataMatrix":
        let bc = DataMatrix(outputFormat: outputFormat)
        if let sz = req.datamatrixSize, sz >= 0 {
            bc.setCodeSize(sz)
        }
        applyColors2D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, size: width)
        return bc

    case "PDF417":
        let bc = PDF417(outputFormat: outputFormat)
        if let el = req.pdf417ErrorLevel, el >= -1, el <= 8 {
            bc.setErrorCorrectionLevel(el)
        }
        if let cols = req.pdf417Cols, cols > 0 {
            bc.setColumns(cols)
        }
        if let rows = req.pdf417Rows, rows > 0 {
            bc.setRows(rows)
        }
        applyColors2D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "Code128":
        let bc = Code128(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "Code39":
        let bc = Code39(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "Code93":
        let bc = Code93(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "NW7":
        let bc = NW7(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "ITF":
        let bc = ITF(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "Matrix2of5":
        let bc = Matrix2of5(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "NEC2of5":
        let bc = NEC2of5(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "JAN13":
        let bc = JAN13(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        if req.extendedGuard == "0" {
            bc.extendedGuard = false
        }
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "JAN8":
        let bc = JAN8(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        if req.extendedGuard == "0" {
            bc.extendedGuard = false
        }
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "UPCA":
        let bc = UPCA(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        if req.extendedGuard == "0" {
            bc.extendedGuard = false
        }
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "UPCE":
        let bc = UPCE(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        if req.extendedGuard == "0" {
            bc.extendedGuard = false
        }
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "GS1_128":
        let bc = GS1128(outputFormat: outputFormat)
        applyCommon1D(bc, req: req)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "GS1DataBar14":
        var symType: SymbolType14 = .omnidirectional
        if let st = req.databar14Type {
            switch st {
            case "Stacked":     symType = .stacked
            case "StackedOmni": symType = .stackedOmnidirectional
            default: break
            }
        }
        let bc = GS1DataBar14(outputFormat: outputFormat, symbolType: symType)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "GS1DataBarLimited":
        let bc = GS1DataBarLimited(outputFormat: outputFormat)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "GS1DataBarExpanded":
        var symType: ExpandedSymbolType = .unstacked
        if req.databarExpandedType == "Stacked" {
            symType = .stacked
        }
        let numCols = req.databarExpandedCols ?? 2
        let bc = GS1DataBarExpanded(outputFormat: outputFormat, symbolType: symType, numColumns: numCols)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    case "YubinCustomer":
        let bc = YubinCustomer(outputFormat: outputFormat)
        applyColors1D(bc, foreColor: foreColor, backColor: backColor, transparent: transparent)
        try bc.draw(code: code, width: width, height: height)
        return bc

    default:
        throw Abort(.badRequest, reason: "Unknown barcode type: \(typeID)")
    }
}

// MARK: - Application

let app = try Application(.detect())
defer { app.shutdown() }

// Configure Leaf
app.views.use(.leaf)
app.leaf.configuration.rootDirectory = app.directory.resourcesDirectory + "Views"

// Configure server
app.http.server.configuration.hostname = "0.0.0.0"
app.http.server.configuration.port = 5761

// Disable trial mode for demo
setTrialMode(false)

// MARK: - Routes

// GET / -> Render index page
app.get { req -> EventLoopFuture<View> in
    let encoder = JSONEncoder()
    let jsonData = try encoder.encode(barcodeTypes)
    let jsonString = String(data: jsonData, encoding: .utf8) ?? "[]"
    return req.view.render("index", ["barcodeTypesJSON": jsonString])
}

// GET /api/types -> JSON list of barcode types
app.get("api", "types") { req -> [BarcodeTypeDef] in
    return barcodeTypes
}

// POST /draw-base64 -> Generate barcode as PNG/JPEG base64
app.post("draw-base64") { req -> DrawResponse in
    let params = try req.content.decode(DrawRequest.self)
    let typeID = params.type ?? "QR"
    let info = barcodeMap[typeID] ?? barcodeMap["QR"]!
    let code = (params.code?.isEmpty ?? true) ? info.defaultCode : params.code!
    let width = params.width ?? info.w
    let height = params.height ?? info.h

    do {
        let bc = try createAndDraw(
            typeID: typeID, code: code, outputFormat: FORMAT_PNG,
            width: width, height: height, req: params
        )
        let b64 = try bc.getImageBase64()
        return DrawResponse(ok: true, base64: b64, svg: nil, error: nil)
    } catch {
        return DrawResponse(ok: false, base64: nil, svg: nil, error: "\(error)")
    }
}

// POST /draw-svg -> Generate barcode as SVG
app.post("draw-svg") { req -> DrawResponse in
    let params = try req.content.decode(DrawRequest.self)
    let typeID = params.type ?? "QR"
    let info = barcodeMap[typeID] ?? barcodeMap["QR"]!
    let code = (params.code?.isEmpty ?? true) ? info.defaultCode : params.code!
    let width = params.width ?? info.w
    let height = params.height ?? info.h

    do {
        let bc = try createAndDraw(
            typeID: typeID, code: code, outputFormat: FORMAT_SVG,
            width: width, height: height, req: params
        )
        let svg = try bc.getSVG()
        return DrawResponse(ok: true, base64: nil, svg: svg, error: nil)
    } catch {
        return DrawResponse(ok: false, base64: nil, svg: nil, error: "\(error)")
    }
}

// POST /draw -> Unified endpoint (format in request body)
app.post("draw") { req -> DrawResponse in
    let params = try req.content.decode(DrawRequest.self)
    let typeID = params.type ?? "QR"
    let info = barcodeMap[typeID] ?? barcodeMap["QR"]!
    let code = (params.code?.isEmpty ?? true) ? info.defaultCode : params.code!
    let width = params.width ?? info.w
    let height = params.height ?? info.h
    let fmt = params.format ?? "png"

    let outputFormat: String
    switch fmt.lowercased() {
    case "svg":  outputFormat = FORMAT_SVG
    case "jpeg": outputFormat = FORMAT_JPEG
    default:     outputFormat = FORMAT_PNG
    }

    do {
        let bc = try createAndDraw(
            typeID: typeID, code: code, outputFormat: outputFormat,
            width: width, height: height, req: params
        )

        if outputFormat == FORMAT_SVG {
            let svg = try bc.getSVG()
            return DrawResponse(ok: true, base64: nil, svg: svg, error: nil)
        } else {
            let b64 = try bc.getImageBase64()
            return DrawResponse(ok: true, base64: b64, svg: nil, error: nil)
        }
    } catch {
        return DrawResponse(ok: false, base64: nil, svg: nil, error: "\(error)")
    }
}

print("Barcode.Swift All-in-One")
print("-> http://localhost:5761")
try app.run()
