// Barcode.Rust (C++ WASM Engine) - Easy 2 Steps
//
// QR code generation in just 2 steps using axum + C++ WASM engine.
//
//   cargo run
//   -> http://localhost:5722

use axum::{
    extract::Form,
    response::Html,
    routing::{get, post},
    Json, Router,
};
use serde::{Deserialize, Serialize};
use barcode_pao_wasm::{QR, FORMAT_PNG, FORMAT_SVG};

const INDEX_HTML: &str = include_str!("../templates/index.html");

#[derive(Serialize)]
struct ApiResponse {
    ok: bool,
    #[serde(skip_serializing_if = "Option::is_none")]
    base64: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    svg: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    error: Option<String>,
}

impl ApiResponse {
    fn ok_base64(b64: String) -> Self {
        Self { ok: true, base64: Some(b64), svg: None, error: None }
    }
    fn ok_svg(svg: String) -> Self {
        Self { ok: true, base64: None, svg: Some(svg), error: None }
    }
    fn err(msg: String) -> Self {
        Self { ok: false, base64: None, svg: None, error: Some(msg) }
    }
}

#[derive(Deserialize)]
struct DrawForm {
    code: Option<String>,
}

async fn index() -> Html<&'static str> {
    Html(INDEX_HTML)
}

async fn draw_base64(Form(form): Form<DrawForm>) -> Json<ApiResponse> {
    let code = form.code.unwrap_or_else(|| "https://www.pao.ac/".to_string());
    if code.is_empty() {
        return Json(ApiResponse::err("code is empty".to_string()));
    }

    // Step 1: Create barcode
    let qr = QR::new(FORMAT_PNG);
    // Step 2: Draw!
    match qr.draw(&code, 300) {
        Ok(b64) => Json(ApiResponse::ok_base64(b64)),
        Err(e) => Json(ApiResponse::err(e)),
    }
}

async fn draw_svg(Form(form): Form<DrawForm>) -> Json<ApiResponse> {
    let code = form.code.unwrap_or_else(|| "https://www.pao.ac/".to_string());
    if code.is_empty() {
        return Json(ApiResponse::err("code is empty".to_string()));
    }

    // Step 1: Create barcode (SVG mode)
    let qr = QR::new(FORMAT_SVG);
    // Step 2: Draw!
    match qr.draw(&code, 300) {
        Ok(svg) => Json(ApiResponse::ok_svg(svg)),
        Err(e) => Json(ApiResponse::err(e)),
    }
}

#[tokio::main]
async fn main() {
    let app = Router::new()
        .route("/", get(index))
        .route("/draw-base64", post(draw_base64))
        .route("/draw-svg", post(draw_svg));

    let addr = "0.0.0.0:5722";
    println!("Barcode.Rust (C++ WASM Engine) Easy 2 Steps");
    println!("-> http://localhost:5722");

    let listener = tokio::net::TcpListener::bind(addr).await.unwrap();
    axum::serve(listener, app).await.unwrap();
}
