use serde::{Deserialize, Serialize};
use serde_json::Value;
use std::collections::HashMap;
use std::path::PathBuf;
use std::process::Command;
use std::sync::OnceLock;

/// Output format constants.
pub const FORMAT_PNG: &str = "png";
pub const FORMAT_JPEG: &str = "jpg";
pub const FORMAT_SVG: &str = "svg";

// ─── Node.js detection ──────────────────────────────────────────────────────

static NODE_PATH: OnceLock<Result<String, String>> = OnceLock::new();

fn find_node() -> Result<String, String> {
    NODE_PATH
        .get_or_init(|| {
            for cmd in &["node", "nodejs"] {
                if let Ok(output) = Command::new(cmd).arg("--version").output() {
                    if output.status.success() {
                        return Ok(cmd.to_string());
                    }
                }
            }
            Err("Node.js is required for WASM barcode generation. \
                 Please install Node.js from https://nodejs.org/"
                .to_string())
        })
        .clone()
}

// ─── WASM directory resolution ──────────────────────────────────────────────

fn get_wasm_dir() -> PathBuf {
    // 1. Try CARGO_MANIFEST_DIR (development time)
    if let Ok(manifest) = std::env::var("CARGO_MANIFEST_DIR") {
        let dir = PathBuf::from(&manifest).join("wasm");
        if dir.is_dir() {
            return dir;
        }
    }

    // 2. Try relative to this source file (development time)
    let src_dir = PathBuf::from(file!());
    if let Some(parent) = src_dir.parent().and_then(|p| p.parent()) {
        let dir = parent.join("wasm");
        if dir.is_dir() {
            return dir;
        }
    }

    // 3. Try relative to executable (distribution)
    if let Ok(exe) = std::env::current_exe() {
        if let Some(exe_dir) = exe.parent() {
            let dir = exe_dir.join("barcode_pao_wasm").join("wasm");
            if dir.is_dir() {
                return dir;
            }
            let dir = exe_dir.join("wasm");
            if dir.is_dir() {
                return dir;
            }
        }
    }

    // 4. Fallback
    PathBuf::from("wasm")
}

// ─── WASM call protocol ─────────────────────────────────────────────────────

#[derive(Serialize)]
struct WasmRequest {
    #[serde(rename = "className")]
    class_name: String,
    method: String,
    #[serde(rename = "methodArgs")]
    method_args: Vec<Value>,
    settings: HashMap<String, Value>,
}

#[derive(Deserialize)]
struct WasmResponse {
    result: Option<Value>,
    error: Option<String>,
}

fn call_wasm(
    class_name: &str,
    method: &str,
    args: Vec<Value>,
    settings: &HashMap<String, Value>,
) -> Result<String, String> {
    let node = find_node()?;
    let wasm_dir = get_wasm_dir();
    let script_path = wasm_dir.join("_barcode_runner.mjs");

    let req = WasmRequest {
        class_name: class_name.to_string(),
        method: method.to_string(),
        method_args: args,
        settings: settings.clone(),
    };
    let req_json = serde_json::to_string(&req).map_err(|e| format!("JSON serialize error: {e}"))?;

    let output = Command::new(&node)
        .arg(script_path.to_string_lossy().as_ref())
        .arg(&req_json)
        .current_dir(&wasm_dir)
        .output()
        .map_err(|e| format!("Failed to execute Node.js: {e}"))?;

    let stdout = String::from_utf8_lossy(&output.stdout);

    if !output.status.success() {
        let stderr = String::from_utf8_lossy(&output.stderr);
        // Try JSON error from stdout
        if let Ok(resp) = serde_json::from_str::<WasmResponse>(stdout.trim()) {
            if let Some(err) = resp.error {
                return Err(format!("WASM error: {err}"));
            }
        }
        if !stderr.is_empty() {
            return Err(format!("WASM execution failed: {}", stderr.trim()));
        }
        return Err("WASM execution failed with unknown error".to_string());
    }

    // Find JSON line in output
    let json_line = stdout
        .lines()
        .filter(|l| {
            let l = l.trim();
            l.starts_with('{') && l.ends_with('}')
        })
        .last()
        .ok_or("No JSON response from WASM output")?;

    let resp: WasmResponse =
        serde_json::from_str(json_line.trim()).map_err(|e| format!("Invalid JSON: {e}"))?;

    if let Some(err) = resp.error {
        return Err(format!("WASM error: {err}"));
    }

    match resp.result {
        Some(Value::String(s)) => Ok(s),
        Some(v) => Ok(v.to_string()),
        None => Err("Empty result from WASM".to_string()),
    }
}

// ═════════════════════════════════════════════════════════════════════════════
// Base types
// ═════════════════════════════════════════════════════════════════════════════

/// Common base for all barcode types.
pub struct BarcodeWasmBase {
    class_name: String,
    settings: HashMap<String, Value>,
}

impl BarcodeWasmBase {
    fn new(class_name: &str) -> Self {
        let mut settings = HashMap::new();
        settings.insert("outputFormat".to_string(), Value::String(FORMAT_PNG.to_string()));
        Self {
            class_name: class_name.to_string(),
            settings,
        }
    }

    /// Set output format (png, jpg, svg).
    pub fn set_output_format(&mut self, format: &str) {
        self.settings
            .insert("outputFormat".to_string(), Value::String(format.to_string()));
    }

    /// Set foreground color (RGBA).
    pub fn set_foreground_color(&mut self, r: u8, g: u8, b: u8, a: u8) {
        self.settings.insert(
            "foregroundColor".to_string(),
            serde_json::json!([r, g, b, a]),
        );
    }

    /// Set background color (RGBA).
    pub fn set_background_color(&mut self, r: u8, g: u8, b: u8, a: u8) {
        self.settings.insert(
            "backgroundColor".to_string(),
            serde_json::json!([r, g, b, a]),
        );
    }

    fn call(&self, method: &str, args: Vec<Value>) -> Result<String, String> {
        call_wasm(&self.class_name, method, args, &self.settings)
    }
}

/// Base for 1D barcodes.
pub struct Barcode1DBase {
    pub base: BarcodeWasmBase,
}

impl Barcode1DBase {
    fn new(class_name: &str) -> Self {
        Self {
            base: BarcodeWasmBase::new(class_name),
        }
    }

    pub fn set_output_format(&mut self, format: &str) {
        self.base.set_output_format(format);
    }

    pub fn set_foreground_color(&mut self, r: u8, g: u8, b: u8, a: u8) {
        self.base.set_foreground_color(r, g, b, a);
    }

    pub fn set_background_color(&mut self, r: u8, g: u8, b: u8, a: u8) {
        self.base.set_background_color(r, g, b, a);
    }

    pub fn set_show_text(&mut self, show: bool) {
        self.base.settings.insert("showText".to_string(), Value::Bool(show));
    }

    pub fn set_text_gap(&mut self, gap: f64) {
        self.base.settings.insert("textGap".to_string(), serde_json::json!(gap));
    }

    pub fn set_text_font_scale(&mut self, scale: f64) {
        self.base.settings.insert("textFontScale".to_string(), serde_json::json!(scale));
    }

    pub fn set_text_even_spacing(&mut self, even: bool) {
        self.base.settings.insert("textEvenSpacing".to_string(), Value::Bool(even));
    }

    pub fn set_fit_width(&mut self, fit: bool) {
        self.base.settings.insert("fitWidth".to_string(), Value::Bool(fit));
    }

    pub fn set_px_adjust_black(&mut self, adj: i32) {
        self.base.settings.insert("pxAdjustBlack".to_string(), serde_json::json!(adj));
    }

    pub fn set_px_adjust_white(&mut self, adj: i32) {
        self.base.settings.insert("pxAdjustWhite".to_string(), serde_json::json!(adj));
    }

    /// Generate a 1D barcode. Returns Base64 or SVG string.
    pub fn draw(&self, code: &str, width: i32, height: i32) -> Result<String, String> {
        self.base.call(
            "draw",
            vec![
                Value::String(code.to_string()),
                serde_json::json!(width),
                serde_json::json!(height),
            ],
        )
    }
}

/// Base for 2D barcodes.
pub struct Barcode2DBase {
    pub base: BarcodeWasmBase,
}

impl Barcode2DBase {
    fn new(class_name: &str) -> Self {
        Self {
            base: BarcodeWasmBase::new(class_name),
        }
    }

    pub fn set_output_format(&mut self, format: &str) {
        self.base.set_output_format(format);
    }

    pub fn set_foreground_color(&mut self, r: u8, g: u8, b: u8, a: u8) {
        self.base.set_foreground_color(r, g, b, a);
    }

    pub fn set_background_color(&mut self, r: u8, g: u8, b: u8, a: u8) {
        self.base.set_background_color(r, g, b, a);
    }

    pub fn set_string_encoding(&mut self, enc: &str) {
        self.base.settings.insert(
            "stringEncoding".to_string(),
            Value::String(enc.to_string()),
        );
    }

    pub fn set_fit_width(&mut self, fit: bool) {
        self.base.settings.insert("fitWidth".to_string(), Value::Bool(fit));
    }

    /// Generate a 2D barcode. Returns Base64 or SVG string.
    pub fn draw(&self, code: &str, size: i32) -> Result<String, String> {
        self.base.call(
            "draw",
            vec![Value::String(code.to_string()), serde_json::json!(size)],
        )
    }
}

// ═════════════════════════════════════════════════════════════════════════════
// 1D Barcodes
// ═════════════════════════════════════════════════════════════════════════════

macro_rules! barcode_1d {
    ($name:ident, $class:expr) => {
        pub struct $name {
            pub base_1d: Barcode1DBase,
        }

        impl $name {
            pub fn new(output_format: &str) -> Self {
                let mut bc = Self {
                    base_1d: Barcode1DBase::new($class),
                };
                bc.base_1d.set_output_format(output_format);
                bc
            }

            pub fn draw(&self, code: &str, width: i32, height: i32) -> Result<String, String> {
                self.base_1d.draw(code, width, height)
            }
        }
    };
}

barcode_1d!(Code39, "Code39");
barcode_1d!(Code93, "Code93");
barcode_1d!(Code128, "Code128");
barcode_1d!(GS1128, "GS1_128");
barcode_1d!(NW7, "NW7");
barcode_1d!(ITF, "ITF");
barcode_1d!(Matrix2of5, "Matrix2of5");
barcode_1d!(NEC2of5, "NEC2of5");
barcode_1d!(JAN8, "Jan8");
barcode_1d!(JAN13, "Jan13");
barcode_1d!(UPCA, "UPC_A");
barcode_1d!(UPCE, "UPC_E");
barcode_1d!(GS1DataBar14, "GS1DataBar14");
barcode_1d!(GS1DataBarLimited, "GS1DataBarLimited");
barcode_1d!(GS1DataBarExpanded, "GS1DataBarExpanded");

impl Code39 {
    pub fn set_show_start_stop(&mut self, show: bool) {
        self.base_1d.base.settings.insert("showStartStop".to_string(), Value::Bool(show));
    }
}

impl Code128 {
    pub fn set_code_mode(&mut self, mode: &str) {
        self.base_1d.base.settings.insert("codeMode".to_string(), Value::String(mode.to_string()));
    }
}

impl NW7 {
    pub fn set_show_start_stop(&mut self, show: bool) {
        self.base_1d.base.settings.insert("showStartStop".to_string(), Value::Bool(show));
    }
}

impl JAN8 {
    pub fn set_extended_guard(&mut self, ext: bool) {
        self.base_1d.base.settings.insert("extendedGuard".to_string(), Value::Bool(ext));
    }
}

impl JAN13 {
    pub fn set_extended_guard(&mut self, ext: bool) {
        self.base_1d.base.settings.insert("extendedGuard".to_string(), Value::Bool(ext));
    }
}

impl UPCA {
    pub fn set_extended_guard(&mut self, ext: bool) {
        self.base_1d.base.settings.insert("extendedGuard".to_string(), Value::Bool(ext));
    }
}

impl UPCE {
    pub fn set_extended_guard(&mut self, ext: bool) {
        self.base_1d.base.settings.insert("extendedGuard".to_string(), Value::Bool(ext));
    }
}

impl GS1DataBar14 {
    pub fn set_symbol_type(&mut self, sym: &str) {
        self.base_1d.base.settings.insert("symbolType".to_string(), Value::String(sym.to_string()));
    }
}

impl GS1DataBarExpanded {
    pub fn set_symbol_type(&mut self, sym: &str) {
        self.base_1d.base.settings.insert("symbolType".to_string(), Value::String(sym.to_string()));
    }

    pub fn set_no_of_columns(&mut self, cols: i32) {
        self.base_1d.base.settings.insert("noOfColumns".to_string(), serde_json::json!(cols));
    }
}

// ═════════════════════════════════════════════════════════════════════════════
// Special Barcodes
// ═════════════════════════════════════════════════════════════════════════════

/// Japanese postal customer barcode generator.
pub struct YubinCustomer {
    pub base_1d: Barcode1DBase,
}

impl YubinCustomer {
    pub fn new(output_format: &str) -> Self {
        let mut bc = Self {
            base_1d: Barcode1DBase::new("YubinCustomer"),
        };
        bc.base_1d.set_output_format(output_format);
        bc
    }

    /// Draw with auto-calculated width.
    pub fn draw(&self, code: &str, height: i32) -> Result<String, String> {
        self.base_1d.base.call(
            "draw",
            vec![Value::String(code.to_string()), serde_json::json!(height)],
        )
    }

    /// Draw with explicit width.
    pub fn draw_with_width(&self, code: &str, width: i32, height: i32) -> Result<String, String> {
        self.base_1d.base.call(
            "drawWithWidth",
            vec![
                Value::String(code.to_string()),
                serde_json::json!(width),
                serde_json::json!(height),
            ],
        )
    }
}

// ═════════════════════════════════════════════════════════════════════════════
// 2D Barcodes
// ═════════════════════════════════════════════════════════════════════════════

/// QR code generator.
pub struct QR {
    pub base_2d: Barcode2DBase,
}

impl QR {
    pub fn new(output_format: &str) -> Self {
        let mut bc = Self {
            base_2d: Barcode2DBase::new("QR"),
        };
        bc.base_2d.set_output_format(output_format);
        bc
    }

    pub fn set_error_correction_level(&mut self, level: &str) {
        self.base_2d.base.settings.insert(
            "errorCorrectionLevel".to_string(),
            Value::String(level.to_string()),
        );
    }

    pub fn set_version(&mut self, version: i32) {
        self.base_2d.base.settings.insert("version".to_string(), serde_json::json!(version));
    }

    pub fn set_encode_mode(&mut self, mode: &str) {
        self.base_2d.base.settings.insert(
            "encodeMode".to_string(),
            Value::String(mode.to_string()),
        );
    }

    pub fn draw(&self, code: &str, size: i32) -> Result<String, String> {
        self.base_2d.draw(code, size)
    }
}

/// DataMatrix barcode generator.
pub struct DataMatrix {
    pub base_2d: Barcode2DBase,
}

impl DataMatrix {
    pub fn new(output_format: &str) -> Self {
        let mut bc = Self {
            base_2d: Barcode2DBase::new("DataMatrix"),
        };
        bc.base_2d.set_output_format(output_format);
        bc
    }

    pub fn set_code_size(&mut self, size: &str) {
        self.base_2d.base.settings.insert(
            "codeSize".to_string(),
            Value::String(size.to_string()),
        );
    }

    pub fn set_encode_scheme(&mut self, scheme: &str) {
        self.base_2d.base.settings.insert(
            "encodeScheme".to_string(),
            Value::String(scheme.to_string()),
        );
    }

    pub fn draw(&self, code: &str, size: i32) -> Result<String, String> {
        self.base_2d.draw(code, size)
    }
}

/// PDF417 barcode generator.
pub struct PDF417 {
    pub base_2d: Barcode2DBase,
}

impl PDF417 {
    pub fn new(output_format: &str) -> Self {
        let mut bc = Self {
            base_2d: Barcode2DBase::new("PDF417"),
        };
        bc.base_2d.set_output_format(output_format);
        bc
    }

    pub fn set_error_level(&mut self, level: i32) {
        self.base_2d.base.settings.insert("errorLevel".to_string(), serde_json::json!(level));
    }

    pub fn set_columns(&mut self, cols: i32) {
        self.base_2d.base.settings.insert("columns".to_string(), serde_json::json!(cols));
    }

    pub fn set_rows(&mut self, rows: i32) {
        self.base_2d.base.settings.insert("rows".to_string(), serde_json::json!(rows));
    }

    pub fn set_aspect_ratio(&mut self, ratio: f64) {
        self.base_2d.base.settings.insert("aspectRatio".to_string(), serde_json::json!(ratio));
    }

    pub fn set_y_height(&mut self, y_height: i32) {
        self.base_2d.base.settings.insert("yHeight".to_string(), serde_json::json!(y_height));
    }

    /// Draw PDF417 with width × height.
    pub fn draw(&self, code: &str, width: i32, height: i32) -> Result<String, String> {
        self.base_2d.base.call(
            "draw",
            vec![
                Value::String(code.to_string()),
                serde_json::json!(width),
                serde_json::json!(height),
            ],
        )
    }
}

// ═════════════════════════════════════════════════════════════════════════════
// Product Info
// ═════════════════════════════════════════════════════════════════════════════

pub fn get_product_name() -> &'static str {
    "barcode-pao-wasm (Rust)"
}

pub fn get_version() -> &'static str {
    "1.0.0"
}

pub fn get_manufacturer() -> &'static str {
    "Pao"
}
