// Barcode.Rust (C++ WASM Engine) - All-in-One
//
// Full 18-type barcode REST API using axum + C++ WASM engine.
//
//   cargo run
//   -> http://localhost:5723

use axum::{
    extract::Form,
    response::{Html, Json},
    routing::{get, post},
    Router,
};
use serde::{Deserialize, Serialize};

use barcode_pao_wasm::{
    Code128, Code39, Code93, DataMatrix, GS1128, GS1DataBar14,
    GS1DataBarExpanded, GS1DataBarLimited, ITF, JAN13, JAN8, Matrix2of5, NEC2of5, NW7, PDF417,
    QR, UPCA, UPCE, YubinCustomer, FORMAT_PNG, FORMAT_SVG,
};

const BARCODE_TYPES_JSON: &str = r#"[
{"id":"QR","label":"QR Code","group":"2D Barcode","dim":"2d","default":"https://www.pao.ac/","w":200,"h":200},
{"id":"DataMatrix","label":"DataMatrix","group":"2D Barcode","dim":"2d","default":"Hello Barcode.Rust","w":200,"h":200},
{"id":"PDF417","label":"PDF417","group":"2D Barcode","dim":"2d","default":"Hello Barcode.Rust","w":400,"h":120},
{"id":"Code39","label":"Code 39","group":"1D Barcode","dim":"1d","default":"ABC-1234","w":400,"h":100},
{"id":"Code93","label":"Code 93","group":"1D Barcode","dim":"1d","default":"ABC-1234","w":400,"h":100},
{"id":"Code128","label":"Code 128","group":"1D Barcode","dim":"1d","default":"Hello-2026","w":400,"h":100},
{"id":"NW7","label":"NW-7 (Codabar)","group":"1D Barcode","dim":"1d","default":"A12345B","w":400,"h":100},
{"id":"ITF","label":"ITF","group":"1D Barcode","dim":"1d","default":"1234567890","w":400,"h":100},
{"id":"Matrix2of5","label":"Matrix 2 of 5","group":"1D Barcode","dim":"1d","default":"1234567890","w":400,"h":100},
{"id":"NEC2of5","label":"NEC 2 of 5","group":"1D Barcode","dim":"1d","default":"1234567890","w":400,"h":100},
{"id":"GS1-128","label":"GS1-128","group":"1D Barcode","dim":"1d","default":"{01}04912345123459{10}ABC123","w":500,"h":100},
{"id":"JAN13","label":"JAN-13 (EAN-13)","group":"JAN / UPC","dim":"1d","default":"4901234567894","w":300,"h":100},
{"id":"JAN8","label":"JAN-8 (EAN-8)","group":"JAN / UPC","dim":"1d","default":"49123456","w":250,"h":100},
{"id":"UPC-A","label":"UPC-A","group":"JAN / UPC","dim":"1d","default":"01234567890","w":300,"h":100},
{"id":"UPC-E","label":"UPC-E","group":"JAN / UPC","dim":"1d","default":"0123456","w":250,"h":100},
{"id":"GS1DataBar14","label":"GS1 DataBar","group":"GS1 DataBar","dim":"1d","default":"0001234567890","w":400,"h":100},
{"id":"GS1DataBarLimited","label":"GS1 DataBar Limited","group":"GS1 DataBar","dim":"1d","default":"0001234567890","w":400,"h":100},
{"id":"GS1DataBarExpanded","label":"GS1 DataBar Expanded","group":"GS1 DataBar","dim":"1d","default":"{01}90012345678908{3103}000123","w":500,"h":100},
{"id":"YubinCustomer","label":"Yubin Customer","group":"Special","dim":"postal","default":"10408-0506001","w":400,"h":40}
]"#;

#[derive(Debug, Deserialize)]
struct DrawParams {
    #[serde(rename = "type_id")]
    type_id: String,
    #[serde(default)]
    code: String,
    #[serde(default = "default_width")]
    width: i32,
    #[serde(default = "default_height")]
    height: i32,
}

fn default_width() -> i32 { 300 }
fn default_height() -> i32 { 100 }

#[derive(Serialize)]
struct DrawResponse {
    ok: bool,
    #[serde(skip_serializing_if = "Option::is_none")]
    base64: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    svg: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    error: Option<String>,
}

impl DrawResponse {
    fn ok_base64(data: String) -> Self {
        Self { ok: true, base64: Some(data), svg: None, error: None }
    }
    fn ok_svg(data: String) -> Self {
        Self { ok: true, base64: None, svg: Some(data), error: None }
    }
    fn err(msg: String) -> Self {
        Self { ok: false, base64: None, svg: None, error: Some(msg) }
    }
}

fn create_and_draw(params: &DrawParams, output_format: &str) -> Result<String, String> {
    let code = &params.code;
    let w = params.width;
    let h = params.height;

    match params.type_id.as_str() {
        "QR" => QR::new(output_format).draw(code, w),
        "DataMatrix" => DataMatrix::new(output_format).draw(code, w),
        "PDF417" => PDF417::new(output_format).draw(code, w, h),

        "Code39" => Code39::new(output_format).draw(code, w, h),
        "Code93" => Code93::new(output_format).draw(code, w, h),
        "Code128" => Code128::new(output_format).draw(code, w, h),
        "NW7" => NW7::new(output_format).draw(code, w, h),
        "ITF" => ITF::new(output_format).draw(code, w, h),
        "Matrix2of5" => Matrix2of5::new(output_format).draw(code, w, h),
        "NEC2of5" => NEC2of5::new(output_format).draw(code, w, h),
        "GS1-128" => GS1128::new(output_format).draw(code, w, h),

        "JAN13" => JAN13::new(output_format).draw(code, w, h),
        "JAN8" => JAN8::new(output_format).draw(code, w, h),
        "UPC-A" => UPCA::new(output_format).draw(code, w, h),
        "UPC-E" => UPCE::new(output_format).draw(code, w, h),

        "GS1DataBar14" => GS1DataBar14::new(output_format).draw(code, w, h),
        "GS1DataBarLimited" => GS1DataBarLimited::new(output_format).draw(code, w, h),
        "GS1DataBarExpanded" => GS1DataBarExpanded::new(output_format).draw(code, w, h),

        "YubinCustomer" => YubinCustomer::new(output_format).draw(code, h),

        _ => Err(format!("Unknown barcode type: {}", params.type_id)),
    }
}

async fn index_handler() -> Html<String> {
    let template = include_str!("../templates/index.html");
    let html = template.replace("__BARCODE_TYPES_JSON__", BARCODE_TYPES_JSON);
    Html(html)
}

async fn api_types_handler() -> Json<serde_json::Value> {
    let v: serde_json::Value = serde_json::from_str(BARCODE_TYPES_JSON).unwrap();
    Json(v)
}

async fn draw_base64_handler(Form(params): Form<DrawParams>) -> Json<DrawResponse> {
    match create_and_draw(&params, FORMAT_PNG) {
        Ok(data) => Json(DrawResponse::ok_base64(data)),
        Err(e) => Json(DrawResponse::err(e)),
    }
}

async fn draw_svg_handler(Form(params): Form<DrawParams>) -> Json<DrawResponse> {
    match create_and_draw(&params, FORMAT_SVG) {
        Ok(data) => Json(DrawResponse::ok_svg(data)),
        Err(e) => Json(DrawResponse::err(e)),
    }
}

#[tokio::main]
async fn main() {
    let app = Router::new()
        .route("/", get(index_handler))
        .route("/api/types", get(api_types_handler))
        .route("/draw-base64", post(draw_base64_handler))
        .route("/draw-svg", post(draw_svg_handler));

    let addr = "0.0.0.0:5723";
    println!("Barcode.Rust (C++ WASM Engine) All-in-One");
    println!("-> http://localhost:5723");

    let listener = tokio::net::TcpListener::bind(addr).await.unwrap();
    axum::serve(listener, app).await.unwrap();
}
