// Barcode.Rust - Easy 2 Steps
//
// QR code generation in just 2 steps using axum.
// 4 output modes: PNG, SVG, PDF (printpdf), Canvas (tiny-skia)
//
//   cargo run
//   -> http://localhost:5720

use axum::{
    extract::{Form, Query},
    http::{header, StatusCode},
    response::{Html, IntoResponse},
    routing::{get, post},
    Json, Router,
};
use serde::{Deserialize, Serialize};
use barcode_pao::{QR, Code128, JAN13, FORMAT_PNG, FORMAT_SVG};

// ---------------------------------------------------------------------------
// Templates
// ---------------------------------------------------------------------------

const INDEX_HTML: &str = include_str!("../templates/index.html");

// ---------------------------------------------------------------------------
// JSON response
// ---------------------------------------------------------------------------

#[derive(Serialize)]
struct ApiResponse {
    ok: bool,
    #[serde(skip_serializing_if = "Option::is_none")]
    base64: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    svg: Option<String>,
    #[serde(skip_serializing_if = "Option::is_none")]
    error: Option<String>,
}

impl ApiResponse {
    fn ok_base64(b64: String) -> Self {
        Self { ok: true, base64: Some(b64), svg: None, error: None }
    }
    fn ok_svg(svg: String) -> Self {
        Self { ok: true, base64: None, svg: Some(svg), error: None }
    }
    fn err(msg: String) -> Self {
        Self { ok: false, base64: None, svg: None, error: Some(msg) }
    }
}

// ---------------------------------------------------------------------------
// Form input
// ---------------------------------------------------------------------------

#[derive(Deserialize)]
struct DrawForm {
    code: Option<String>,
}

#[derive(Deserialize)]
struct PdfQuery {
    code: Option<String>,
}

// ---------------------------------------------------------------------------
// Handlers
// ---------------------------------------------------------------------------

async fn index() -> Html<&'static str> {
    Html(INDEX_HTML)
}

async fn draw_base64(Form(form): Form<DrawForm>) -> Json<ApiResponse> {
    let code = form.code.unwrap_or_else(|| "https://www.pao.ac/".to_string());
    if code.is_empty() {
        return Json(ApiResponse::err("code is empty".to_string()));
    }

    // Step 1: Create barcode
    let mut qr = QR::new(FORMAT_PNG);
    // Step 2: Draw!
    if let Err(e) = qr.draw(&code, 300) {
        return Json(ApiResponse::err(e.to_string()));
    }

    match qr.base_2d.base.get_image_base64() {
        Ok(b64) => Json(ApiResponse::ok_base64(b64)),
        Err(e) => Json(ApiResponse::err(e.to_string())),
    }
}

async fn draw_svg(Form(form): Form<DrawForm>) -> Json<ApiResponse> {
    let code = form.code.unwrap_or_else(|| "https://www.pao.ac/".to_string());
    if code.is_empty() {
        return Json(ApiResponse::err("code is empty".to_string()));
    }

    // Step 1: Create barcode (SVG mode)
    let mut qr = QR::new(FORMAT_SVG);
    // Step 2: Draw!
    if let Err(e) = qr.draw(&code, 300) {
        return Json(ApiResponse::err(e.to_string()));
    }

    match qr.base_2d.base.get_svg() {
        Ok(svg) => Json(ApiResponse::ok_svg(svg)),
        Err(e) => Json(ApiResponse::err(e.to_string())),
    }
}

// Canvas: Draw barcodes onto a tiny-skia pixmap with graphics
async fn draw_canvas(Form(form): Form<DrawForm>) -> Json<ApiResponse> {
    let code = form.code.unwrap_or_else(|| "https://www.pao.ac/".to_string());
    if code.is_empty() {
        return Json(ApiResponse::err("code is empty".to_string()));
    }

    match draw_canvas_scene(&code) {
        Ok(b64) => Json(ApiResponse::ok_base64(b64)),
        Err(e) => Json(ApiResponse::err(e)),
    }
}

fn draw_canvas_scene(code: &str) -> Result<String, String> {
    use barcode_pao::renderers::tiny_skia::Renderer;
    use base64::Engine;
    use tiny_skia::*;

    let w = 900u32;
    let h = 520u32;
    let mut pixmap = Pixmap::new(w, h).ok_or("Failed to create pixmap")?;

    // --- Night sky gradient ---
    for y in 0..h {
        let t = y as f32 / h as f32;
        let r = (30.0 + 20.0 * t) as u8;
        let g = (60.0 + 40.0 * t) as u8;
        let b = (120.0 + 80.0 * t) as u8;
        let mut paint = Paint::default();
        paint.set_color_rgba8(r, g, b, 255);
        if let Some(rect) = Rect::from_xywh(0.0, y as f32, w as f32, 1.0) {
            pixmap.fill_rect(rect, &paint, Transform::identity(), None);
        }
    }

    // --- Stars ---
    let stars = [
        (120, 50, 3), (300, 80, 2), (500, 30, 4), (650, 90, 2), (780, 60, 3),
        (50, 120, 2), (200, 160, 3), (400, 140, 2), (600, 170, 4), (750, 130, 2),
        (100, 200, 3), (350, 220, 2), (550, 190, 3), (700, 210, 2), (830, 180, 3),
        (160, 280, 2), (280, 310, 3), (480, 270, 2), (620, 300, 4), (770, 290, 2),
        (80, 350, 3), (230, 380, 2), (430, 340, 3), (570, 370, 2), (820, 360, 3),
    ];
    for (sx, sy, sz) in &stars {
        let mut paint = Paint::default();
        paint.set_color_rgba8(255, 255, 200, 150);
        if let Some(path) = {
            let mut pb = PathBuilder::new();
            pb.push_circle(*sx as f32, *sy as f32, *sz as f32);
            pb.finish()
        } {
            pixmap.fill_path(&path, &paint, FillRule::Winding, Transform::identity(), None);
        }
    }

    // --- White card ---
    let card_x = 340.0f32;
    let card_y = 40.0f32;
    let card_w = 520.0f32;
    let card_h = 440.0f32;
    {
        let mut paint = Paint::default();
        paint.set_color_rgba8(255, 255, 255, 230);
        if let Some(rect) = Rect::from_xywh(card_x, card_y, card_w, card_h) {
            pixmap.fill_rect(rect, &paint, Transform::identity(), None);
        }
    }

    // --- Draw barcodes using renderer ---
    let renderer = Renderer::new();

    // QR Code
    let mut qr = QR::new(FORMAT_PNG);
    let qr_size = 220.0_f64;
    let qr_x = card_x as f64 + (card_w as f64 - qr_size) / 2.0;
    let qr_y = card_y as f64 + 75.0;
    renderer.draw_barcode(&mut pixmap, &mut qr, code, qr_x, qr_y, qr_size, qr_size)
        .map_err(|e| e.to_string())?;

    // Code128
    let mut c128 = Code128::new(FORMAT_PNG);
    c128.base_1d.show_text = true;
    let c128_w = 380.0_f64;
    let c128_h = 60.0_f64;
    let c128_x = card_x as f64 + (card_w as f64 - c128_w) / 2.0;
    let c128_y = card_y as f64 + 330.0;
    renderer.draw_barcode(&mut pixmap, &mut c128, "BARCODE-2026", c128_x, c128_y, c128_w, c128_h)
        .map_err(|e| e.to_string())?;

    // --- Encode to PNG base64 ---
    let png_data = pixmap.encode_png().map_err(|e| e.to_string())?;
    let b64 = base64::engine::general_purpose::STANDARD.encode(&png_data);
    Ok(format!("data:image/png;base64,{}", b64))
}

// PDF: Generate a DELIVERY NOTE invoice
async fn pdf_handler(Query(query): Query<PdfQuery>) -> impl IntoResponse {
    let code = query.code.unwrap_or_else(|| "https://www.pao.ac/".to_string());

    match generate_invoice_pdf(&code) {
        Ok(bytes) => {
            let headers = [
                (header::CONTENT_TYPE, "application/pdf"),
                (header::CONTENT_DISPOSITION, "inline; filename=invoice_sample.pdf"),
            ];
            (StatusCode::OK, headers, bytes).into_response()
        }
        Err(e) => {
            (StatusCode::INTERNAL_SERVER_ERROR, e).into_response()
        }
    }
}

fn mm(v: f64) -> printpdf::Mm {
    printpdf::Mm(v as f32)
}

fn generate_invoice_pdf(code: &str) -> Result<Vec<u8>, String> {
    use barcode_pao::renderers::printpdf::Renderer;
    use printpdf::*;
    use printpdf::path::{PaintMode, WindingOrder};

    let page_height: f64 = 297.0;
    let (doc, page1, layer1) = PdfDocument::new("Invoice", mm(210.0), mm(page_height), "Layer 1");
    let layer = doc.get_page(page1).get_layer(layer1);

    let renderer = Renderer::new();

    // Header background (blue)
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(
        30.0 / 255.0, 64.0 / 255.0, 175.0 / 255.0, None,
    )));
    let header_points = vec![
        (Point::new(mm(0.0), mm(page_height)), false),
        (Point::new(mm(210.0), mm(page_height)), false),
        (Point::new(mm(210.0), mm(page_height - 25.0)), false),
        (Point::new(mm(0.0), mm(page_height - 25.0)), false),
    ];
    layer.add_polygon(Polygon {
        rings: vec![header_points],
        mode: PaintMode::Fill,
        winding_order: WindingOrder::NonZero,
    });

    // Accent line
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(
        124.0 / 255.0, 58.0 / 255.0, 237.0 / 255.0, None,
    )));
    let accent_points = vec![
        (Point::new(mm(0.0), mm(page_height - 25.0)), false),
        (Point::new(mm(210.0), mm(page_height - 25.0)), false),
        (Point::new(mm(210.0), mm(page_height - 26.2)), false),
        (Point::new(mm(0.0), mm(page_height - 26.2)), false),
    ];
    layer.add_polygon(Polygon {
        rings: vec![accent_points],
        mode: PaintMode::Fill,
        winding_order: WindingOrder::NonZero,
    });

    // Header text
    let font = doc.add_builtin_font(BuiltinFont::HelveticaBold).map_err(|e| e.to_string())?;
    let font_regular = doc.add_builtin_font(BuiltinFont::Helvetica).map_err(|e| e.to_string())?;

    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(1.0, 1.0, 1.0, None)));
    layer.use_text("DELIVERY NOTE", 20.0, mm(15.0), mm(page_height - 15.0), &font);

    let today = chrono::Local::now().format("%Y-%m-%d").to_string();
    let inv_no = chrono::Local::now().format("INV-%Y-%m%d").to_string();

    layer.use_text("Pao@Office", 8.0, mm(160.0), mm(page_height - 8.0), &font_regular);
    layer.use_text(&format!("No: {}", inv_no), 8.0, mm(160.0), mm(page_height - 13.0), &font_regular);
    layer.use_text(&format!("Date: {}", today), 8.0, mm(160.0), mm(page_height - 18.0), &font_regular);
    layer.use_text("Tokyo, Japan", 8.0, mm(160.0), mm(page_height - 23.0), &font_regular);

    // Recipient
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(0.0, 0.0, 0.0, None)));
    layer.use_text("Sample Corporation", 10.0, mm(15.0), mm(page_height - 35.0), &font);

    // Barcode area background
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(
        241.0 / 255.0, 245.0 / 255.0, 249.0 / 255.0, None,
    )));
    let bc_area = vec![
        (Point::new(mm(12.0), mm(page_height - 42.0)), false),
        (Point::new(mm(198.0), mm(page_height - 42.0)), false),
        (Point::new(mm(198.0), mm(page_height - 85.0)), false),
        (Point::new(mm(12.0), mm(page_height - 85.0)), false),
    ];
    layer.add_polygon(Polygon {
        rings: vec![bc_area],
        mode: PaintMode::Fill,
        winding_order: WindingOrder::NonZero,
    });

    // Code128 barcode
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(
        71.0 / 255.0, 85.0 / 255.0, 105.0 / 255.0, None,
    )));
    layer.use_text("INVOICE BARCODE (Code128)", 7.0, mm(18.0), mm(page_height - 47.0), &font);

    let mut c128 = Code128::new(FORMAT_PNG);
    renderer.draw_barcode(&layer, &mut c128, &inv_no, 18.0, 50.0, 70.0, 14.0, page_height)
        .map_err(|e| e.to_string())?;

    // QR barcode
    layer.use_text("QR CODE", 7.0, mm(110.0), mm(page_height - 47.0), &font);
    let mut qr = QR::new(FORMAT_PNG);
    renderer.draw_barcode(&layer, &mut qr, code, 110.0, 50.0, 32.0, 32.0, page_height)
        .map_err(|e| e.to_string())?;

    // Data label
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(
        148.0 / 255.0, 163.0 / 255.0, 184.0 / 255.0, None,
    )));
    let label = if code.len() > 40 { &code[..40] } else { code };
    layer.use_text(&format!("Data: {}", label), 6.0, mm(110.0), mm(page_height - 83.0), &font_regular);

    // Table header
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(
        30.0 / 255.0, 64.0 / 255.0, 175.0 / 255.0, None,
    )));
    let table_y: f64 = page_height - 95.0;
    let th_points = vec![
        (Point::new(mm(15.0), mm(table_y)), false),
        (Point::new(mm(195.0), mm(table_y)), false),
        (Point::new(mm(195.0), mm(table_y - 7.0)), false),
        (Point::new(mm(15.0), mm(table_y - 7.0)), false),
    ];
    layer.add_polygon(Polygon {
        rings: vec![th_points],
        mode: PaintMode::Fill,
        winding_order: WindingOrder::NonZero,
    });
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(1.0, 1.0, 1.0, None)));
    layer.use_text("Item", 7.0, mm(18.0), mm(table_y - 5.0), &font);
    layer.use_text("Qty", 7.0, mm(115.0), mm(table_y - 5.0), &font);
    layer.use_text("Unit Price", 7.0, mm(140.0), mm(table_y - 5.0), &font);
    layer.use_text("Amount", 7.0, mm(175.0), mm(table_y - 5.0), &font);

    // Table rows
    let items = [
        ("Barcode.Rust Pure Rust Edition", "1", "\\33,000", "\\33,000"),
        ("Annual Support (3 years)", "1", "\\9,900", "\\9,900"),
    ];
    for (i, (name, qty, price, amount)) in items.iter().enumerate() {
        let ry: f64 = table_y - 7.0 - (i as f64 + 1.0) * 7.0;
        if i % 2 == 0 {
            layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(
                248.0 / 255.0, 250.0 / 255.0, 252.0 / 255.0, None,
            )));
            let row_pts = vec![
                (Point::new(mm(15.0), mm(ry + 7.0)), false),
                (Point::new(mm(195.0), mm(ry + 7.0)), false),
                (Point::new(mm(195.0), mm(ry)), false),
                (Point::new(mm(15.0), mm(ry)), false),
            ];
            layer.add_polygon(Polygon {
                rings: vec![row_pts],
                mode: PaintMode::Fill,
                winding_order: WindingOrder::NonZero,
            });
        }
        layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(0.0, 0.0, 0.0, None)));
        layer.use_text(*name, 7.0, mm(18.0), mm(ry + 2.0), &font_regular);
        layer.use_text(*qty, 7.0, mm(118.0), mm(ry + 2.0), &font_regular);
        layer.use_text(*price, 7.0, mm(142.0), mm(ry + 2.0), &font_regular);
        layer.use_text(*amount, 7.0, mm(177.0), mm(ry + 2.0), &font_regular);
    }

    // Total
    let total_y: f64 = table_y - 7.0 - (items.len() as f64 + 1.5) * 7.0;
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(
        30.0 / 255.0, 64.0 / 255.0, 175.0 / 255.0, None,
    )));
    layer.use_text("Total:  \\42,900", 11.0, mm(150.0), mm(total_y), &font);

    // JAN-13 sample
    let jan_y: f64 = total_y - 15.0;
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(
        71.0 / 255.0, 85.0 / 255.0, 105.0 / 255.0, None,
    )));
    layer.use_text("PRODUCT BARCODE (JAN-13)", 7.0, mm(15.0), mm(jan_y), &font);
    let mut jan = JAN13::new(FORMAT_PNG);
    renderer.draw_barcode(&layer, &mut jan, "490123456789", 15.0, page_height - jan_y + 5.0, 60.0, 15.0, page_height)
        .map_err(|e| e.to_string())?;

    // Footer
    layer.set_fill_color(printpdf::Color::Rgb(Rgb::new(
        148.0 / 255.0, 163.0 / 255.0, 184.0 / 255.0, None,
    )));
    layer.use_text("Generated by Barcode.Rust + PrintpdfRenderer", 6.0, mm(15.0), mm(12.0), &font_regular);
    layer.use_text("pao.ac", 6.0, mm(185.0), mm(12.0), &font_regular);

    doc.save_to_bytes().map_err(|e| e.to_string())
}

// ---------------------------------------------------------------------------
// Main
// ---------------------------------------------------------------------------

#[tokio::main]
async fn main() {
    let app = Router::new()
        .route("/", get(index))
        .route("/draw-base64", post(draw_base64))
        .route("/draw-svg", post(draw_svg))
        .route("/draw-canvas", post(draw_canvas))
        .route("/pdf", get(pdf_handler));

    let addr = "0.0.0.0:5720";
    println!("Barcode.Rust Easy 2 Steps");
    println!("-> http://localhost:5720");

    let listener = tokio::net::TcpListener::bind(addr).await.unwrap();
    axum::serve(listener, app).await.unwrap();
}
