//! WASM bindings for barcode generation via wasm-bindgen.
//!
//! Feature-gated behind the `wasm` feature flag.
//! Exposes JavaScript-callable functions for generating barcodes as Base64 PNG or SVG.

use wasm_bindgen::prelude::*;
use js_sys::{JSON, Reflect};

use crate::base::{BarcodeBase, FORMAT_PNG, FORMAT_SVG};
use crate::encoders::code39::Code39;
use crate::encoders::code93::Code93;
use crate::encoders::code128::Code128;
use crate::encoders::gs1_128::GS1128;
use crate::encoders::nw7::NW7;
use crate::encoders::itf::ITF;
use crate::encoders::matrix2of5::Matrix2of5;
use crate::encoders::nec2of5::NEC2of5;
use crate::encoders::jan8::JAN8;
use crate::encoders::jan13::JAN13;
use crate::encoders::upc_a::UPCA;
use crate::encoders::upc_e::UPCE;
use crate::encoders::gs1_databar_14::{GS1DataBar14, SymbolType14};
use crate::encoders::gs1_databar_limited::GS1DataBarLimited;
use crate::encoders::gs1_databar_expanded::{GS1DataBarExpanded, ExpandedSymbolType};
use crate::encoders::yubin_customer::YubinCustomer;
use crate::encoders::qr::QR;
use crate::encoders::datamatrix::DataMatrix;
use crate::encoders::pdf417::PDF417;

/// Options parsed from JSON for barcode generation.
struct DrawOptions {
    show_text: bool,
    even_spacing: bool,
    qr_error_level: i32,
    qr_version: i32,
    datamatrix_size: i32,
    pdf417_error_level: i32,
    pdf417_cols: i32,
    databar14_type: String,
    databar_exp_type: String,
    databar_exp_cols: i32,
}

impl DrawOptions {
    fn from_json(json: &str) -> Self {
        let js_val = JSON::parse(json).unwrap_or(JsValue::NULL);

        Self {
            show_text: get_bool(&js_val, "showText", true),
            even_spacing: get_bool(&js_val, "evenSpacing", false),
            qr_error_level: get_int(&js_val, "qrErrorLevel", 1),
            qr_version: get_int(&js_val, "qrVersion", 0),
            datamatrix_size: get_int(&js_val, "datamatrixSize", -1),
            pdf417_error_level: get_int(&js_val, "pdf417ErrorLevel", 2),
            pdf417_cols: get_int(&js_val, "pdf417Cols", 0),
            databar14_type: get_string(&js_val, "databar14Type", "omni"),
            databar_exp_type: get_string(&js_val, "databarExpType", "unstacked"),
            databar_exp_cols: get_int(&js_val, "databarExpCols", 2),
        }
    }
}

fn get_bool(obj: &JsValue, key: &str, default: bool) -> bool {
    Reflect::get(obj, &JsValue::from_str(key))
        .ok()
        .and_then(|v| v.as_bool())
        .unwrap_or(default)
}

fn get_int(obj: &JsValue, key: &str, default: i32) -> i32 {
    Reflect::get(obj, &JsValue::from_str(key))
        .ok()
        .and_then(|v| v.as_f64())
        .map(|v| v as i32)
        .unwrap_or(default)
}

fn get_string(obj: &JsValue, key: &str, default: &str) -> String {
    Reflect::get(obj, &JsValue::from_str(key))
        .ok()
        .and_then(|v| v.as_string())
        .unwrap_or_else(|| default.to_string())
}

/// Generate a barcode as a Base64-encoded PNG data URI.
///
/// Returns the base64 string on success, or `"error: <message>"` on failure.
#[wasm_bindgen]
pub fn barcode_draw_base64(type_id: &str, code: &str, width: i32, height: i32, options_json: &str) -> String {
    let opts = DrawOptions::from_json(options_json);
    match draw_internal(type_id, code, FORMAT_PNG, width, height, &opts) {
        Ok(s) => s,
        Err(e) => format!("error: {}", e),
    }
}

/// Generate a barcode as an SVG string.
///
/// Returns the SVG markup on success, or `"error: <message>"` on failure.
#[wasm_bindgen]
pub fn barcode_draw_svg(type_id: &str, code: &str, width: i32, height: i32, options_json: &str) -> String {
    let opts = DrawOptions::from_json(options_json);
    match draw_internal(type_id, code, FORMAT_SVG, width, height, &opts) {
        Ok(s) => s,
        Err(e) => format!("error: {}", e),
    }
}

/// Get supported barcode types as a JSON string.
#[wasm_bindgen]
pub fn barcode_get_types() -> String {
    BARCODE_TYPES_JSON.to_string()
}

/// Set trial mode (true = show SAMPLE watermark).
#[wasm_bindgen]
pub fn barcode_set_trial_mode(is_trial: bool) {
    crate::product_info::set_trial_mode(is_trial);
}

// --- Internal implementation ---

fn apply_1d_opts(base_1d: &mut crate::base_1d::BarcodeBase1D, opts: &DrawOptions) {
    base_1d.show_text = opts.show_text;
    base_1d.text_even_spacing = opts.even_spacing;
}

fn get_output(base: &BarcodeBase, format: &str) -> Result<String, String> {
    if format == FORMAT_SVG {
        base.get_svg().map_err(|e| e.to_string())
    } else {
        base.get_image_base64().map_err(|e| e.to_string())
    }
}

#[allow(clippy::too_many_lines)]
fn draw_internal(type_id: &str, code: &str, format: &str, width: i32, height: i32, opts: &DrawOptions) -> Result<String, String> {
    match type_id {
        "QR" => {
            let mut bc = QR::new(format);
            bc.error_correction = opts.qr_error_level;
            if opts.qr_version > 0 {
                bc.version = opts.qr_version;
            }
            bc.draw(code, width).map_err(|e| e.to_string())?;
            get_output(&bc.base_2d.base, format)
        }
        "DataMatrix" => {
            let mut bc = DataMatrix::new(format);
            if opts.datamatrix_size >= 0 {
                bc.set_code_size(opts.datamatrix_size);
            }
            bc.draw(code, width).map_err(|e| e.to_string())?;
            get_output(&bc.base_2d.base, format)
        }
        "PDF417" => {
            let mut bc = PDF417::new(format);
            if opts.pdf417_error_level >= 0 {
                bc.set_error_correction_level(opts.pdf417_error_level);
            }
            if opts.pdf417_cols > 0 {
                bc.set_columns(opts.pdf417_cols);
            }
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_2d.base, format)
        }
        "Code39" => {
            let mut bc = Code39::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "Code93" => {
            let mut bc = Code93::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "Code128" => {
            let mut bc = Code128::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "GS1-128" => {
            let mut bc = GS1128::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "NW7" => {
            let mut bc = NW7::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "ITF" => {
            let mut bc = ITF::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "Matrix2of5" => {
            let mut bc = Matrix2of5::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "NEC2of5" => {
            let mut bc = NEC2of5::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "JAN8" => {
            let mut bc = JAN8::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "JAN13" => {
            let mut bc = JAN13::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "UPC-A" => {
            let mut bc = UPCA::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "UPC-E" => {
            let mut bc = UPCE::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "GS1DataBar14" => {
            let sym_type = match opts.databar14_type.as_str() {
                "stacked" => SymbolType14::Stacked,
                "stacked_omni" => SymbolType14::StackedOmnidirectional,
                _ => SymbolType14::Omnidirectional,
            };
            let mut bc = GS1DataBar14::new(format, sym_type);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "GS1DataBarLimited" => {
            let mut bc = GS1DataBarLimited::new(format);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "GS1DataBarExpanded" => {
            let sym_type = match opts.databar_exp_type.as_str() {
                "stacked" => ExpandedSymbolType::Stacked,
                _ => ExpandedSymbolType::Unstacked,
            };
            let mut bc = GS1DataBarExpanded::new(format, sym_type, opts.databar_exp_cols);
            apply_1d_opts(&mut bc.base_1d, opts);
            bc.draw(code, width, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        "YubinCustomer" => {
            let mut bc = YubinCustomer::new(format);
            bc.draw(code, height).map_err(|e| e.to_string())?;
            get_output(&bc.base_1d.base, format)
        }
        _ => Err(format!("Unknown barcode type: {}", type_id)),
    }
}

/// JSON array of supported barcode types with metadata.
const BARCODE_TYPES_JSON: &str = r#"[
{"id":"QR","label":"QR Code","group":"2D Barcode","dim":"2d","default":"https://www.pao.ac/","w":200,"h":200},
{"id":"DataMatrix","label":"DataMatrix","group":"2D Barcode","dim":"2d","default":"Hello Barcode.Rust","w":200,"h":200},
{"id":"PDF417","label":"PDF417","group":"2D Barcode","dim":"2d","default":"Hello Barcode.Rust","w":400,"h":120},
{"id":"Code39","label":"Code 39","group":"1D Barcode","dim":"1d","default":"ABC-1234","w":400,"h":100},
{"id":"Code93","label":"Code 93","group":"1D Barcode","dim":"1d","default":"ABC-1234","w":400,"h":100},
{"id":"Code128","label":"Code 128","group":"1D Barcode","dim":"1d","default":"Hello-2026","w":400,"h":100},
{"id":"NW7","label":"NW-7 (Codabar)","group":"1D Barcode","dim":"1d","default":"A12345B","w":400,"h":100},
{"id":"ITF","label":"ITF","group":"1D Barcode","dim":"1d","default":"1234567890","w":400,"h":100},
{"id":"Matrix2of5","label":"Matrix 2 of 5","group":"1D Barcode","dim":"1d","default":"1234567890","w":400,"h":100},
{"id":"NEC2of5","label":"NEC 2 of 5","group":"1D Barcode","dim":"1d","default":"1234567890","w":400,"h":100},
{"id":"GS1-128","label":"GS1-128","group":"1D Barcode","dim":"1d","default":"{01}04912345123459{10}ABC123","w":500,"h":100},
{"id":"JAN13","label":"JAN-13 (EAN-13)","group":"JAN / UPC","dim":"1d","default":"4901234567894","w":300,"h":100},
{"id":"JAN8","label":"JAN-8 (EAN-8)","group":"JAN / UPC","dim":"1d","default":"49123456","w":250,"h":100},
{"id":"UPC-A","label":"UPC-A","group":"JAN / UPC","dim":"1d","default":"01234567890","w":300,"h":100},
{"id":"UPC-E","label":"UPC-E","group":"JAN / UPC","dim":"1d","default":"0123456","w":250,"h":100},
{"id":"GS1DataBar14","label":"GS1 DataBar","group":"GS1 DataBar","dim":"1d","default":"0001234567890","w":400,"h":100},
{"id":"GS1DataBarLimited","label":"GS1 DataBar Limited","group":"GS1 DataBar","dim":"1d","default":"0001234567890","w":400,"h":100},
{"id":"GS1DataBarExpanded","label":"GS1 DataBar Expanded","group":"GS1 DataBar","dim":"1d","default":"{01}90012345678908{3103}000123","w":500,"h":100},
{"id":"YubinCustomer","label":"Yubin Customer","group":"Special","dim":"postal","default":"10408-0506001","w":400,"h":40}
]"#;
