//! tiny-skia renderer — draws barcodes onto tiny_skia::Pixmap canvases.
//!
//! This module provides a pure Rust 2D renderer using the tiny-skia crate.
//! It can render all barcode types onto Pixmap canvases or directly to PNG files.

use tiny_skia::*;
use crate::barcode_data::{BarcodeData, BarcodeDataType, EncodeBarcodeData};
use crate::color::Rgba;
use crate::error::{BarcodeError, Result};
use crate::encoders::jan_upc_render;

#[cfg(feature = "font")]
use ab_glyph::{Font, FontVec, PxScale, ScaleFont};

/// tiny-skia barcode renderer.
pub struct Renderer {
    /// Foreground color (bars/modules).
    pub fore_color: Rgba,
    /// Background color.
    pub back_color: Rgba,
    /// Whether to show text on 1D barcodes.
    pub show_text: bool,
    /// Font size for text rendering.
    pub font_size: f64,
}

impl Default for Renderer {
    fn default() -> Self {
        Self::new()
    }
}

impl Renderer {
    /// Create a new renderer with default settings (black on white, show text, 12pt font).
    pub fn new() -> Self {
        Self {
            fore_color: Rgba::new(0, 0, 0, 255),
            back_color: Rgba::new(255, 255, 255, 255),
            show_text: true,
            font_size: 12.0,
        }
    }

    /// Draw BarcodeData onto an existing Pixmap at (x, y) with given dimensions.
    pub fn draw(
        &self,
        pixmap: &mut Pixmap,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        // Fill background
        fill_rect(
            pixmap,
            x as f32,
            y as f32,
            width as f32,
            height as f32,
            data.back_color,
        );

        match data.data_type {
            BarcodeDataType::Type1D => self.draw_1d(pixmap, data, x, y, width, height),
            BarcodeDataType::TypeJANUPC => self.draw_jan_upc(pixmap, data, x, y, width, height),
            BarcodeDataType::Type4State => self.draw_4state(pixmap, data, x, y, width, height),
            BarcodeDataType::Type2DMatrix => self.draw_2d(pixmap, data, x, y, width, height),
            BarcodeDataType::TypeMultiRow => self.draw_multi_row(pixmap, data, x, y, width, height),
        }?;

        // Draw SAMPLE overlay if trial mode
        if crate::product_info::is_trial_mode() {
            self.draw_sample_overlay(pixmap, x, y, width, height);
        }

        Ok(())
    }

    /// Convenience: encode and draw a barcode.
    #[allow(clippy::too_many_arguments)]
    pub fn draw_barcode(
        &self,
        pixmap: &mut Pixmap,
        bc: &mut dyn EncodeBarcodeData,
        code: &str,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        let data = bc.encode_data(code)?;
        self.draw(pixmap, &data, x, y, width, height)
    }

    /// Create a new Pixmap and render the barcode.
    pub fn render(&self, data: &BarcodeData, width: u32, height: u32) -> Result<Pixmap> {
        let mut pixmap = Pixmap::new(width, height)
            .ok_or_else(|| BarcodeError::RenderError("Failed to create pixmap".into()))?;
        self.draw(&mut pixmap, data, 0.0, 0.0, width as f64, height as f64)?;
        Ok(pixmap)
    }

    /// Render to a PNG file.
    pub fn render_to_file(
        &self,
        data: &BarcodeData,
        width: u32,
        height: u32,
        path: &str,
    ) -> Result<()> {
        let pixmap = self.render(data, width, height)?;
        pixmap
            .save_png(path)
            .map_err(|e| BarcodeError::RenderError(format!("Failed to save PNG: {}", e)))?;
        Ok(())
    }

    // --- Type-specific renderers ---

    fn draw_1d(
        &self,
        pixmap: &mut Pixmap,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        let total_units: i32 = data.pattern.iter().sum();
        if total_units == 0 {
            return Ok(());
        }

        let unit_width = width / total_units as f64;
        let mut current_x = x;

        // Calculate text area
        let bar_height = if data.show_text && self.show_text {
            let text_height = self.font_size * 1.5;
            height - text_height
        } else {
            height
        };

        // Draw bars (even indices)
        for (i, &units) in data.pattern.iter().enumerate() {
            let bar_width = units as f64 * unit_width;
            if i % 2 == 0 {
                fill_rect(
                    pixmap,
                    current_x as f32,
                    y as f32,
                    bar_width as f32,
                    bar_height as f32,
                    data.fore_color,
                );
            }
            current_x += bar_width;
        }

        // Draw text
        #[cfg(feature = "font")]
        if data.show_text && self.show_text && !data.display_code.is_empty() {
            if let Some(font) = crate::font::get_default_font() {
                let text_y = y + bar_height;
                let text_x = x + width / 2.0;
                draw_text_centered(
                    pixmap,
                    &font,
                    &data.display_code,
                    text_x as f32,
                    text_y as f32,
                    self.font_size as f32,
                    data.fore_color,
                );
            }
        }

        Ok(())
    }

    fn draw_jan_upc(
        &self,
        pixmap: &mut Pixmap,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        if !data.extended_guard {
            return self.draw_1d(pixmap, data, x, y, width, height);
        }

        let params = jan_upc_render::compute_params(
            height as i32,
            data.text_font_scale,
            data.text_v_offset_scale,
            &data.pattern,
            width as i32,
            &data.jan_upc_type,
        );

        let total_units: i32 = data.pattern.iter().sum();
        if total_units == 0 {
            return Ok(());
        }

        let unit_width = params.barcode_width / total_units as f64;
        let mut current_x = x + params.barcode_start_x;
        let mut bar_index = 0;

        let tall_bars = get_tall_bar_indices(&data.jan_upc_type, data.pattern.len());

        // Draw bars with variable heights (screen coords: y down)
        for (i, &unit_count) in data.pattern.iter().enumerate() {
            let bar_width = unit_count as f64 * unit_width;

            if i % 2 == 0 {
                let is_tall = tall_bars.contains(&bar_index);
                let bar_height = if is_tall {
                    params.tall_bar_height
                } else {
                    params.normal_bar_height
                };

                // All bars start from top (y), tall bars extend further down
                fill_rect(
                    pixmap,
                    current_x as f32,
                    y as f32,
                    bar_width as f32,
                    bar_height as f32,
                    data.fore_color,
                );
                bar_index += 1;
            }

            current_x += bar_width;
        }

        // Draw text below bars
        #[cfg(feature = "font")]
        if data.show_text && self.show_text && !data.display_code.is_empty() {
            if let Some(font) = crate::font::get_default_font() {
                let text_x = x + width / 2.0;
                let text_y = y + params.text_y;
                if data.text_even_spacing {
                    draw_text_even(
                        pixmap,
                        &font,
                        &data.display_code,
                        (x + params.barcode_start_x) as f32,
                        text_y as f32,
                        params.barcode_width as f32,
                        params.svg_font_size as f32,
                        data.fore_color,
                        data.text_h_spacing_scale as f32,
                    );
                } else {
                    draw_text_centered(
                        pixmap,
                        &font,
                        &data.display_code,
                        text_x as f32,
                        text_y as f32,
                        params.svg_font_size as f32,
                        data.fore_color,
                    );
                }
            }
        }

        Ok(())
    }

    fn draw_4state(
        &self,
        pixmap: &mut Pixmap,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        let num_bars = data.bars_4state.len();
        if num_bars == 0 {
            return Ok(());
        }

        let bar_width = width / num_bars as f64;
        let segment_height = height / 3.0;

        for (i, &bar_type) in data.bars_4state.iter().enumerate() {
            let bar_x = x + i as f64 * bar_width;
            // Screen coords: y increases downward
            let (bar_y, bar_h) = match bar_type {
                1 => (y, height),                                    // Full height
                2 => (y, segment_height * 2.0),                      // Upper 2/3 (ascender)
                3 => (y + segment_height, segment_height * 2.0),     // Lower 2/3 (descender)
                4 => (y + segment_height, segment_height),           // Middle 1/3 (tracker)
                _ => continue,
            };
            fill_rect(
                pixmap,
                bar_x as f32,
                bar_y as f32,
                bar_width as f32,
                bar_h as f32,
                data.fore_color,
            );
        }

        Ok(())
    }

    fn draw_2d(
        &self,
        pixmap: &mut Pixmap,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        if data.matrix.is_empty() {
            return Ok(());
        }

        let (rows, cols) = if data.col_major {
            (data.matrix[0].len(), data.matrix.len())
        } else {
            (data.matrix.len(), data.matrix[0].len())
        };

        if rows == 0 || cols == 0 {
            return Ok(());
        }

        let module_width = width / cols as f64;
        let module_height = height / rows as f64;

        for row in 0..rows {
            for col in 0..cols {
                let is_filled = if data.col_major {
                    col < data.matrix.len() && row < data.matrix[col].len() && data.matrix[col][row]
                } else {
                    row < data.matrix.len() && col < data.matrix[row].len() && data.matrix[row][col]
                };

                if is_filled {
                    let px = x + col as f64 * module_width;
                    let py = y + row as f64 * module_height;
                    fill_rect(
                        pixmap,
                        px as f32,
                        py as f32,
                        module_width as f32,
                        module_height as f32,
                        data.fore_color,
                    );
                }
            }
        }

        Ok(())
    }

    fn draw_multi_row(
        &self,
        pixmap: &mut Pixmap,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        let num_rows = data.multi_row_patterns.len();
        if num_rows == 0 {
            return Ok(());
        }

        let total_height_units: i32 = data.multi_row_heights.iter().sum();
        if total_height_units == 0 {
            return Ok(());
        }

        let unit_height = height / total_height_units as f64;
        let mut current_y = y;

        for (i, pattern_str) in data.multi_row_patterns.iter().enumerate() {
            let row_height = data.multi_row_heights.get(i).copied().unwrap_or(1) as f64 * unit_height;
            if row_height <= 0.0 {
                current_y += row_height;
                continue;
            }

            let pattern: Vec<i32> = pattern_str
                .split_whitespace()
                .filter_map(|s| s.parse().ok())
                .collect();

            let total_units: i32 = pattern.iter().sum();
            if total_units == 0 {
                current_y += row_height;
                continue;
            }

            let unit_width = width / total_units as f64;
            let mut current_x = x;

            for (j, &units) in pattern.iter().enumerate() {
                let bar_width = units as f64 * unit_width;
                if j % 2 == 0 {
                    fill_rect(
                        pixmap,
                        current_x as f32,
                        current_y as f32,
                        bar_width as f32,
                        row_height as f32,
                        data.fore_color,
                    );
                }
                current_x += bar_width;
            }

            current_y += row_height;
        }

        Ok(())
    }

    #[cfg(feature = "font")]
    fn draw_sample_overlay(
        &self,
        pixmap: &mut Pixmap,
        x: f64,
        y: f64,
        _width: f64,
        height: f64,
    ) {
        let font_size = (height * 0.12).clamp(8.0, 40.0);
        let margin = (height * 0.01).max(2.0);
        let text = crate::product_info::get_trial_text();
        let red = Rgba::new(255, 0, 0, 255);

        if let Some(font) = crate::font::get_default_font() {
            draw_text_at(
                pixmap,
                &font,
                text,
                (x + margin) as f32,
                (y + margin) as f32,
                font_size as f32,
                red,
            );
        }
    }

    #[cfg(not(feature = "font"))]
    fn draw_sample_overlay(&self, _pixmap: &mut Pixmap, _x: f64, _y: f64, _width: f64, _height: f64) {
        // No-op without font feature
    }
}

// --- Helper functions ---

/// Fill a rectangle on the pixmap.
fn fill_rect(pixmap: &mut Pixmap, x: f32, y: f32, w: f32, h: f32, color: Rgba) {
    if let Some(rect) = tiny_skia::Rect::from_xywh(x, y, w, h) {
        let mut paint = Paint::default();
        paint.set_color_rgba8(color.r, color.g, color.b, color.a);
        paint.anti_alias = false;
        pixmap.fill_rect(rect, &paint, Transform::identity(), None);
    }
}

/// Get indices of tall bars (guard bars) for JAN/UPC barcodes.
fn get_tall_bar_indices(jan_upc_type: &str, pattern_len: usize) -> Vec<usize> {
    let total_bars = pattern_len.div_ceil(2);

    match jan_upc_type {
        "JAN13" | "UPC-A" => {
            let mut tall = vec![0, 1];
            tall.push(14);
            tall.push(15);
            if total_bars >= 2 {
                tall.push(total_bars - 2);
                tall.push(total_bars - 1);
            }
            tall
        }
        "JAN8" => {
            let mut tall = vec![0, 1];
            tall.push(10);
            tall.push(11);
            if total_bars >= 2 {
                tall.push(total_bars - 2);
                tall.push(total_bars - 1);
            }
            tall
        }
        "UPC-E" => {
            let mut tall = vec![0, 1];
            if total_bars >= 3 {
                tall.push(total_bars - 3);
                tall.push(total_bars - 2);
                tall.push(total_bars - 1);
            }
            tall
        }
        _ => Vec::new(),
    }
}

// --- Text rendering functions (font feature required) ---

#[cfg(feature = "font")]
fn draw_text_at(
    pixmap: &mut Pixmap,
    font: &FontVec,
    text: &str,
    x: f32,
    y: f32,
    font_size: f32,
    color: Rgba,
) {
    let scale = PxScale::from(font_size);
    let scaled_font = font.as_scaled(scale);
    let ascent = scaled_font.ascent();
    let w = pixmap.width();
    let h = pixmap.height();
    let data = pixmap.data_mut();

    let mut cursor_x = x;
    let y_baseline = y + ascent;

    for ch in text.chars() {
        let glyph_id = scaled_font.glyph_id(ch);
        let glyph = glyph_id.with_scale_and_position(
            scale,
            ab_glyph::point(cursor_x, y_baseline),
        );

        if let Some(outlined) = scaled_font.outline_glyph(glyph) {
            let bounds = outlined.px_bounds();
            outlined.draw(|gx, gy, coverage| {
                let px = bounds.min.x as i32 + gx as i32;
                let py = bounds.min.y as i32 + gy as i32;
                if px >= 0 && py >= 0 && (px as u32) < w && (py as u32) < h {
                    let alpha = coverage * (color.a as f32 / 255.0);
                    if alpha > 0.0 {
                        let base = ((py as u32 * w + px as u32) * 4) as usize;
                        if base + 3 < data.len() {
                            let inv_a = 1.0 - alpha;
                            // Blend in premultiplied space
                            data[base] = (color.r as f32 * alpha + data[base] as f32 * inv_a)
                                .round().clamp(0.0, 255.0) as u8;
                            data[base + 1] = (color.g as f32 * alpha + data[base + 1] as f32 * inv_a)
                                .round().clamp(0.0, 255.0) as u8;
                            data[base + 2] = (color.b as f32 * alpha + data[base + 2] as f32 * inv_a)
                                .round().clamp(0.0, 255.0) as u8;
                            data[base + 3] = (alpha * 255.0 + data[base + 3] as f32 * inv_a)
                                .round().clamp(0.0, 255.0) as u8;
                        }
                    }
                }
            });
        }

        cursor_x += scaled_font.h_advance(glyph_id);
    }
}

#[cfg(feature = "font")]
fn draw_text_centered(
    pixmap: &mut Pixmap,
    font: &FontVec,
    text: &str,
    cx: f32,
    y: f32,
    font_size: f32,
    color: Rgba,
) {
    let scale = PxScale::from(font_size);
    let scaled_font = font.as_scaled(scale);

    // Measure text width
    let text_width: f32 = text
        .chars()
        .map(|ch| scaled_font.h_advance(scaled_font.glyph_id(ch)))
        .sum();

    let x = cx - text_width / 2.0;
    draw_text_at(pixmap, font, text, x, y, font_size, color);
}

#[cfg(feature = "font")]
#[allow(clippy::too_many_arguments)]
fn draw_text_even(
    pixmap: &mut Pixmap,
    font: &FontVec,
    text: &str,
    x: f32,
    y: f32,
    width: f32,
    font_size: f32,
    color: Rgba,
    h_scale: f32,
) {
    let chars: Vec<char> = text.chars().collect();
    if chars.is_empty() {
        return;
    }

    let margin_ratio: f32 = if width < 100.0 {
        0.03
    } else if width > 400.0 {
        0.07
    } else {
        0.05
    };

    let horizontal_margin = width * margin_ratio * h_scale;
    let text_width = width - 2.0 * horizontal_margin;

    let num_chars = chars.len();
    let char_spacing = if num_chars > 1 {
        (text_width / (num_chars - 1) as f32).max(font_size / 2.0)
    } else {
        0.0
    };

    let total_text_width = if num_chars > 1 {
        char_spacing * (num_chars - 1) as f32
    } else {
        0.0
    };
    let start_x = x + (width - total_text_width) / 2.0;

    for (i, ch) in chars.iter().enumerate() {
        let char_x = start_x + i as f32 * char_spacing;
        draw_text_at(
            pixmap,
            font,
            &ch.to_string(),
            char_x,
            y,
            font_size,
            color,
        );
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_renderer_new() {
        let renderer = Renderer::new();
        assert_eq!(renderer.fore_color.r, 0);
        assert_eq!(renderer.back_color.r, 255);
        assert!(renderer.show_text);
        assert_eq!(renderer.font_size, 12.0);
    }

    #[test]
    fn test_get_tall_bar_indices_jan13() {
        let indices = get_tall_bar_indices("JAN13", 59);
        assert!(indices.contains(&0));
        assert!(indices.contains(&1));
        assert!(indices.contains(&14));
        assert!(indices.contains(&15));
    }

    #[test]
    fn test_fill_rect_zero_size() {
        // Rect with zero size should not panic
        let mut pixmap = Pixmap::new(10, 10).unwrap();
        fill_rect(&mut pixmap, 0.0, 0.0, 0.0, 0.0, Rgba::BLACK);
    }

    #[test]
    fn test_render_creates_pixmap() {
        let renderer = Renderer::new();
        // Create minimal 1D data
        let data = BarcodeData {
            data_type: BarcodeDataType::Type1D,
            fore_color: Rgba::BLACK,
            back_color: Rgba::WHITE,
            pattern: vec![1, 1, 1, 1],
            display_code: String::new(),
            show_text: false,
            text_even_spacing: false,
            text_font_scale: 1.0,
            text_h_spacing_scale: 1.0,
            text_v_offset_scale: 1.0,
            jan_upc_type: String::new(),
            extended_guard: false,
            bars_4state: Vec::new(),
            matrix: Vec::new(),
            col_major: false,
            multi_row_patterns: Vec::new(),
            multi_row_heights: Vec::new(),
        };
        let pixmap = renderer.render(&data, 100, 50).unwrap();
        assert_eq!(pixmap.width(), 100);
        assert_eq!(pixmap.height(), 50);
    }
}
