//! printpdf renderer — draws barcodes onto PDF documents.
//!
//! This renderer is gated behind the `printpdf-renderer` feature flag.
//!
//! # Design
//!
//! - The Renderer holds options (colors, font_size, show_text) but NOT a PDF reference
//! - The `draw()` method takes a `&PdfLayerReference` parameter to draw on
//! - Coordinates are in mm, with y=0 at top (converted internally to bottom-left)
//! - **Text rendering is not supported** in this renderer (printpdf font handling is complex)
//!
//! # Example
//!
//! ```ignore
//! use barcode_pao::renderers::printpdf::Renderer;
//! use barcode_pao::encoders::code128::Code128;
//! use barcode_pao::barcode_data::EncodeBarcodeData;
//! use printpdf::*;
//!
//! let (doc, page1, layer1) = PdfDocument::new("Barcode", Mm(210.0), Mm(297.0), "Layer 1");
//! let layer = doc.get_page(page1).get_layer(layer1);
//!
//! let mut code128 = Code128::new("png");
//! let renderer = Renderer::new();
//! renderer.draw_barcode(&layer, &mut code128, "ABC123", 10.0, 10.0, 80.0, 30.0, 297.0).unwrap();
//! ```

use printpdf::*;
use crate::barcode_data::{BarcodeData, BarcodeDataType, EncodeBarcodeData};
use crate::color::Rgba;
use crate::error::Result;
use crate::encoders::jan_upc_render;

/// printpdf renderer for drawing barcodes onto PDF documents.
pub struct Renderer {
    /// Foreground color (bar color)
    pub fore_color: Rgba,
    /// Background color (space color)
    pub back_color: Rgba,
    /// Whether to show text below barcode (currently ignored - text not supported)
    pub show_text: bool,
    /// Font size for text (currently ignored - text not supported)
    pub font_size: f64,
}

impl Default for Renderer {
    fn default() -> Self {
        Self::new()
    }
}

impl Renderer {
    /// Create a new printpdf renderer with default settings.
    pub fn new() -> Self {
        Self {
            fore_color: Rgba { r: 0, g: 0, b: 0, a: 255 },
            back_color: Rgba { r: 255, g: 255, b: 255, a: 255 },
            show_text: true,
            font_size: 10.0,
        }
    }

    /// Draw BarcodeData onto a PDF layer.
    ///
    /// # Parameters
    ///
    /// - `layer`: The PDF layer to draw on
    /// - `data`: The encoded barcode data
    /// - `x`: X position in mm (from left)
    /// - `y`: Y position in mm (from top of page)
    /// - `width`: Barcode width in mm
    /// - `height`: Barcode height in mm
    /// - `page_height`: Total page height in mm (needed to convert y coordinate)
    #[allow(clippy::too_many_arguments)]
    pub fn draw(
        &self,
        layer: &PdfLayerReference,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
        page_height: f64,
    ) -> Result<()> {
        // Convert y from top-down to bottom-up (PDF coordinate system)
        let y_bottom = page_height - y - height;

        match data.data_type {
            BarcodeDataType::Type1D => {
                self.draw_1d(layer, data, x, y_bottom, width, height)?;
            }
            BarcodeDataType::TypeJANUPC => {
                self.draw_jan_upc(layer, data, x, y_bottom, width, height)?;
            }
            BarcodeDataType::Type4State => {
                self.draw_4state(layer, data, x, y_bottom, width, height)?;
            }
            BarcodeDataType::Type2DMatrix => {
                self.draw_2d(layer, data, x, y_bottom, width, height)?;
            }
            BarcodeDataType::TypeMultiRow => {
                self.draw_multi_row(layer, data, x, y_bottom, width, height)?;
            }
        }

        if crate::product_info::is_trial_mode() {
            self.draw_trial_watermark(layer, x, y_bottom, width, height)?;
        }

        Ok(())
    }

    /// Convenience method: encode and draw a barcode in one call.
    #[allow(clippy::too_many_arguments)]
    pub fn draw_barcode(
        &self,
        layer: &PdfLayerReference,
        bc: &mut dyn EncodeBarcodeData,
        code: &str,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
        page_height: f64,
    ) -> Result<()> {
        let data = bc.encode_data(code)?;
        self.draw(layer, &data, x, y, width, height, page_height)
    }

    /// Draw a standard 1D barcode.
    fn draw_1d(
        &self,
        layer: &PdfLayerReference,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        let total_units: i32 = data.pattern.iter().sum();
        if total_units == 0 {
            return Ok(());
        }

        let unit_width = width / total_units as f64;
        let mut current_x = x;

        for (i, &unit_count) in data.pattern.iter().enumerate() {
            let bar_width = unit_count as f64 * unit_width;
            if i % 2 == 0 {
                fill_rect(layer, current_x, y, bar_width, height, &self.fore_color);
            }
            current_x += bar_width;
        }

        Ok(())
    }

    /// Draw a JAN/UPC barcode with extended guards.
    fn draw_jan_upc(
        &self,
        layer: &PdfLayerReference,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        if !data.extended_guard {
            return self.draw_1d(layer, data, x, y, width, height);
        }

        let params = jan_upc_render::compute_params(
            height as i32,
            data.text_font_scale,
            data.text_v_offset_scale,
            &data.pattern,
            width as i32,
            &data.jan_upc_type,
        );

        let total_units: i32 = data.pattern.iter().sum();
        if total_units == 0 {
            return Ok(());
        }

        let unit_width = params.barcode_width / total_units as f64;
        let mut current_x = x + params.barcode_start_x;
        let mut bar_index = 0;

        let tall_bars = get_tall_bar_indices(&data.jan_upc_type, data.pattern.len());

        for (i, &unit_count) in data.pattern.iter().enumerate() {
            let bar_width = unit_count as f64 * unit_width;

            if i % 2 == 0 {
                let is_tall = tall_bars.contains(&bar_index);
                let bar_height = if is_tall {
                    params.tall_bar_height
                } else {
                    params.normal_bar_height
                };

                // In PDF coords (y up), bars are flush at top: y + height - bar_height
                fill_rect(
                    layer,
                    current_x,
                    y + height - bar_height,
                    bar_width,
                    bar_height,
                    &self.fore_color,
                );
                bar_index += 1;
            }

            current_x += bar_width;
        }

        Ok(())
    }

    /// Draw a 4-state barcode (e.g., YubinCustomer).
    fn draw_4state(
        &self,
        layer: &PdfLayerReference,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        let bar_count = data.bars_4state.len();
        if bar_count == 0 {
            return Ok(());
        }

        let bar_width = width / bar_count as f64;
        let segment_height = height / 3.0;

        for (i, &bar_type) in data.bars_4state.iter().enumerate() {
            let bar_x = x + i as f64 * bar_width;

            // PDF coords: y increases upward, y is the bottom of the barcode
            let (bar_y, bar_h) = match bar_type {
                1 => (y, height),                                    // Full height
                2 => (y + segment_height, segment_height * 2.0),     // Upper 2/3 (ascender)
                3 => (y, segment_height * 2.0),                      // Lower 2/3 (descender)
                4 => (y + segment_height, segment_height),           // Middle 1/3 (tracker)
                _ => continue,
            };

            fill_rect(layer, bar_x, bar_y, bar_width, bar_h, &self.fore_color);
        }

        Ok(())
    }

    /// Draw a 2D matrix barcode (QR, DataMatrix).
    fn draw_2d(
        &self,
        layer: &PdfLayerReference,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        if data.matrix.is_empty() {
            return Ok(());
        }

        let (rows, cols) = if data.col_major {
            (data.matrix[0].len(), data.matrix.len())
        } else {
            (data.matrix.len(), data.matrix[0].len())
        };

        if rows == 0 || cols == 0 {
            return Ok(());
        }

        let module_width = width / cols as f64;
        let module_height = height / rows as f64;

        for row in 0..rows {
            for col in 0..cols {
                let is_filled = if data.col_major {
                    col < data.matrix.len() && row < data.matrix[col].len() && data.matrix[col][row]
                } else {
                    row < data.matrix.len() && col < data.matrix[row].len() && data.matrix[row][col]
                };

                if is_filled {
                    let module_x = x + col as f64 * module_width;
                    // Flip Y: row 0 at top → highest y in PDF
                    let module_y = y + (rows - 1 - row) as f64 * module_height;
                    fill_rect(
                        layer,
                        module_x,
                        module_y,
                        module_width,
                        module_height,
                        &self.fore_color,
                    );
                }
            }
        }

        Ok(())
    }

    /// Draw a multi-row barcode (e.g., PDF417, stacked DataBar).
    fn draw_multi_row(
        &self,
        layer: &PdfLayerReference,
        data: &BarcodeData,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        if data.multi_row_patterns.is_empty() {
            return Ok(());
        }

        let total_height_units: i32 = data.multi_row_heights.iter().sum();
        if total_height_units == 0 {
            return Ok(());
        }

        let unit_height = height / total_height_units as f64;
        // In PDF coords: start from top (y + height) and work downward
        let mut current_y = y + height;

        for (row_idx, pattern) in data.multi_row_patterns.iter().enumerate() {
            let row_height = data.multi_row_heights.get(row_idx).copied().unwrap_or(1) as f64 * unit_height;
            current_y -= row_height;

            let row_pattern: Vec<i32> = pattern
                .split_whitespace()
                .filter_map(|s| s.parse().ok())
                .collect();

            let total_units: i32 = row_pattern.iter().sum();
            if total_units == 0 {
                continue;
            }

            let unit_width = width / total_units as f64;
            let mut current_x = x;

            for (i, &unit_count) in row_pattern.iter().enumerate() {
                let bar_width = unit_count as f64 * unit_width;
                if i % 2 == 0 {
                    fill_rect(
                        layer,
                        current_x,
                        current_y,
                        bar_width,
                        row_height,
                        &self.fore_color,
                    );
                }
                current_x += bar_width;
            }
        }

        Ok(())
    }

    /// Draw trial watermark (gray bar since text is not supported).
    fn draw_trial_watermark(
        &self,
        layer: &PdfLayerReference,
        x: f64,
        y: f64,
        width: f64,
        height: f64,
    ) -> Result<()> {
        let _text = crate::product_info::get_trial_text();
        let gray = Rgba { r: 200, g: 200, b: 200, a: 128 };
        let text_height = height * 0.15;
        let text_y = y + (height - text_height) / 2.0;
        fill_rect(layer, x, text_y, width, text_height, &gray);
        Ok(())
    }
}

/// Fill a rectangle on the PDF layer.
///
/// Coordinates are in mm (f64), converted to printpdf's Mm(f32) internally.
/// Uses printpdf::Rect with lower-left (x, y) and upper-right (x+w, y+h).
fn fill_rect(layer: &PdfLayerReference, x: f64, y: f64, w: f64, h: f64, color: &Rgba) {
    layer.set_fill_color(Color::Rgb(Rgb::new(
        color.r as f32 / 255.0,
        color.g as f32 / 255.0,
        color.b as f32 / 255.0,
        None,
    )));

    let rect = Rect::new(
        Mm(x as f32),
        Mm(y as f32),
        Mm((x + w) as f32),
        Mm((y + h) as f32),
    );
    layer.add_rect(rect);
}

/// Get indices of tall bars (guard bars) for JAN/UPC barcodes.
///
/// Returns bar indices (counting only bars, not spaces) that should be drawn taller.
fn get_tall_bar_indices(jan_upc_type: &str, pattern_len: usize) -> Vec<usize> {
    // Count total bars (even-index elements in the pattern)
    let total_bars = pattern_len.div_ceil(2);

    match jan_upc_type {
        "JAN13" | "UPC-A" => {
            // Pattern: [start(3)] [6 digits * 4 elements] [center(5)] [6 digits * 4 elements] [end(3)]
            // Start guard: bars 0, 1 (from 3 elements: bar, space, bar)
            // Center guard: bars at middle (from 5 elements: space, bar, space, bar, space)
            // End guard: last 2 bars (from 3 elements: bar, space, bar)
            let mut tall = vec![0, 1]; // Start guard bars

            // Center guard: after 6 digits (each digit = 2 bars + 2 spaces = 4 elements)
            // Start: 3 elements (2 bars), then 6*4=24 elements (12 bars)
            // Center starts at element index 27, contains 5 elements (2 bars)
            // Bar indices: 2 + 12 = 14, 15
            tall.push(14);
            tall.push(15);

            // End guard: last 2 bars
            if total_bars >= 2 {
                tall.push(total_bars - 2);
                tall.push(total_bars - 1);
            }
            tall
        }
        "JAN8" => {
            // Pattern: [start(3)] [4 digits * 4] [center(5)] [4 digits * 4] [end(3)]
            let mut tall = vec![0, 1]; // Start guard

            // Center: after 4 digits → bar index 2 + 8 = 10, 11
            tall.push(10);
            tall.push(11);

            if total_bars >= 2 {
                tall.push(total_bars - 2);
                tall.push(total_bars - 1);
            }
            tall
        }
        "UPC-E" => {
            // UPC-E has different structure: start(3) + 6 digits + end(6)
            let mut tall = vec![0, 1]; // Start guard

            if total_bars >= 3 {
                tall.push(total_bars - 3);
                tall.push(total_bars - 2);
                tall.push(total_bars - 1);
            }
            tall
        }
        _ => Vec::new(),
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_renderer_new() {
        let renderer = Renderer::new();
        assert_eq!(renderer.fore_color.r, 0);
        assert_eq!(renderer.back_color.r, 255);
        assert!(renderer.show_text);
        assert_eq!(renderer.font_size, 10.0);
    }

    #[test]
    fn test_get_tall_bar_indices_jan13() {
        let indices = get_tall_bar_indices("JAN13", 59);
        assert!(indices.contains(&0));
        assert!(indices.contains(&1));
        assert!(indices.contains(&14));
        assert!(indices.contains(&15));
    }

    #[test]
    fn test_get_tall_bar_indices_jan8() {
        let indices = get_tall_bar_indices("JAN8", 43);
        assert!(indices.contains(&0));
        assert!(indices.contains(&1));
        assert!(indices.contains(&10));
        assert!(indices.contains(&11));
    }
}
