//! # barcode-pao
//!
//! Pure Rust barcode generation library — 18 symbologies.
//!
//! ## Supported Symbologies
//!
//! **1D:** Code39, Code93, Code128, GS1-128, NW-7, ITF, Matrix2of5, NEC2of5,
//! JAN-8, JAN-13, UPC-A, UPC-E, GS1 DataBar (Standard/Limited/Expanded)
//!
//! **2D:** QR Code, DataMatrix, PDF417
//!
//! **Special:** Japan Post Customer Barcode (Yubin Customer)
//!
//! ## Output Formats
//!
//! PNG, JPEG, SVG (feature-gated)

pub mod color;
pub mod error;
pub mod product_info;
pub mod base;
pub mod base_1d;
pub mod base_2d;
pub mod font;
pub mod encoders;
pub mod barcode_data;
pub mod renderers;

#[cfg(feature = "wasm")]
pub mod wasm;

// Re-export main types
pub use color::Rgba;
pub use error::{BarcodeError, Result};
pub use base::{BarcodeBase, FORMAT_PNG, FORMAT_JPEG, FORMAT_SVG};
pub use base_1d::{BarcodeBase1D, Encoder1D};
pub use base_2d::BarcodeBase2D;
pub use product_info::{set_trial_mode, is_trial_mode};
pub use barcode_data::{BarcodeData, BarcodeDataType, EncodeBarcodeData};

// Re-export encoder types
pub use encoders::code39::Code39;
pub use encoders::code93::Code93;
pub use encoders::nw7::NW7;
pub use encoders::itf::ITF;
pub use encoders::matrix2of5::Matrix2of5;
pub use encoders::nec2of5::NEC2of5;
pub use encoders::code128::Code128;
pub use encoders::gs1_128::GS1128;
pub use encoders::jan8::JAN8;
pub use encoders::jan13::JAN13;
pub use encoders::upc_a::UPCA;
pub use encoders::upc_e::UPCE;
pub use encoders::gs1_databar_14::{GS1DataBar14, SymbolType14};
pub use encoders::gs1_databar_limited::GS1DataBarLimited;
pub use encoders::gs1_databar_expanded::{GS1DataBarExpanded, ExpandedSymbolType};
pub use encoders::yubin_customer::YubinCustomer;
pub use encoders::datamatrix::DataMatrix;
pub use encoders::qr::{QR, QR_ECC_L, QR_ECC_M, QR_ECC_Q, QR_ECC_H};
pub use encoders::pdf417::{PDF417, SIZE_MODE_AUTO, SIZE_MODE_ROWS, SIZE_MODE_COLUMNS, SIZE_MODE_COLUMNS_AND_ROWS};
