use crate::color::Rgba;

/// Embedded Roboto-Regular font data.
const ROBOTO_FONT_DATA: &[u8] = include_bytes!("../Roboto-Regular.ttf");

/// Returns the embedded Roboto font data.
pub fn get_roboto_font_data() -> &'static [u8] {
    ROBOTO_FONT_DATA
}

/// Returns the parsed default font, or None on failure.
#[cfg(feature = "font")]
pub fn get_default_font() -> Option<ab_glyph::FontVec> {
    ab_glyph::FontVec::try_from_vec(ROBOTO_FONT_DATA.to_vec()).ok()
}

/// Draw text at a specific position on an RGBA image.
#[cfg(all(feature = "font", any(feature = "png", feature = "jpeg")))]
pub fn draw_text_at_png(
    img: &mut image::RgbaImage,
    font: &ab_glyph::FontVec,
    text: &str,
    x: i32,
    y: i32,
    font_size: i32,
    color: Rgba,
) {
    use ab_glyph::{Font, ScaleFont};

    let scale = ab_glyph::PxScale::from(font_size as f32);
    let scaled_font = font.as_scaled(scale);
    let ascent = scaled_font.ascent();

    let mut cursor_x = x as f32;
    let baseline_y = y as f32 + ascent;

    for ch in text.chars() {
        let glyph_id = scaled_font.glyph_id(ch);
        let glyph = glyph_id.with_scale_and_position(scale, ab_glyph::point(cursor_x, baseline_y));

        if let Some(outlined) = scaled_font.outline_glyph(glyph) {
            let bounds = outlined.px_bounds();
            outlined.draw(|gx, gy, coverage| {
                let px = bounds.min.x as i32 + gx as i32;
                let py = bounds.min.y as i32 + gy as i32;
                if px >= 0 && py >= 0 && (px as u32) < img.width() && (py as u32) < img.height() {
                    let alpha = (coverage * color.a as f32) as u8;
                    if alpha > 0 {
                        blend_pixel(img, px as u32, py as u32, color, alpha);
                    }
                }
            });
        }

        cursor_x += scaled_font.h_advance(glyph_id);
    }
}

/// Draw text centered horizontally at cx.
#[cfg(all(feature = "font", any(feature = "png", feature = "jpeg")))]
pub fn draw_text_centered_png(
    img: &mut image::RgbaImage,
    font: &ab_glyph::FontVec,
    text: &str,
    cx: i32,
    y: i32,
    font_size: i32,
    color: Rgba,
) {
    use ab_glyph::{Font, ScaleFont};

    let scale = ab_glyph::PxScale::from(font_size as f32);
    let scaled_font = font.as_scaled(scale);

    // Measure total width
    let total_w: f32 = text
        .chars()
        .map(|ch| scaled_font.h_advance(scaled_font.glyph_id(ch)))
        .sum();

    let start_x = cx as f32 - total_w / 2.0;
    draw_text_at_png(img, font, text, start_x as i32, y, font_size, color);
}

/// Draw a single character centered at cx.
#[cfg(all(feature = "font", any(feature = "png", feature = "jpeg")))]
pub fn draw_char_centered_png(
    img: &mut image::RgbaImage,
    font: &ab_glyph::FontVec,
    ch: char,
    cx: i32,
    y: i32,
    font_size: i32,
    color: Rgba,
) {
    use ab_glyph::{Font, ScaleFont};

    let scale = ab_glyph::PxScale::from(font_size as f32);
    let scaled_font = font.as_scaled(scale);
    let adv = scaled_font.h_advance(scaled_font.glyph_id(ch));
    let start_x = cx as f32 - adv / 2.0;

    draw_text_at_png(
        img,
        font,
        &ch.to_string(),
        start_x as i32,
        y,
        font_size,
        color,
    );
}

/// Draw evenly-distributed text on a PNG image.
#[cfg(all(feature = "font", any(feature = "png", feature = "jpeg")))]
#[allow(clippy::too_many_arguments)]
pub fn draw_text_even_png(
    img: &mut image::RgbaImage,
    font: &ab_glyph::FontVec,
    text: &str,
    x: f64,
    y: f64,
    width: f64,
    font_size: i32,
    color: Rgba,
    h_scale: f64,
) {
    let num_chars = text.len();
    if num_chars == 0 {
        return;
    }

    let margin_ratio = if width < 100.0 {
        0.03
    } else if width > 400.0 {
        0.07
    } else {
        0.05
    };

    let horizontal_margin = width * margin_ratio * h_scale;
    let text_width = width - 2.0 * horizontal_margin;

    let mut char_spacing = text_width;
    if num_chars > 1 {
        char_spacing = text_width / (num_chars - 1) as f64;
    }

    let min_spacing = font_size as f64 / 2.0;
    if char_spacing < min_spacing {
        char_spacing = min_spacing;
    }

    let total_text_width = if num_chars > 1 {
        char_spacing * (num_chars - 1) as f64
    } else {
        0.0
    };
    let start_x = x + (width - total_text_width) / 2.0;

    for (i, ch) in text.chars().enumerate() {
        let char_x = start_x + i as f64 * char_spacing;
        draw_char_centered_png(
            img,
            font,
            ch,
            char_x.round() as i32,
            y.round() as i32,
            font_size,
            color,
        );
    }
}

/// Alpha-blend a pixel onto the image.
#[cfg(any(feature = "png", feature = "jpeg"))]
fn blend_pixel(img: &mut image::RgbaImage, x: u32, y: u32, color: Rgba, alpha: u8) {
    let existing = img.get_pixel(x, y);
    let a = alpha as f32 / 255.0;
    let inv_a = 1.0 - a;
    let r = (color.r as f32 * a + existing[0] as f32 * inv_a) as u8;
    let g = (color.g as f32 * a + existing[1] as f32 * inv_a) as u8;
    let b = (color.b as f32 * a + existing[2] as f32 * inv_a) as u8;
    let out_a = (alpha as f32 + existing[3] as f32 * inv_a) as u8;
    img.put_pixel(x, y, image::Rgba([r, g, b, out_a]));
}
