use crate::base_1d::BarcodeBase1D;
use crate::error::{BarcodeError, Result};

// Digit patterns (3 bars each): "1"=long, "2"=semi-upper, "3"=semi-lower, "4"=timing
const YUBIN_PTN_N: &[&str] = &[
    "144", "114", "132", "312", "123", "141", "321", "213", "231", "411",
];
const YUBIN_PTN_C: &[&str] = &["324", "342", "234", "432", "243", "423", "441", "111"];
const YUBIN_START: &str = "13";
const YUBIN_STOP: &str = "31";
const YUBIN_PTN_HI: &str = "414";
const YUBIN_ASC_CHRS: &[u8] = b"0123456789-ABCDEFGHIJKLMNOPQRSTUVWXYZ";

fn yubin_get_pattern(index: usize) -> String {
    if index > 36 {
        return String::new();
    }
    if index <= 9 {
        YUBIN_PTN_N[index].to_string()
    } else if index == 10 {
        YUBIN_PTN_HI.to_string()
    } else if index <= 20 {
        format!("{}{}", YUBIN_PTN_C[0], YUBIN_PTN_N[index - 11])
    } else if index <= 30 {
        format!("{}{}", YUBIN_PTN_C[1], YUBIN_PTN_N[index - 21])
    } else {
        format!("{}{}", YUBIN_PTN_C[2], YUBIN_PTN_N[index - 31])
    }
}

fn yubin_get_check_pattern(chk_d: i32) -> String {
    if !(0..=18).contains(&chk_d) {
        return String::new();
    }
    if chk_d <= 9 {
        YUBIN_PTN_N[chk_d as usize].to_string()
    } else if chk_d == 10 {
        YUBIN_PTN_HI.to_string()
    } else {
        YUBIN_PTN_C[(chk_d - 11) as usize].to_string()
    }
}

fn yubin_pattern_to_check_value(pattern: &str) -> i32 {
    for (j, &ptn) in YUBIN_PTN_N.iter().enumerate().take(10) {
        if pattern == ptn {
            return j as i32;
        }
    }
    if pattern == YUBIN_PTN_HI {
        return 10;
    }
    for (j, &ptn) in YUBIN_PTN_C.iter().enumerate().take(8) {
        if pattern == ptn {
            return 11 + j as i32;
        }
    }
    -1
}

/// Japan Post Customer Barcode encoder (4-state barcode).
pub struct YubinCustomer {
    pub base_1d: BarcodeBase1D,
    bars: Vec<i32>,
    check_digit_value: i32,
}

impl YubinCustomer {
    pub fn new(output_format: &str) -> Self {
        Self {
            base_1d: BarcodeBase1D::new(output_format),
            bars: Vec::new(),
            check_digit_value: -1,
        }
    }

    /// Returns the bar pattern after encoding.
    pub fn bars(&self) -> &[i32] {
        &self.bars
    }

    /// Returns the check digit value (0-18) after encoding.
    pub fn check_digit(&self) -> i32 {
        self.check_digit_value
    }

    pub fn encode(&mut self, code: &str) -> Result<Vec<i32>> {
        if code.is_empty() {
            return Err(BarcodeError::EmptyCode);
        }

        let upper_code: Vec<u8> = code.bytes().map(|b| b.to_ascii_uppercase()).collect();

        let mut bars: Vec<i32> = Vec::new();
        let mut chk_str = String::new();

        // Start code
        for ch in YUBIN_START.bytes() {
            bars.push((ch - b'0') as i32);
        }

        // Data part (20 character positions)
        let mut code_len = 0usize;
        let mut i = 0usize;
        while i < upper_code.len() && code_len < 20 {
            let p = YUBIN_ASC_CHRS.iter().position(|&c| c == upper_code[i]);
            let p = match p {
                Some(p) => p,
                None => {
                    i += 1;
                    continue;
                }
            };

            let pattern = yubin_get_pattern(p);

            // Special case: at position 19, if pattern > 3 bars (alpha char), use CC1 only
            if code_len == 19 && pattern.len() > 3 {
                for ch in YUBIN_PTN_C[0].bytes() {
                    bars.push((ch - b'0') as i32);
                    chk_str.push(ch as char);
                }
                code_len += 1;
                break;
            }

            for ch in pattern.bytes() {
                bars.push((ch - b'0') as i32);
                chk_str.push(ch as char);
            }

            if pattern.len() <= 3 {
                code_len += 1;
            } else {
                code_len += 2;
            }
            i += 1;
        }

        // Pad with CC4 to fill 20 positions
        while code_len < 20 {
            for ch in YUBIN_PTN_C[3].bytes() {
                bars.push((ch - b'0') as i32);
            }
            chk_str.push_str(YUBIN_PTN_C[3]);
            code_len += 1;
        }

        // Check digit calculation (modulo 19)
        let mut chk_sum = 0;
        let chk_bytes = chk_str.as_bytes();
        let mut j = 0;
        while j + 3 <= chk_bytes.len() {
            let pat = &chk_str[j..j + 3];
            let value = yubin_pattern_to_check_value(pat);
            if value >= 0 {
                chk_sum += value;
            }
            j += 3;
        }

        let mut chk_d = 19 - (chk_sum % 19);
        if chk_d == 19 {
            chk_d = 0;
        }
        self.check_digit_value = chk_d;

        // Check digit bars
        let chk_pattern = yubin_get_check_pattern(chk_d);
        for ch in chk_pattern.bytes() {
            bars.push((ch - b'0') as i32);
        }

        // Stop code
        for ch in YUBIN_STOP.bytes() {
            bars.push((ch - b'0') as i32);
        }

        self.bars = bars.clone();
        Ok(bars)
    }

    /// Draws the YubinCustomer barcode (4-state bars).
    pub fn draw(&mut self, code: &str, height: i32) -> Result<()> {
        let bars = self.encode(code)?;
        if bars.is_empty() {
            return Err(BarcodeError::EncodingError("empty bar pattern".into()));
        }

        // Width is determined by bar count
        let width = (bars.len() as f64 * 3.0).ceil() as i32;

        if self.base_1d.base.is_svg_output() {
            return self.draw_svg_4state(&bars, width, height);
        }
        #[cfg(any(feature = "png", feature = "jpeg"))]
        {
            self.draw_png_4state(&bars, width, height)
        }
        #[cfg(not(any(feature = "png", feature = "jpeg")))]
        Err(BarcodeError::FormatError("PNG/JPEG features not enabled".into()))
    }

    fn draw_svg_4state(&mut self, bars: &[i32], width: i32, height: i32) -> Result<()> {
        let n = bars.len();
        if n == 0 {
            return Err(BarcodeError::EncodingError("empty bar pattern".into()));
        }
        let pitch = width as f64 / n as f64;
        let bar_w = (pitch * 0.45).round().max(1.0);
        let h3 = height as f64 / 3.0;

        self.base_1d.base.svg_begin(width, height);
        self.base_1d
            .base
            .svg_rect(0.0, 0.0, width as f64, height as f64, self.base_1d.base.back_color);

        for (i, &bt) in bars.iter().enumerate() {
            let bx = (i as f64 * pitch + (pitch - bar_w) / 2.0).round();
            let (by, bh) = match bt {
                1 => (0.0, height as f64),
                2 => (0.0, (h3 * 2.0).round()),
                3 => (h3.round(), (h3 * 2.0).round()),
                4 => (h3.round(), h3.round()),
                _ => continue,
            };
            self.base_1d
                .base
                .svg_rect(bx, by, bar_w, bh, self.base_1d.base.fore_color);
        }

        self.base_1d.base.svg_end();
        Ok(())
    }

    #[cfg(any(feature = "png", feature = "jpeg"))]
    fn draw_png_4state(&mut self, bars: &[i32], width: i32, height: i32) -> Result<()> {
        let n = bars.len();
        if n == 0 {
            return Err(BarcodeError::EncodingError("empty bar pattern".into()));
        }

        let bc = self.base_1d.base.back_color;
        let mut img = image::RgbaImage::from_pixel(
            width as u32,
            height as u32,
            image::Rgba([bc.r, bc.g, bc.b, bc.a]),
        );

        let pitch = width as f64 / n as f64;
        let bar_w = (pitch * 0.45).round().max(1.0) as i32;
        let h3 = height as f64 / 3.0;

        for (i, &bt) in bars.iter().enumerate() {
            let bx = (i as f64 * pitch + (pitch - bar_w as f64) / 2.0).round() as i32;
            let (by, bh) = match bt {
                1 => (0, height),
                2 => (0, (h3 * 2.0).round() as i32),
                3 => (h3.round() as i32, (h3 * 2.0).round() as i32),
                4 => (h3.round() as i32, h3.round() as i32),
                _ => continue,
            };
            let fc = self.base_1d.base.fore_color;
            crate::base_1d::fill_rect(
                &mut img,
                bx,
                by,
                bx + bar_w,
                by + bh,
                image::Rgba([fc.r, fc.g, fc.b, fc.a]),
            );
        }

        if crate::product_info::is_trial_mode() {
            #[cfg(feature = "font")]
            crate::base_1d::draw_sample_overlay_png(&mut img, 0, 0, width, height);
        }

        self.base_1d.base.encode_image(&img)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_basic() {
        let mut y = YubinCustomer::new("svg");
        let bars = y.encode("1050011").unwrap();
        assert!(!bars.is_empty());
    }

    #[test]
    fn test_encode_alpha() {
        let mut y = YubinCustomer::new("svg");
        let bars = y.encode("105-0011").unwrap();
        assert!(!bars.is_empty());
    }

    #[test]
    fn test_check_digit() {
        let mut y = YubinCustomer::new("svg");
        y.encode("1050011").unwrap();
        assert!(y.check_digit() >= 0 && y.check_digit() <= 18);
    }

    #[test]
    fn test_encode_empty() {
        let mut y = YubinCustomer::new("svg");
        assert!(y.encode("").is_err());
    }

    #[test]
    fn test_encode_with_letters() {
        let mut y = YubinCustomer::new("svg");
        let bars = y.encode("100-0001A").unwrap();
        assert!(!bars.is_empty());
    }
}
