use crate::base_1d::BarcodeBase1D;
use crate::error::{BarcodeError, Result};

const UPCE_OE: &[i32] = &[0x00, 0x0b, 0x0d, 0x0e, 0x13, 0x19, 0x1c, 0x15, 0x16, 0x1a];
const UPCE_LEFT_G: &[i32] = &[0x27, 0x33, 0x1b, 0x21, 0x1d, 0x39, 0x05, 0x11, 0x09, 0x17];
const UPCE_LEFT_L: &[i32] = &[0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b];

fn expand_to_upca(code: &str) -> Option<String> {
    if code.len() != 7 {
        return None;
    }
    let bytes = code.as_bytes();
    let c0 = bytes[0];
    let c6 = bytes[6];

    if c0 == b'0' && (c6 == b'0' || c6 == b'1' || c6 == b'2') {
        Some(format!(
            "{}{}0000{}",
            &code[..3],
            c6 as char,
            &code[3..6]
        ))
    } else if c0 == b'0' && c6 == b'3' {
        Some(format!("{}00000{}", &code[..4], &code[4..6]))
    } else if c0 == b'0' && c6 == b'4' {
        Some(format!("{}00000{}", &code[..5], &code[5..6]))
    } else if c0 == b'0' && (b'5'..=b'9').contains(&c6) {
        Some(format!("{}0000{}", &code[..6], c6 as char))
    } else {
        None
    }
}

/// Calculates the UPC-E check digit via UPC-A expansion.
pub fn calculate_check_digit_upce(src: &str) -> String {
    let expanded = if src.len() == 7 {
        match expand_to_upca(src) {
            Some(e) => e,
            None => return String::new(),
        }
    } else {
        src.to_string()
    };

    let mut odds = 0;
    let mut evens = 0;
    let mut odd = true;
    let bytes = expanded.as_bytes();
    for i in (1..=bytes.len()).rev() {
        let n = (bytes[i - 1] - b'0') as i32;
        if odd {
            odds += n;
        } else {
            evens += n;
        }
        odd = !odd;
    }

    let s = odds * 3 + evens;
    let mut cd = s % 10;
    if cd != 0 {
        cd = 10 - cd;
    }
    cd.to_string()
}

/// UPC-E barcode encoder.
pub struct UPCE {
    pub base_1d: BarcodeBase1D,
    pub extended_guard: bool,
}

impl UPCE {
    pub fn new(output_format: &str) -> Self {
        Self {
            base_1d: BarcodeBase1D::new(output_format),
            extended_guard: true,
        }
    }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        let pattern = encode_upce(code)?;
        let mut display_code = code.to_string();
        if display_code.len() == 6 {
            display_code = format!("0{}", display_code);
            display_code = format!("{}{}", display_code, calculate_check_digit_upce(&display_code));
        } else if display_code.len() == 7 {
            display_code = format!("{}{}", display_code, calculate_check_digit_upce(&display_code));
        }

        if self.base_1d.base.is_svg_output() {
            if self.base_1d.show_text && self.extended_guard {
                return super::jan_upc_render::draw_svg_upce(
                    &mut self.base_1d, &pattern, &display_code, width, height,
                );
            }
            let display = if self.base_1d.show_text {
                &display_code
            } else {
                ""
            };
            return self.base_1d.draw_svg_bars(&pattern, width, height, display);
        }
        super::jan_upc_render::draw_png_jan_upc(
            &mut self.base_1d, &pattern, &display_code, width, height, "UPCE",
        )
    }
}

pub fn encode_upce(code: &str) -> Result<Vec<i32>> {
    if code.is_empty() {
        return Err(BarcodeError::EmptyCode);
    }

    let mut display_code = code.to_string();
    if display_code.len() == 6 {
        display_code = format!("0{}", display_code);
        display_code = format!(
            "{}{}",
            display_code,
            calculate_check_digit_upce(&display_code)
        );
    } else if display_code.len() == 7 {
        display_code = format!(
            "{}{}",
            display_code,
            calculate_check_digit_upce(&display_code)
        );
    } else if display_code.len() != 8 {
        return Err(BarcodeError::InvalidData(
            "UPC-E requires 6, 7 or 8 digits".into(),
        ));
    }

    for c in display_code.chars() {
        if !c.is_ascii_digit() {
            return Err(BarcodeError::InvalidData(
                "UPC-E barcode requires numeric digits only".into(),
            ));
        }
    }

    encode_upce_internal(&display_code)
}

fn encode_upce_internal(src: &str) -> Result<Vec<i32>> {
    if src.len() != 8 {
        return Err(BarcodeError::EncodingError(
            "EncodeUPCE: code must be 8 digits".into(),
        ));
    }

    let cd = (src.as_bytes()[7] - b'0') as usize;
    let data = &src[1..7];

    let pre = cd;
    let chk_oe: i32 = 0x20;

    // Build module sequence
    let mut modules = Vec::new();

    // Start guard "101"
    modules.extend_from_slice(&[1, 0, 1]);

    // Data 6 digits
    for (i, c) in data.bytes().enumerate() {
        let d = (c - b'0') as usize;
        let flg_oe = if (UPCE_OE[pre] & (chk_oe >> i as i32)) != 0 {
            1
        } else {
            0
        };
        let bits = if flg_oe == 0 {
            UPCE_LEFT_G[d]
        } else {
            UPCE_LEFT_L[d]
        };
        let mask: i32 = 0x40;
        for j in 0..7 {
            if (bits & (mask >> j)) != 0 {
                modules.push(1);
            } else {
                modules.push(0);
            }
        }
    }

    // End guard "010101"
    for j in 0..6 {
        if j % 2 == 0 {
            modules.push(0);
        } else {
            modules.push(1);
        }
    }

    // Run-length encode
    if modules.is_empty() || modules[0] != 1 {
        return Err(BarcodeError::EncodingError(
            "EncodeUPCE: internal module sequence error".into(),
        ));
    }

    let mut pattern = Vec::new();
    let mut current = modules[0];
    let mut count = 1;

    for &m in &modules[1..] {
        if m == current {
            count += 1;
        } else {
            pattern.push(count);
            current = m;
            count = 1;
        }
    }
    pattern.push(count);

    Ok(pattern)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_6_digits() {
        let pattern = encode_upce("123456").unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_8_digits() {
        let pattern = encode_upce("01234565").unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_empty() {
        assert!(encode_upce("").is_err());
    }
}
