//! Shared RSS/DataBar utility algorithms.
//! Common algorithms used by GS1 DataBar 14, DataBar Limited, and DataBar Expanded.

/// Calculates nCr (binomial coefficient).
pub fn get_combinations(n: i32, r: i32) -> i32 {
    let (min_denom, max_denom) = if n - r > r {
        (r, n - r)
    } else {
        (n - r, r)
    };

    let mut val: i32 = 1;
    let mut j = 1;
    for i in (max_denom + 1..=n).rev() {
        val *= i;
        if j <= min_denom {
            val /= j;
            j += 1;
        }
    }
    while j <= min_denom {
        val /= j;
        j += 1;
    }
    val
}

/// RSS subset width algorithm (ISO 24724).
/// Converts a value into an element width pattern.
pub fn get_widths(
    mut val: i32,
    mut n: i32,
    elements: i32,
    max_width: i32,
    no_narrow: i32,
) -> Vec<i32> {
    let mut widths = vec![0i32; elements as usize];
    let mut narrow_mask: i32 = 0;

    for bar in 0..elements - 1 {
        let mut elm_width = 1;
        narrow_mask |= 1 << bar;
        let mut sub_val;

        loop {
            sub_val = get_combinations(n - elm_width - 1, elements - bar - 2);

            if no_narrow == 0
                && narrow_mask == 0
                && n - elm_width - (elements - bar - 1) >= elements - bar - 1
            {
                sub_val -=
                    get_combinations(n - elm_width - (elements - bar), elements - bar - 2);
            }

            if elements - bar - 1 > 1 {
                let mut less_val = 0;
                let mut mxw_element = n - elm_width - (elements - bar - 2);
                while mxw_element > max_width {
                    less_val += get_combinations(
                        n - elm_width - mxw_element - 1,
                        elements - bar - 3,
                    );
                    mxw_element -= 1;
                }
                sub_val -= less_val * (elements - 1 - bar);
            } else if n - elm_width > max_width {
                sub_val -= 1;
            }

            val -= sub_val;
            if val < 0 {
                break;
            }

            elm_width += 1;
            narrow_mask &= !(1 << bar);
        }

        val += sub_val;
        n -= elm_width;
        widths[bar as usize] = elm_width;
    }

    widths[(elements - 1) as usize] = n;
    widths
}

/// Calculates GS1 GTIN-14 check digit from 13 digits.
pub fn calculate_gtin_check_digit(src13: &str) -> i32 {
    let bytes = src13.as_bytes();
    let mut total = 0;
    for (i, &b) in bytes.iter().enumerate().take(13) {
        let weight = if i % 2 != 0 { 1 } else { 3 };
        total += (b - b'0') as i32 * weight;
    }
    (10 - (total % 10)) % 10
}

/// Converts a binary string to run-length encoded pattern string.
pub fn bin2pat(binary: &str) -> String {
    if binary.is_empty() {
        return String::new();
    }
    let bytes = binary.as_bytes();
    let mut result = Vec::new();
    let mut count: u8 = 1;
    let mut last_char = bytes[0];

    for &b in &bytes[1..] {
        if b == last_char {
            count += 1;
        } else {
            result.push(b'0' + count);
            count = 1;
            last_char = b;
        }
    }
    result.push(b'0' + count);
    String::from_utf8(result).unwrap_or_default()
}

/// Left-pads a string with '0' to the given length.
pub fn pad_left(s: &str, length: usize) -> String {
    if s.len() >= length {
        return s.to_string();
    }
    let mut result = "0".repeat(length - s.len());
    result.push_str(s);
    result
}

/// Parses a numeric string to i64.
pub fn parse_int64(s: &str) -> i64 {
    let mut v: i64 = 0;
    for c in s.bytes() {
        v = v * 10 + (c - b'0') as i64;
    }
    v
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_combinations() {
        assert_eq!(get_combinations(5, 2), 10);
        assert_eq!(get_combinations(10, 3), 120);
    }

    #[test]
    fn test_gtin_check_digit() {
        let cd = calculate_gtin_check_digit("0000000000000");
        assert!(cd >= 0 && cd <= 9);
    }

    #[test]
    fn test_bin2pat() {
        assert_eq!(bin2pat("111001110"), "3231");
        assert_eq!(bin2pat("10110"), "1121");
        assert_eq!(bin2pat(""), "");
        assert_eq!(bin2pat("1"), "1");
        assert_eq!(bin2pat("1100011"), "232");
    }

    #[test]
    fn test_pad_left() {
        assert_eq!(pad_left("123", 6), "000123");
        assert_eq!(pad_left("123456", 6), "123456");
    }

    #[test]
    fn test_parse_int64() {
        assert_eq!(parse_int64("12345"), 12345);
        assert_eq!(parse_int64("0"), 0);
    }
}
