//! QR Code encoder — full port of the Go/C++ QR engine.
//!
//! Supports versions 1-40 (auto or manual), four error correction levels,
//! and numeric / alphanumeric / binary encoding modes.

use crate::base_2d::BarcodeBase2D;
use crate::error::{BarcodeError, Result};

// ── Public constants ────────────────────────────────────────────────────────

/// Error correction level: Low (~7% recovery).
pub const QR_ECC_L: i32 = 0;
/// Error correction level: Medium (~15% recovery).
pub const QR_ECC_M: i32 = 1;
/// Error correction level: Quartile (~25% recovery).
pub const QR_ECC_Q: i32 = 2;
/// Error correction level: High (~30% recovery).
pub const QR_ECC_H: i32 = 3;

/// Encoding mode: Numeric.
pub const QR_MODE_NUMERIC: &str = "N";
/// Encoding mode: Alphanumeric.
pub const QR_MODE_ALPHANUMERIC: &str = "A";
/// Encoding mode: Binary (8-bit byte).
pub const QR_MODE_BINARY: &str = "B";
/// Encoding mode: Kanji.
pub const QR_MODE_KANJI: &str = "K";

// ── Internal text-type constants ────────────────────────────────────────────

const QR_TEXT_AUTO: i32 = 0;
const QR_TEXT_NUMERIC: i32 = 1;
const QR_TEXT_ALPHA_NUMERIC: i32 = 2;
// const QR_TEXT_KANJI: i32 = 3; // kept for reference only
const QR_TEXT_BINARY: i32 = 4;

// ── Internal ECC rate constants ─────────────────────────────────────────────

const QR_ECC_MEDIUM15: i32 = 1;

// ── Lookup tables ───────────────────────────────────────────────────────────

/// Data words per RS block [ecc_level][version-1].
const G_DATA_WORDS: [[i32; 40]; 4] = [
    [19,34,55,80,108,68,78,97,116,68,81,92,107,115,87,98,107,120,113,107,
     116,111,121,117,106,114,122,117,116,115,115,115,115,115,121,121,122,122,117,118],
    [16,28,44,32,43,27,31,38,36,43,50,36,37,40,41,45,46,43,44,41,
     42,46,47,45,47,46,45,45,45,47,46,46,46,46,47,47,46,46,47,47],
    [13,22,17,24,15,19,14,18,16,19,22,20,20,16,24,19,22,22,21,24,
     22,24,24,24,24,22,23,24,23,24,24,24,24,24,24,24,24,24,24,24],
    [9,16,13,9,11,15,13,14,12,15,12,14,11,12,12,15,14,14,13,15,
     16,13,15,16,15,16,15,15,15,15,15,15,15,16,15,15,15,15,15,15],
];

/// Extra data words per RS block [ecc_level][version-1].
const G_EXTRA_DATA_WORDS: [[i32; 40]; 4] = [
    [0,0,0,0,0,0,0,0,0,2,0,2,0,1,1,1,5,1,4,5,
     4,7,5,4,4,2,4,10,7,10,3,0,1,6,7,14,4,18,4,6],
    [0,0,0,0,0,0,0,2,2,1,4,2,1,5,5,3,1,4,11,13,
     0,0,14,14,13,4,3,23,7,10,29,23,21,23,26,34,14,32,7,31],
    [0,0,0,0,2,0,4,2,4,2,4,6,4,5,7,2,15,1,4,5,
     6,16,14,16,22,6,26,31,37,25,1,35,19,7,14,10,10,14,22,34],
    [0,0,0,0,2,0,1,2,4,2,8,4,4,5,7,13,17,19,16,10,
     6,0,14,2,13,4,28,31,26,25,28,35,46,1,41,64,46,32,67,61],
];

/// ECC words per RS block [ecc_level][version-1].
const G_ECC_WORDS: [[i32; 40]; 4] = [
    [7,10,15,20,26,18,20,24,30,18,20,24,26,30,22,24,28,30,28,28,
     28,28,30,30,26,28,30,30,30,30,30,30,30,30,30,30,30,30,30,30],
    [10,16,26,18,24,16,18,22,22,26,30,22,22,24,24,28,28,26,26,26,
     26,28,28,28,28,28,28,28,28,28,28,28,28,28,28,28,28,28,28,28],
    [13,22,18,26,18,24,18,22,20,24,28,26,24,20,30,24,28,28,26,30,
     28,30,30,30,30,28,30,30,30,30,30,30,30,30,30,30,30,30,30,30],
    [17,28,22,16,22,28,26,26,24,28,24,28,22,24,24,30,28,28,26,28,
     30,24,30,30,30,30,30,30,30,30,30,30,30,30,30,30,30,30,30,30],
];

/// RS block count [ecc_level][version-1].
const G_RS_BLOCK_COUNT: [[i32; 40]; 4] = [
    [1,1,1,1,1,2,2,2,2,4,4,4,4,4,6,6,6,6,7,8,
     8,9,9,10,12,12,12,13,14,15,16,17,18,19,19,20,21,22,24,25],
    [1,1,1,2,2,4,4,4,5,5,5,8,9,9,10,10,11,13,14,16,
     17,17,18,20,21,23,25,26,28,29,31,33,35,37,38,40,43,45,47,49],
    [1,1,2,2,4,4,6,6,8,8,8,10,12,16,12,17,16,18,21,20,
     23,23,25,27,29,34,34,35,38,40,43,45,48,51,53,56,59,62,65,68],
    [1,1,2,4,4,4,5,6,8,8,11,11,16,16,18,16,19,21,25,25,
     25,34,30,32,35,37,40,42,45,48,51,54,57,60,63,66,70,74,77,81],
];

/// Alignment pattern centre positions by version (index 0 unused).
const QR_ALIGNMENT_PATTERNS: [&[i32]; 41] = [
    &[],                                // 0 — unused
    &[6],                               // 1
    &[6,18],                            // 2
    &[6,22],                            // 3
    &[6,26],                            // 4
    &[6,30],                            // 5
    &[6,34],                            // 6
    &[6,22,38],                         // 7
    &[6,24,42],                         // 8
    &[6,26,46],                         // 9
    &[6,28,50],                         // 10
    &[6,30,54],                         // 11
    &[6,32,58],                         // 12
    &[6,34,62],                         // 13
    &[6,26,46,66],                      // 14
    &[6,26,48,70],                      // 15
    &[6,26,50,74],                      // 16
    &[6,30,54,78],                      // 17
    &[6,30,56,82],                      // 18
    &[6,30,58,86],                      // 19
    &[6,34,62,90],                      // 20
    &[6,28,50,72,94],                   // 21
    &[6,26,50,74,98],                   // 22
    &[6,30,54,78,102],                  // 23
    &[6,28,54,80,106],                  // 24
    &[6,32,58,84,110],                  // 25
    &[6,30,58,86,114],                  // 26
    &[6,34,62,90,118],                  // 27
    &[6,26,50,74,98,122],              // 28
    &[6,30,54,78,102,126],             // 29
    &[6,26,52,78,104,130],             // 30
    &[6,30,56,82,108,134],             // 31
    &[6,34,60,86,112,138],             // 32
    &[6,30,58,86,114,142],             // 33
    &[6,34,62,90,118,146],             // 34
    &[6,30,54,78,102,126,150],         // 35
    &[6,24,50,76,102,128,154],         // 36
    &[6,28,54,80,106,132,158],         // 37
    &[6,32,58,84,110,136,162],         // 38
    &[6,26,54,82,110,138,166],         // 39
    &[6,30,58,86,114,142,170],         // 40
];

/// Version information bit patterns (versions 7-40, index 0 = version 7).
const QR_VERSION_INFO: [i32; 34] = [
    31892, 34236, 39577, 42195, 48118, 51042, 55367, 58893,
    63784, 68472, 70749, 76311, 79154, 84390, 87683, 92361,
    96236, 102084, 102881, 110507, 110734, 117786, 119615, 126325,
    127568, 133589, 136944, 141498, 145311, 150283, 152622, 158308,
    161089, 167017,
];

/// GF(256) exponentiation table.
const GF_EXP: [i32; 256] = [
    1,2,4,8,16,32,64,128,29,58,116,232,205,135,19,38,76,152,45,90,
    180,117,234,201,143,3,6,12,24,48,96,192,157,39,78,156,37,74,148,53,
    106,212,181,119,238,193,159,35,70,140,5,10,20,40,80,160,93,186,105,210,
    185,111,222,161,95,190,97,194,153,47,94,188,101,202,137,15,30,60,120,240,
    253,231,211,187,107,214,177,127,254,225,223,163,91,182,113,226,217,175,67,134,
    17,34,68,136,13,26,52,104,208,189,103,206,129,31,62,124,248,237,199,147,
    59,118,236,197,151,51,102,204,133,23,46,92,184,109,218,169,79,158,33,66,
    132,21,42,84,168,77,154,41,82,164,85,170,73,146,57,114,228,213,183,115,
    230,209,191,99,198,145,63,126,252,229,215,179,123,246,241,255,227,219,171,75,
    150,49,98,196,149,55,110,220,165,87,174,65,130,25,50,100,200,141,7,14,
    28,56,112,224,221,167,83,166,81,162,89,178,121,242,249,239,195,155,43,86,
    172,69,138,9,18,36,72,144,61,122,244,245,247,243,251,235,203,139,11,22,
    44,88,176,125,250,233,207,131,27,54,108,216,173,71,142,1,
];

/// GF(256) logarithm table.
const GF_LOG: [i32; 256] = [
    0,0,1,25,2,50,26,198,3,223,51,238,27,104,199,75,4,100,224,14,
    52,141,239,129,28,193,105,248,200,8,76,113,5,138,101,47,225,36,15,33,
    53,147,142,218,240,18,130,69,29,181,194,125,106,39,249,185,201,154,9,120,
    77,228,114,166,6,191,139,98,102,221,48,253,226,152,37,179,16,145,34,136,
    54,208,148,206,143,150,219,189,241,210,19,92,131,56,70,64,30,66,182,163,
    195,72,126,110,107,58,40,84,250,133,186,61,202,94,155,159,10,21,121,43,
    78,212,229,172,115,243,167,87,7,112,192,247,140,128,99,13,103,74,222,237,
    49,197,254,24,227,165,153,119,38,184,180,124,17,68,146,217,35,32,137,46,
    55,63,209,91,149,188,207,205,144,135,151,178,220,252,190,97,242,86,211,171,
    20,42,93,158,132,60,57,83,71,109,65,162,31,45,67,216,183,123,164,118,
    196,23,73,236,127,12,111,246,108,161,59,82,41,157,85,170,251,96,134,177,
    187,204,62,90,203,89,95,176,156,169,160,81,11,245,22,235,122,117,44,215,
    79,174,213,233,230,231,173,232,116,214,244,234,168,80,88,175,
];

/// Generator polynomials keyed by ECC word count.
/// Returns `None` when the key is not one of the known sizes.
fn qr_gen_poly(ecc: i32) -> Option<&'static [i32]> {
    match ecc {
        7  => Some(&[127,122,154,164,11,68,117]),
        10 => Some(&[216,194,159,111,199,94,95,113,157,193]),
        13 => Some(&[137,73,227,17,177,17,52,13,46,43,83,132,120]),
        15 => Some(&[29,196,111,163,112,74,10,105,105,139,132,151,32,134,26]),
        16 => Some(&[59,13,104,189,68,209,30,8,163,65,41,229,98,50,36,59]),
        17 => Some(&[119,66,83,120,119,22,197,83,249,41,143,134,85,53,125,99,79]),
        18 => Some(&[239,251,183,113,149,175,199,215,240,220,73,82,173,75,32,67,217,146]),
        20 => Some(&[152,185,240,5,111,99,6,220,112,150,69,36,187,22,228,198,121,121,165,174]),
        22 => Some(&[89,179,131,176,182,244,19,189,69,40,28,137,29,123,67,253,86,218,230,26,145,245]),
        24 => Some(&[122,118,169,70,178,237,216,102,115,150,229,73,130,72,61,43,206,1,237,247,127,217,144,117]),
        26 => Some(&[246,51,183,4,136,98,199,152,77,56,206,24,145,40,209,117,233,42,135,68,70,144,146,77,43,94]),
        28 => Some(&[252,9,28,13,18,251,208,150,103,174,100,41,167,12,247,56,117,119,233,127,181,100,121,147,176,74,58,197]),
        30 => Some(&[212,246,77,73,195,192,75,98,5,70,103,177,22,217,138,51,181,246,72,25,18,46,228,74,216,195,11,106,130,150]),
        _  => None,
    }
}

/// Module capacity by version (index 0 unused).
const QR_ARRAY_CAP: [i32; 41] = [
    0,26,44,70,100,134,172,196,242,292,
    346,404,466,532,581,655,733,815,901,991,
    1085,1156,1258,1364,1474,1588,1706,1828,1921,2051,
    2185,2323,2465,2611,2761,2876,3034,3196,3362,3532,
    3706,
];

/// Extra bits by version (index 0 unused).
const QR_ARRAY2_EXTRA: [i32; 41] = [
    0,0,7,7,7,7,7,0,0,0,
    0,0,0,0,3,3,3,3,3,3,
    3,4,4,4,4,4,4,4,3,3,
    3,3,3,3,3,0,0,0,0,0,
    0,
];

/// Bit capacity by ECC level and version [ecc][version].
const QR_ARRAY3_CAP: [[i32; 41]; 4] = [
    [0,152,272,440,640,864,1088,1248,1552,1856,
     2192,2592,2960,3424,3688,4184,4712,5176,5768,6360,
     6888,7456,8048,8752,9392,10208,10960,11744,12248,13048,
     13880,14744,15640,16568,17528,18448,19472,20528,21616,22496,
     23648],
    [0,128,224,352,512,688,864,992,1232,1456,
     1728,2032,2320,2672,2920,3320,3624,4056,4504,5016,
     5352,5712,6256,6880,7312,8000,8496,9024,9544,10136,
     10984,11640,12328,13048,13800,14496,15312,15936,16816,17728,
     18672],
    [0,104,176,272,384,496,608,704,880,1056,
     1232,1440,1648,1952,2088,2360,2600,2936,3176,3560,
     3880,4096,4544,4912,5312,5744,6032,6464,6968,7288,
     7880,8264,8920,9368,9848,10288,10832,11408,12016,12656,
     13328],
    [0,72,128,208,288,368,480,528,688,800,
     976,1120,1264,1440,1576,1784,2024,2264,2504,2728,
     3080,3248,3536,3712,4112,4304,4768,5024,5288,5608,
     5960,6344,6760,7208,7688,7888,8432,8768,9136,9776,
     10208],
];

/// Format information strings (32 patterns).
const QR_FORMAT_INFO: [&str; 32] = [
    "111011111000100","111001011110011","111110110101010","111100010011101",
    "110011000101111","110001100011000","110110001000001","110100101110110",
    "101010000010010","101000100100101","101111001111100","101101101001011",
    "100010111111001","100000011001110","100111110010111","100101010100000",
    "011010101011111","011000001101000","011111100110001","011101000000110",
    "010010010110100","010000110000011","010111011011010","010101111101101",
    "001011010001001","001001110111110","001110011100111","001100111010000",
    "000011101100010","000001001010101","000110100001100","000100000111011",
];

// ── Public QR struct ────────────────────────────────────────────────────────

/// QR Code encoder.
///
/// # Example
/// ```rust,no_run
/// use barcode_pao::encoders::qr::{QR, QR_ECC_M};
///
/// let mut qr = QR::new("svg");
/// qr.error_correction = QR_ECC_M as i32;
/// qr.draw("Hello", 200).unwrap();
/// let svg = qr.base_2d.base.get_svg().unwrap();
/// ```
pub struct QR {
    pub base_2d: BarcodeBase2D,
    /// QR version (1-40), or 0 for AUTO.
    pub version: i32,
    /// Error correction level (0=L, 1=M, 2=Q, 3=H).
    pub error_correction: i32,
    /// Encode mode: `"N"`, `"A"`, `"B"`, or `"K"`.
    encode_mode: String,
}

impl QR {
    /// Creates a new QR encoder.
    pub fn new(output_format: &str) -> Self {
        Self {
            base_2d: BarcodeBase2D::new(output_format),
            version: 0,
            error_correction: QR_ECC_M,
            encode_mode: QR_MODE_BINARY.to_string(),
        }
    }

    /// Sets the encoding mode (`"N"`, `"A"`, `"B"`, `"K"`).
    pub fn set_encode_mode(&mut self, mode: &str) {
        match mode {
            QR_MODE_NUMERIC | QR_MODE_ALPHANUMERIC | QR_MODE_BINARY | QR_MODE_KANJI => {
                self.encode_mode = mode.to_string();
            }
            _ => {
                self.encode_mode = QR_MODE_BINARY.to_string();
            }
        }
    }

    /// Returns the current encode mode.
    pub fn encode_mode(&self) -> &str {
        &self.encode_mode
    }

    /// Draws the QR code into the internal buffer.
    /// `size` is used for both width and height (square).
    pub fn draw(&mut self, code: &str, size: i32) -> Result<()> {
        let data = code.as_bytes();
        if data.is_empty() {
            return Err(BarcodeError::EmptyCode);
        }

        let mut jam = self.create_jam();
        jam.make_arr(data)?;
        let pattern = &jam.patt;
        if pattern.is_empty() {
            return Err(BarcodeError::EncodingError(
                "QR encoding produced empty pattern".into(),
            ));
        }

        if self.base_2d.base.is_svg_output() {
            self.draw_svg(pattern, size, size)
        } else {
            self.draw_png(pattern, size, size)
        }
    }

    /// Generates and returns the QR code module pattern as a 2D bool grid.
    pub fn get_pattern(&mut self, code: &str) -> Result<Vec<Vec<bool>>> {
        let data = code.as_bytes();
        if data.is_empty() {
            return Err(BarcodeError::EmptyCode);
        }
        let mut jam = self.create_jam();
        jam.make_arr(data)?;
        Ok(jam.patt)
    }

    fn create_jam(&self) -> QrJam {
        let mut jam = QrJam::new();
        jam.set_encode_mode(&self.encode_mode);
        jam.set_version(self.version);
        jam.set_ecc_rate(self.error_correction);
        jam
    }

    // ── SVG rendering ───────────────────────────────────────────────────

    fn draw_svg(&mut self, pattern: &[Vec<bool>], width: i32, height: i32) -> Result<()> {
        let pattern_size = pattern.len();
        if pattern_size == 0 {
            return Err(BarcodeError::EncodingError("empty pattern".into()));
        }

        let base = &mut self.base_2d.base;
        base.svg_begin(width, height);
        base.svg_rect(0.0, 0.0, width as f64, height as f64, base.back_color);

        let (module_w, module_h) = if base.fit_width {
            (
                width as f64 / pattern_size as f64,
                height as f64 / pattern_size as f64,
            )
        } else {
            let mw = (width / pattern_size as i32) as f64;
            let mh = (height / pattern_size as i32) as f64;
            (mw.max(1.0), mh.max(1.0))
        };

        let adjust_x = base.px_adj_black as f64;
        let adjust_y = base.px_adj_black as f64;

        let fore = base.fore_color;
        for r in 0..pattern_size {
            for c in 0..pattern_size {
                if c < pattern.len() && r < pattern[c].len() && pattern[c][r] {
                    let draw_x = c as f64 * module_w;
                    let draw_y = r as f64 * module_h;
                    let draw_w = (module_w + adjust_x + 0.5).max(0.0);
                    let draw_h = (module_h + adjust_y + 0.5).max(0.0);
                    base.svg_rect(draw_x, draw_y, draw_w, draw_h, fore);
                }
            }
        }

        base.svg_end();
        Ok(())
    }

    // ── PNG rendering ───────────────────────────────────────────────────

    #[cfg(any(feature = "png", feature = "jpeg"))]
    fn draw_png(&mut self, pattern: &[Vec<bool>], width: i32, height: i32) -> Result<()> {
        let pattern_size = pattern.len();
        if pattern_size == 0 {
            return Err(BarcodeError::EncodingError("empty pattern".into()));
        }

        let bc = self.base_2d.base.back_color;
        let mut img = image::RgbaImage::from_pixel(
            width as u32,
            height as u32,
            image::Rgba([bc.r, bc.g, bc.b, bc.a]),
        );

        let module_w = width as f64 / pattern_size as f64;
        let module_h = height as f64 / pattern_size as f64;

        let fc = self.base_2d.base.fore_color;
        let fore_px = image::Rgba([fc.r, fc.g, fc.b, fc.a]);

        for r in 0..pattern_size {
            for c in 0..pattern_size {
                if c < pattern.len() && r < pattern[c].len() && pattern[c][r] {
                    let x1 = (c as f64 * module_w) as i32;
                    let y1 = (r as f64 * module_h) as i32;
                    let x2 = ((c + 1) as f64 * module_w) as i32;
                    let y2 = ((r + 1) as f64 * module_h) as i32;
                    crate::base_1d::fill_rect(&mut img, x1, y1, x2, y2, fore_px);
                }
            }
        }

        if crate::product_info::is_trial_mode() {
            #[cfg(feature = "font")]
            crate::base_1d::draw_sample_overlay_png(&mut img, 0, 0, width, height);
        }

        self.base_2d.base.encode_image(&img)
    }

    #[cfg(not(any(feature = "png", feature = "jpeg")))]
    fn draw_png(&mut self, _pattern: &[Vec<bool>], _width: i32, _height: i32) -> Result<()> {
        Err(BarcodeError::FormatError(
            "PNG/JPEG support requires 'png' or 'jpeg' feature".into(),
        ))
    }
}

// ══════════════════════════════════════════════════════════════════════════════
// Internal QR generation engine
// ══════════════════════════════════════════════════════════════════════════════

struct QrJam {
    l_format_first_x: [i32; 15],
    l_format_first_y: [i32; 15],
    l_format_second_x: [i32; 15],
    l_format_second_y: [i32; 15],

    l_deploy_table_x: Vec<i32>,
    l_deploy_table_y: Vec<i32>,
    l_deploy_order_x: Vec<i32>,
    l_deploy_order_y: Vec<i32>,
    l_rs_block: Vec<i32>,

    l_qr_set_version: i32,
    l_qr_set_ecc_rate: i32,
    l_qr_set_text_type: i32,
    _l_qr_quit_zone: i32,

    l_qr_inquire_version: i32,
    l_qr_inquire_text_type: i32,
    l_qr_inquire_numeric_bits: i32,
    _l_qr_inquire_alpha_numeric_bits: i32,
    _l_qr_inquire_kanji_bits: i32,
    l_qr_inquire_binary_bits: i32,
    l_qr_inquire_module_size: i32,

    encode_mode: String,

    patt: Vec<Vec<bool>>,
}

impl QrJam {
    fn new() -> Self {
        Self {
            l_format_first_x: [0, 1, 2, 3, 4, 5, 7, 8, 8, 8, 8, 8, 8, 8, 8],
            l_format_first_y: [8, 8, 8, 8, 8, 8, 8, 8, 7, 5, 4, 3, 2, 1, 0],
            l_format_second_x: [0; 15],
            l_format_second_y: [0; 15],

            l_deploy_table_x: Vec::new(),
            l_deploy_table_y: Vec::new(),
            l_deploy_order_x: Vec::new(),
            l_deploy_order_y: Vec::new(),
            l_rs_block: Vec::new(),

            l_qr_set_version: 0,
            l_qr_set_ecc_rate: QR_ECC_MEDIUM15,
            l_qr_set_text_type: QR_TEXT_AUTO,
            _l_qr_quit_zone: 4,

            l_qr_inquire_version: 0,
            l_qr_inquire_text_type: 0,
            l_qr_inquire_numeric_bits: 0,
            _l_qr_inquire_alpha_numeric_bits: 0,
            _l_qr_inquire_kanji_bits: 0,
            l_qr_inquire_binary_bits: 0,
            l_qr_inquire_module_size: 0,

            encode_mode: "Z".to_string(),

            patt: Vec::new(),
        }
    }

    fn set_encode_mode(&mut self, mode: &str) {
        self.encode_mode = mode.to_string();
        match mode {
            "N" => self.l_qr_set_text_type = QR_TEXT_NUMERIC,
            "A" => self.l_qr_set_text_type = QR_TEXT_ALPHA_NUMERIC,
            _ => self.l_qr_set_text_type = QR_TEXT_BINARY,
        }
    }

    fn set_version(&mut self, version: i32) {
        if (0..=40).contains(&version) {
            self.l_qr_set_version = version;
        }
    }

    fn set_ecc_rate(&mut self, rate: i32) {
        self.l_qr_set_ecc_rate = rate;
    }

    // ── Helpers ─────────────────────────────────────────────────────────

    fn qr_code_get_format_x(&self, p_version: i32) -> [i32; 15] {
        let mut arr = [0i32; 15];
        arr[..7].fill(8);
        for (i, slot) in arr[7..15].iter_mut().enumerate() {
            *slot = 4 * p_version + 9 + i as i32;
        }
        arr
    }

    fn qr_code_get_format_y(&self, p_version: i32) -> [i32; 15] {
        let mut arr = [0i32; 15];
        arr[7..15].fill(8);
        for (i, slot) in arr[..7].iter_mut().enumerate() {
            *slot = 4 * p_version + 16 - i as i32;
        }
        arr
    }

    fn qr_code_inquire_bits(&mut self, data: &[u8]) {
        let length = data.len() as i32;

        if self.encode_mode == "N" {
            let rem = length % 3;
            let base = (length / 3) * 10;
            self.l_qr_inquire_numeric_bits = if rem == 0 {
                base
            } else if rem == 1 {
                base + 4
            } else {
                base + 7
            };
            self._l_qr_inquire_alpha_numeric_bits = 0;
            self._l_qr_inquire_kanji_bits = 0;
        } else if self.encode_mode == "A" {
            let rem = length % 2;
            self._l_qr_inquire_alpha_numeric_bits = if rem == 0 {
                (length / 2) * 11
            } else {
                (length / 2) * 11 + 6
            };
            self.l_qr_inquire_numeric_bits = 0;
            self._l_qr_inquire_kanji_bits = 0;
        } else {
            self.l_qr_inquire_numeric_bits = 0;
            self._l_qr_inquire_alpha_numeric_bits = 0;
            self._l_qr_inquire_kanji_bits = 0;
        }
        self.l_qr_inquire_binary_bits = length * 8;
    }

    fn qr_code_set_inquire_text_type(&self) -> i32 {
        if self.encode_mode == "N" {
            QR_TEXT_NUMERIC
        } else if self.encode_mode == "A" {
            QR_TEXT_ALPHA_NUMERIC
        } else {
            QR_TEXT_BINARY
        }
    }

    // ── Symbol data (finder, alignment, timing, version info) ───────────

    fn qr_code_get_symbol_data(&mut self, p_version: i32) -> Vec<i32> {
        let num = (p_version * 4 + 17) as usize;
        let mut array3 = vec![0i32; num * (num + 1)];
        let array = [3i32, num as i32 - 4];

        let ap = if (1..=40).contains(&p_version) {
            QR_ALIGNMENT_PATTERNS[p_version as usize]
        } else {
            &[0]
        };

        // Finder patterns
        for jj in 0..array.len() {
            for kk in 0..array.len() {
                let num8 = array[jj];
                let num9 = array[kk];
                if jj == array.len() - 1 && kk == array.len() - 1 {
                    continue;
                }

                // Centre and immediate neighbours
                for dx in -1..=1i32 {
                    for dy in -1..=1i32 {
                        let idx = ga(num8 + dx, num9 + dy, num as i32);
                        array3[idx] = 3;
                    }
                }

                // Outer ring
                for i in -3..=3i32 {
                    array3[ga(num8 + i, num9 - 3, num as i32)] = 3;
                    array3[ga(num8 + i, num9 + 3, num as i32)] = 3;
                }
                for i in -2..=2i32 {
                    array3[ga(num8 - 3, num9 + i, num as i32)] = 3;
                    array3[ga(num8 + 3, num9 + i, num as i32)] = 3;
                }

                // Inner ring
                for i in -2..=2i32 {
                    array3[ga(num8 + i, num9 - 2, num as i32)] = 2;
                    array3[ga(num8 + i, num9 + 2, num as i32)] = 2;
                }
                for i in -1..=1i32 {
                    array3[ga(num8 - 2, num9 + i, num as i32)] = 2;
                    array3[ga(num8 + 2, num9 + i, num as i32)] = 2;
                }

                // Border markings for specific corners
                if jj == 0 && kk == 0 {
                    for i in -3..=4i32 {
                        array3[ga(num8 + 4, num9 + i, num as i32)] = 8;
                    }
                    for i in -3..=4i32 {
                        array3[ga(num8 + i, num9 + 4, num as i32)] = 8;
                    }
                }
                if jj == 0 && kk == 1 {
                    for i in -4..=3i32 {
                        array3[ga(num8 + 4, num9 + i, num as i32)] = 8;
                    }
                    for i in -4..=3i32 {
                        array3[ga(num8 - i, num9 - 4, num as i32)] = 8;
                    }
                }
                if jj == 1 && kk == 0 {
                    for i in -3..=4i32 {
                        array3[ga(num8 - 4, num9 + i, num as i32)] = 8;
                    }
                    for i in -4..=3i32 {
                        array3[ga(num8 + i, num9 + 4, num as i32)] = 8;
                    }
                }
            }
        }

        // Alignment patterns
        let num16 = ap.len() as i32 - 1;
        for l_idx in 0..=num16 {
            for m_idx in 0..ap.len() as i32 {
                let num18 = ap[l_idx as usize];
                let num19 = ap[m_idx as usize];
                let ap_last = ap.len() as i32 - 1;
                if (l_idx == 0 && (m_idx == 0 || m_idx == ap_last))
                    || (m_idx == 0 && l_idx == ap_last)
                {
                    continue;
                }

                array3[ga(num18, num19, num as i32)] = 5;
                // Outer ring
                for i in -2..=2i32 {
                    array3[ga(num18 + i, num19 - 2, num as i32)] = 5;
                    array3[ga(num18 + i, num19 + 2, num as i32)] = 5;
                }
                for i in -1..=1i32 {
                    array3[ga(num18 - 2, num19 + i, num as i32)] = 5;
                    array3[ga(num18 + 2, num19 + i, num as i32)] = 5;
                }
                // Inner ring
                for dx in -1..=1i32 {
                    for dy in -1..=1i32 {
                        if dx == 0 && dy == 0 {
                            continue;
                        }
                        array3[ga(num18 + dx, num19 + dy, num as i32)] = 4;
                    }
                }
                array3[ga(num18, num19, num as i32)] = 5; // centre
            }
        }

        // Timing patterns
        let num20 = num as i32 - 9;
        let mut n = 8;
        while n <= num20 {
            array3[ga(n, 6, num as i32)] |= 0x11;
            array3[ga(6, n, num as i32)] |= 0x11;
            if n != num20 {
                array3[ga(n + 1, 6, num as i32)] |= 0x10;
                array3[ga(6, n + 1, num as i32)] |= 0x10;
            }
            n += 2;
        }

        array3[ga(8, num as i32 - 8, num as i32)] = 129;

        // Version information
        if p_version >= 7 {
            let mut num21 = QR_VERSION_INFO[(p_version - 7) as usize];
            for num22 in 0..6 {
                for num23 in 0..3 {
                    let num24 = num21 & 1;
                    if num24 == 1 {
                        array3[ga(num as i32 - 11 + num23, num22, num as i32)] |= 0x21;
                        array3[ga(num22, num as i32 - 11 + num23, num as i32)] |= 0x21;
                    } else {
                        array3[ga(num as i32 - 11 + num23, num22, num as i32)] |= 0x20;
                        array3[ga(num22, num as i32 - 11 + num23, num as i32)] |= 0x20;
                    }
                    num21 >>= 1;
                }
            }
        }

        let fmt_x = self.qr_code_get_format_x(self.l_qr_inquire_version);
        let fmt_y = self.qr_code_get_format_y(self.l_qr_inquire_version);
        self.l_format_second_x = fmt_x;
        self.l_format_second_y = fmt_y;
        for num25 in 0..15 {
            array3[ga(
                self.l_format_first_x[num25],
                self.l_format_first_y[num25],
                num as i32,
            )] |= 0x40;
            array3[ga(
                self.l_format_second_x[num25],
                self.l_format_second_y[num25],
                num as i32,
            )] |= 0x40;
        }

        array3
    }

    // ── RS block interleaving ───────────────────────────────────────────

    fn qr_code_trans_rs_block(&self, p_deploy_data: &[i32]) -> Vec<i32> {
        let ecc_idx = self.l_qr_set_ecc_rate as usize;
        let ver_idx = (self.l_qr_inquire_version - 1) as usize;
        let num = G_DATA_WORDS[ecc_idx][ver_idx];
        let num2 = G_EXTRA_DATA_WORDS[ecc_idx][ver_idx];
        let num3 = G_ECC_WORDS[ecc_idx][ver_idx] as f64;
        let num4 = G_RS_BLOCK_COUNT[ecc_idx][ver_idx];

        let mut arr = vec![0i32; p_deploy_data.len()];
        let mut num5: usize;
        let mut i: usize = 0;
        let mut array2 = vec![0i32; 129];
        let mut num6: usize = 0;

        for jj in 0..num {
            for kk in 0..num4 {
                let mut num9 = 0;
                if kk > num4 - num2 {
                    num9 = kk - (num4 - num2);
                    if jj == 0 {
                        array2[num6] = jj + kk * num + num9 - 1;
                        num6 += 1;
                    }
                }
                for num10 in 0..8 {
                    num5 = ((jj + kk * num + num9) * 8 + num10) as usize;
                    if num5 < arr.len() && i < p_deploy_data.len() {
                        arr[num5] = p_deploy_data[i];
                    }
                    i += 1;
                }
            }
        }

        if num2 > 0 {
            array2[num6] = num * num4 + num2 - 1;
            num6 += 1;
        }

        for &a2_val in array2.iter().take(num6) {
            for num12 in 0..8 {
                num5 = (a2_val * 8 + num12) as usize;
                if num5 < arr.len() && i < p_deploy_data.len() {
                    arr[num5] = p_deploy_data[i];
                }
                i += 1;
            }
        }

        let num13 = i;
        let num14 = (num3 - 1.0).round() as i32;
        for m in 0..=num14 {
            for n_val in 0..num4 {
                for num16 in 0..8 {
                    num5 = (num13 as f64
                        + (m as f64 + n_val as f64 * num3) * 8.0
                        + num16 as f64)
                        .round() as usize;
                    if num5 < arr.len() && i < p_deploy_data.len() {
                        arr[num5] = p_deploy_data[i];
                    }
                    i += 1;
                }
            }
        }

        while i < p_deploy_data.len() {
            if i < arr.len() {
                arr[i] = p_deploy_data[i];
            }
            i += 1;
        }

        arr
    }

    // ── 8-bit data packing ──────────────────────────────────────────────

    fn qr_code_get_8bit_data(
        &self,
        p_input_data: &[i32],
        p_input_data_bits: &[i32],
        p_max_bytes: usize,
    ) -> Vec<i32> {
        let mut arr = vec![0i32; p_max_bytes];
        let num_items = p_input_data_bits.len();
        let mut num2: usize = 0;
        let mut num3: i32 = 8;

        let num4: i32 = p_input_data_bits.iter().take(num_items).sum();

        let num6 = ((num4 - 1) as f64 / 8.0).floor() as usize;
        for jj in 0..=num6 {
            if jj < arr.len() {
                arr[jj] = 0;
            }
        }

        for k in 0..num_items {
            let mut num8 = p_input_data[k];
            let mut num9 = p_input_data_bits[k];
            if num9 == 0 {
                break;
            }

            let mut flag = true;
            while flag {
                if num3 > num9 {
                    arr[num2] = ((arr[num2] << (num9 & 7)) | num8) & 0xFF;
                    num3 -= num9;
                    flag = false;
                } else {
                    num9 -= num3;
                    arr[num2] = ((arr[num2] << (num3 & 7)) | (num8 >> num9)) & 0xFF;
                    if num9 == 0 {
                        flag = false;
                    } else {
                        num8 &= (1 << num9) - 1;
                        flag = true;
                    }
                    num2 += 1;
                    num3 = 8;
                }
            }
        }

        if num3 != 8 {
            arr[num2] = (arr[num2] << (num3 & 7)) & 0xFF;
        } else {
            num2 = num2.saturating_sub(1);
        }

        if num2 < p_max_bytes - 1 {
            let mut flag = true;
            while num2 < p_max_bytes - 1 {
                num2 += 1;
                arr[num2] = if flag { 236 } else { 17 };
                flag = !flag;
            }
        }

        arr
    }

    // ── Error correction calculation ────────────────────────────────────

    fn qr_code_get_total_data(
        &mut self,
        p_input_data: &[i32],
        p_ecc: i32,
        p_data: i32,
        p_total: i32,
    ) -> Vec<i32> {
        let ecc_idx = self.l_qr_set_ecc_rate as usize;
        let ver_idx = (self.l_qr_inquire_version - 1) as usize;
        let num = G_DATA_WORDS[ecc_idx][ver_idx];
        let num2 = G_EXTRA_DATA_WORDS[ecc_idx][ver_idx];
        let num3 = G_ECC_WORDS[ecc_idx][ver_idx] as f64;
        let num4 = G_RS_BLOCK_COUNT[ecc_idx][ver_idx];

        let array4 = qr_gen_poly(p_ecc).unwrap_or(&[]);

        // Build multiplication table
        let mut arr_table = vec![vec![0i32; array4.len()]; 256];
        for (i, &coeff) in array4.iter().enumerate() {
            let num6_val = GF_LOG[coeff as usize];
            for num7 in 0..256 {
                let num8 = ((num6_val + GF_LOG[num7]) % 255) as usize;
                arr_table[num7][i] = GF_EXP[num8];
            }
        }

        // RS block sizes
        self.l_rs_block = vec![0i32; num4 as usize];
        for jj in 0..num4 as usize {
            if jj as i32 >= num4 - num2 {
                self.l_rs_block[jj] = (num as f64 + num3 + 1.0).round() as i32;
            } else {
                self.l_rs_block[jj] = (num as f64 + num3).round() as i32;
            }
        }

        // Split input data into RS blocks
        let mut array5: Vec<Vec<i32>> = Vec::with_capacity(self.l_rs_block.len());
        for _ in 0..self.l_rs_block.len() {
            // Will be resized properly below
            array5.push(Vec::new());
        }
        let mut array6 = vec![0i32; p_total as usize];
        let copy_len = p_input_data.len().min(array6.len());
        array6[..copy_len].copy_from_slice(&p_input_data[..copy_len]);

        // Allocate array5 properly
        for (k, block_size) in self.l_rs_block.iter().enumerate() {
            let size = (*block_size - p_ecc) as usize;
            array5[k] = vec![0i32; size + 1];
        }

        let mut num11: usize = 0;
        let mut num12: usize = 0;
        for num10 in 0..p_data as usize {
            if num12 < self.l_rs_block.len() && num10 < p_input_data.len() {
                array5[num12][num11] = p_input_data[num10];
                num11 += 1;
                if num11 >= (self.l_rs_block[num12] - p_ecc) as usize {
                    num11 = 0;
                    num12 += 1;
                }
            }
        }

        for (num12_idx, &block_size) in self.l_rs_block.iter().enumerate() {
            let mut array8 = array5[num12_idx].clone();
            let num16 = block_size - p_ecc;

            for _cnt in (1..=num16).rev() {
                let b = array8[0];
                if b != 0 {
                    let array9: Vec<i32> = array8[1..].to_vec();
                    let ecc_len = p_ecc as usize;
                    let array10 = arr_table[b as usize][..ecc_len].to_vec();
                    array8 = qr_code_get_byte_data(&array9, &array10);
                } else if (p_ecc as usize) < array8.len() {
                    array8 = array8[1..].to_vec();
                } else {
                    let mut array12 = vec![0i32; p_ecc as usize];
                    for idx in 0..p_ecc as usize {
                        if idx < array8.len() - 1 {
                            array12[idx] = array8[idx + 1];
                        }
                    }
                    if p_ecc > 0 {
                        array12[p_ecc as usize - 1] = 0;
                    }
                    array8 = array12;
                }
            }

            let dest_start = p_input_data.len() + num12_idx * p_ecc as usize;
            for idx in 0..p_ecc as usize {
                if dest_start + idx < array6.len() && idx < array8.len() {
                    array6[dest_start + idx] = array8[idx];
                }
            }
        }

        array6
    }

    // ── Main encoding pipeline ──────────────────────────────────────────

    fn qr_get_data(&mut self, data: &[u8]) -> Result<Vec<Vec<i32>>> {
        let num4 = data.len();
        self.qr_code_inquire_bits(data);
        self.l_qr_inquire_text_type = self.qr_code_set_inquire_text_type();
        let mut qr_text_types = self.l_qr_inquire_text_type;

        // Kanji is treated as binary
        if qr_text_types == 3 {
            // qrTextKanji
            qr_text_types = QR_TEXT_BINARY;
            self.l_qr_inquire_text_type = QR_TEXT_BINARY;
        }

        let mut array6 = vec![0i32; num4 + 32];
        let mut array7 = vec![0i32; num4 + 32];

        if num4 == 0 {
            return Ok(vec![vec![0]]);
        }

        array7[0] = 4;
        let mut num: usize = 0; // index into array6/array7

        // Version-dependent extra bits for character count indicator
        let array5_ver: [i32; 41];

        if qr_text_types == QR_TEXT_NUMERIC {
            array5_ver = [
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
                2, 2, 2, 2, 2, 2, 2, 4, 4, 4,
                4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
            ];
            array6[num] = 1;
            num += 1;
            array6[num] = num4 as i32;
            array7[num] = 10;
            num += 1;

            for (jj, &byte) in data.iter().enumerate().take(num4) {
                let b = byte as i32 - 0x30;
                if jj % 3 == 0 {
                    array6[num] = b;
                    array7[num] = 4;
                } else {
                    array6[num] = array6[num] * 10 + b;
                    if jj % 3 == 1 {
                        array7[num] = 7;
                    } else {
                        array7[num] = 10;
                        if jj < num4 - 1 {
                            num += 1;
                        }
                    }
                }
            }
            num += 1;
        } else {
            // Binary / default
            array5_ver = [
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
                8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
                8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
            ];
            array6[num] = 4; // Binary mode indicator
            num += 1;
            array6[num] = num4 as i32;
            array7[num] = 8;
            num += 1;

            for i in 0..num4 {
                array6[i + num] = data[i] as i32;
                array7[i + num] = 8;
            }
            num += num4;
        }

        // Calculate total bits
        let mut num2: i32 = array7.iter().take(num).sum();

        // num5 is the index of the character count indicator
        let num5: usize = 1;

        // Auto version selection
        self.l_qr_inquire_version = 1;
        let mut num12 = 1;
        let mut num3 = 0i32;
        while num12 <= 40 {
            if QR_ARRAY3_CAP[self.l_qr_set_ecc_rate as usize][num12 as usize]
                >= num2 + array5_ver[self.l_qr_inquire_version as usize]
            {
                num3 = QR_ARRAY3_CAP[self.l_qr_set_ecc_rate as usize][num12 as usize];
                break;
            }
            self.l_qr_inquire_version += 1;
            num12 += 1;
        }

        if self.l_qr_inquire_version > 40 {
            return Err(BarcodeError::EncodingError(
                "data too long for QR Code".into(),
            ));
        }

        // Version bug fix: when user specifies a version (not AUTO),
        // num3 (max data bits) must be updated for the user-specified version.
        // If auto-calculated version > requested version, that is an error.
        // Otherwise use the requested version and recalculate num3.
        if self.l_qr_set_version != 0 {
            if self.l_qr_inquire_version > self.l_qr_set_version {
                // Data doesn't fit in the requested version — error
                return Err(BarcodeError::EncodingError(format!(
                    "data requires version {} but version {} was requested",
                    self.l_qr_inquire_version, self.l_qr_set_version
                )));
            }
            self.l_qr_inquire_version = self.l_qr_set_version;
            num3 = QR_ARRAY3_CAP[self.l_qr_set_ecc_rate as usize]
                [self.l_qr_inquire_version as usize];
        }

        num2 += array5_ver[self.l_qr_inquire_version as usize];
        array7[num5] += array5_ver[self.l_qr_inquire_version as usize];
        let num14 = QR_ARRAY_CAP[self.l_qr_inquire_version as usize];
        let num15 = QR_ARRAY2_EXTRA[self.l_qr_inquire_version as usize] + (num14 << 3);

        let fmt_x = self.qr_code_get_format_x(self.l_qr_inquire_version);
        let fmt_y = self.qr_code_get_format_y(self.l_qr_inquire_version);
        self.l_format_second_x = fmt_x;
        self.l_format_second_y = fmt_y;

        let p_ecc = G_ECC_WORDS[self.l_qr_set_ecc_rate as usize]
            [(self.l_qr_inquire_version - 1) as usize];
        let num16 = (num3 as f64 / 8.0).round() as i32;
        let num17 = 4 * self.l_qr_inquire_version + 17;
        self.l_qr_inquire_module_size = num17;

        let array9 = self.qr_code_get_symbol_data(self.l_qr_inquire_version);

        self.l_deploy_table_x = vec![0i32; num15 as usize + 1];
        self.l_deploy_table_y = vec![0i32; num15 as usize + 1];
        self.l_deploy_order_x = vec![0i32; num15 as usize];
        self.l_deploy_order_y = vec![0i32; num15 as usize];

        // Data placement order
        let mut num19: usize = 0;
        let mut m_val = num17 - 1;
        while m_val >= 3 {
            for n_val in (0..num17).rev() {
                let num20 = (n_val * (num17 + 1) + m_val) as usize;
                if num20 < array9.len()
                    && (array9[num20] & 0xFE) == 0
                    && num19 < self.l_deploy_table_x.len()
                {
                    self.l_deploy_table_x[num19] = m_val;
                    self.l_deploy_table_y[num19] = n_val;
                    num19 += 1;
                }
                let num20b = (n_val * (num17 + 1) + m_val - 1) as usize;
                if num20b < array9.len()
                    && (array9[num20b] & 0xFE) == 0
                    && num19 < self.l_deploy_table_x.len()
                {
                    self.l_deploy_table_x[num19] = m_val - 1;
                    self.l_deploy_table_y[num19] = n_val;
                    num19 += 1;
                }
            }

            if m_val == 8 {
                m_val = 7;
            }
            for num22 in 0..num17 {
                let num23 = (num22 * (num17 + 1) + m_val - 2) as usize;
                if num23 < array9.len()
                    && (array9[num23] & 0xFE) == 0
                    && num19 < self.l_deploy_table_x.len()
                {
                    self.l_deploy_table_x[num19] = m_val - 2;
                    self.l_deploy_table_y[num19] = num22;
                    num19 += 1;
                }
                let num23b = (num22 * (num17 + 1) + m_val - 3) as usize;
                if num23b < array9.len()
                    && (array9[num23b] & 0xFE) == 0
                    && num19 < self.l_deploy_table_x.len()
                {
                    self.l_deploy_table_x[num19] = m_val - 3;
                    self.l_deploy_table_y[num19] = num22;
                    num19 += 1;
                }
            }
            m_val -= 4;
        }

        self.l_deploy_order_x = self.qr_code_trans_rs_block(&self.l_deploy_table_x.clone());
        self.l_deploy_order_y = self.qr_code_trans_rs_block(&self.l_deploy_table_y.clone());

        // Terminator
        if num2 <= num3 - 4 {
            array6[num] = 0;
            array7[num] = 4;
        } else if num2 < num3 {
            array6[num] = 0;
            array7[num] = num3 - num2;
        }

        // Convert to 8-bit data
        let p_input_data = self.qr_code_get_8bit_data(&array6, &array7, num16 as usize);

        // Add error correction data
        let array10 = self.qr_code_get_total_data(&p_input_data, p_ecc, num16, num14);

        // Initialise QR matrix
        let mut array11 = vec![vec![0i32; num17 as usize]; num17 as usize];

        // Place data into matrix
        for num29 in 0..num14 as usize {
            let mut b5 = if num29 < array10.len() { array10[num29] } else { 0 };
            for num30 in (0..=7).rev() {
                let num31 = num29 * 8 + num30;
                if num31 < self.l_deploy_order_x.len()
                    && num31 < self.l_deploy_order_y.len()
                {
                    let ox = self.l_deploy_order_x[num31] as usize;
                    let oy = self.l_deploy_order_y[num31] as usize;
                    let num32 = qr_code_get_mask_bit_group(ox as i32, oy as i32);
                    if ox < array11.len() && oy < array11[0].len() {
                        array11[ox][oy] = (255 * (b5 & 1)) ^ num32;
                    }
                }
                b5 >>= 1;
            }
        }

        let extra = QR_ARRAY2_EXTRA[self.l_qr_inquire_version as usize];
        for num33 in (1..=extra).rev() {
            let num34 = (num33 + num14 * 8 - 1) as usize;
            if num34 < self.l_deploy_order_x.len()
                && num34 < self.l_deploy_order_y.len()
            {
                let ox = self.l_deploy_order_x[num34] as usize;
                let oy = self.l_deploy_order_y[num34] as usize;
                let num32 = qr_code_get_mask_bit_group(ox as i32, oy as i32);
                if ox < array11.len() && oy < array11[0].len() {
                    array11[ox][oy] = 0xFF ^ num32;
                }
            }
        }

        // Choose mask pattern
        let b6 = qr_code_choose_mask_number(
            &array11,
            QR_ARRAY2_EXTRA[self.l_qr_inquire_version as usize] + num14 * 8,
        );
        let b7 = 1i32 << b6;
        let b8 = (b6 + self.l_qr_set_ecc_rate * 8) as usize;

        // Place format information
        let format_str = QR_FORMAT_INFO[b8].as_bytes();
        for (num35, &fch) in format_str.iter().enumerate().take(15) {
            let b9 = (fch - b'0') as i32;
            let fx = self.l_format_first_x[num35] as usize;
            let fy = self.l_format_first_y[num35] as usize;
            if fx < array11.len() && fy < array11[0].len() {
                array11[fx][fy] = b9 * 255;
            }
            let sx = self.l_format_second_x[num35] as usize;
            let sy = self.l_format_second_y[num35] as usize;
            if sx < array11.len() && sy < array11[0].len() {
                array11[sx][sy] = b9 * 255;
            }
        }

        // Build final matrix
        let num17u = num17 as usize;
        let mut array12 = vec![vec![0i32; num17u]; num17u];
        let mut num36: usize = 0;
        for num38 in 0..num17u {
            for num40 in 0..num17u {
                if num36 < array9.len() {
                    if (array11[num40][num38] & b7) != 0 || (array9[num36] & 1) == 1 {
                        array12[num40][num38] = 1 | (array9[num36] & 0xFE);
                    } else {
                        array12[num40][num38] = array9[num36] & 0xFE;
                    }
                }
                num36 += 1;
            }
            num36 += 1;
        }

        Ok(array12)
    }

    // ── Top-level entry point ───────────────────────────────────────────

    fn make_arr(&mut self, data: &[u8]) -> Result<()> {
        if data.is_empty() {
            return Err(BarcodeError::EmptyCode);
        }

        let array = self.qr_get_data(data)?;
        let size = array.len();
        if size == 0 {
            return Err(BarcodeError::EncodingError(
                "QR encoding produced empty array".into(),
            ));
        }

        self.patt = Vec::with_capacity(size);
        for col in array.iter().take(size) {
            let mut row = vec![false; size];
            for (jj, cell) in col.iter().enumerate().take(size) {
                row[jj] = (cell & 1) == 1;
            }
            self.patt.push(row);
        }

        Ok(())
    }
}

// ── Free functions ──────────────────────────────────────────────────────────

/// 1D address calculation for symbol data array.
#[inline]
fn ga(px: i32, py: i32, symbol_size: i32) -> usize {
    (px + py * (symbol_size + 1)) as usize
}

/// XOR two byte arrays (the shorter one is zero-extended).
fn qr_code_get_byte_data(p_input1: &[i32], p_input2: &[i32]) -> Vec<i32> {
    let (arr, arr2) = if p_input1.len() > p_input2.len() {
        (p_input1, p_input2)
    } else {
        (p_input2, p_input1)
    };

    let num_a = arr.len();
    let num2 = arr2.len();
    let mut arr3 = vec![0i32; num_a + 1];

    for i in 0..num_a {
        arr3[i] = if i < num2 { arr[i] ^ arr2[i] } else { arr[i] };
    }

    arr3
}

/// Returns whether a specific mask pattern bit is set for position (py, px).
fn qr_code_get_mask_bit(mask_no: i32, py: i32, px: i32) -> bool {
    match mask_no {
        0 => (px + py) % 2 == 0,
        1 => px % 2 == 0,
        2 => py % 3 == 0,
        3 => (px + py) % 3 == 0,
        4 => (px / 2 + py / 3) % 2 == 0,
        5 => (px * py) % 2 + (px * py) % 3 == 0,
        6 => ((px * py) % 2 + (px * py) % 3) % 2 == 0,
        7 => ((px * py) % 3 + (px + py) % 2) % 2 == 0,
        _ => false,
    }
}

/// Returns a bitmask encoding which of the 8 mask patterns are set at (py, px).
fn qr_code_get_mask_bit_group(py: i32, px: i32) -> i32 {
    let mut num = 0i32;
    for m in 0..8 {
        if qr_code_get_mask_bit(m, py, px) {
            num += 1 << m;
        }
    }
    num
}

/// Evaluates all 8 mask patterns and returns the one with the lowest penalty.
fn qr_code_choose_mask_number(p_module_data: &[Vec<i32>], p_extra_bits: i32) -> i32 {
    let length = p_module_data.len();
    let mut array = [0i32; 8];
    let mut array2 = [0i32; 8];
    let mut array3 = [0i32; 8];
    let mut array4 = [0i32; 8];
    let mut array5 = [0i32; 8];
    let mut array6 = [0i32; 8];
    let mut array7 = [0i32; 9];
    let mut array8 = [0i32; 9];
    let mut array9 = [false; 9];
    let mut array10 = [false; 9];

    for i in 0..length {
        for n in 0..8 {
            array7[n] = 0;
            array8[n] = 0;
            array9[n] = false;
            array10[n] = false;
        }

        for jj in 0..length {
            let mut num = 0i32;
            let mut num2 = 0i32;
            if jj > 0 && i > 0 {
                num = p_module_data[jj][i]
                    & p_module_data[jj - 1][i]
                    & p_module_data[jj][i - 1]
                    & p_module_data[jj - 1][i - 1];
                num2 = p_module_data[jj][i]
                    | p_module_data[jj - 1][i]
                    | p_module_data[jj][i - 1]
                    | p_module_data[jj - 1][i - 1];
            }

            for m in 0..8 {
                array7[m] = ((array7[m] & 0x3F) << 1) | ((p_module_data[jj][i] >> (m & 7)) & 1);
                array8[m] = ((array8[m] & 0x3F) << 1) | ((p_module_data[i][jj] >> (m & 7)) & 1);

                if p_module_data[jj][i] & (1 << m) != 0 {
                    array4[m] += 1;
                }
                if array7[m] == 0b1011101 {
                    array3[m] += 40;
                }
                if array8[m] == 0b1011101 {
                    array3[m] += 40;
                }
                if jj > 0 && i > 0 {
                    if (num & 1) != 0 || (num2 & 1) == 0 {
                        array2[m] += 3;
                    }
                    num >>= 1;
                    num2 >>= 1;
                }

                if (array7[m] & 0x1F) == 0 || (array7[m] & 0x1F) == 31 {
                    if jj > 3 {
                        if array9[m] {
                            array[m] += 1;
                        } else {
                            array[m] += 3;
                            array9[m] = true;
                        }
                    }
                } else {
                    array9[m] = false;
                }

                if (array8[m] & 0x1F) == 0 || (array8[m] & 0x1F) == 31 {
                    if jj > 3 {
                        if array10[m] {
                            array[m] += 1;
                        } else {
                            array[m] += 3;
                            array10[m] = true;
                        }
                    }
                } else {
                    array10[m] = false;
                }
            }
        }
    }

    let array11: [i32; 21] = [
        90, 80, 70, 60, 50, 40, 30, 20, 10, 0, 0, 10, 20, 30, 40, 50, 60, 70, 80, 90, 90,
    ];

    let mut num8 = 0i32;
    let mut result = 0i32;

    for n in 0..8 {
        let mut idx =
            ((20.0 * array4[n] as f64) / p_extra_bits as f64).round() as i32;
        idx = idx.clamp(0, 20);
        array5[n] = array11[idx as usize];
        array6[n] = array[n] + array2[n] + array3[n] + array5[n];
    }

    for (o, &penalty) in array6.iter().enumerate() {
        if penalty < num8 || o == 0 {
            result = o as i32;
            num8 = penalty;
        }
    }

    result
}

// ── Tests ───────────────────────────────────────────────────────────────────

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_new_defaults() {
        let qr = QR::new("svg");
        assert_eq!(qr.version, 0);
        assert_eq!(qr.error_correction, QR_ECC_M as i32);
        assert_eq!(qr.encode_mode(), QR_MODE_BINARY);
    }

    #[test]
    fn test_set_encode_mode() {
        let mut qr = QR::new("svg");
        qr.set_encode_mode("N");
        assert_eq!(qr.encode_mode(), "N");
        qr.set_encode_mode("invalid");
        assert_eq!(qr.encode_mode(), "B");
    }

    #[test]
    fn test_draw_svg_basic() {
        let mut qr = QR::new("svg");
        qr.draw("Hello", 200).unwrap();
        let svg = qr.base_2d.base.get_svg().unwrap();
        assert!(svg.contains("<svg"));
        assert!(svg.contains("</svg>"));
        assert!(svg.contains("<rect"));
    }

    #[test]
    fn test_draw_svg_numeric() {
        let mut qr = QR::new("svg");
        qr.set_encode_mode("N");
        qr.draw("12345678", 200).unwrap();
        let svg = qr.base_2d.base.get_svg().unwrap();
        assert!(svg.contains("<svg"));
    }

    #[test]
    fn test_draw_empty_code() {
        let mut qr = QR::new("svg");
        assert!(qr.draw("", 200).is_err());
    }

    #[test]
    fn test_get_pattern() {
        let mut qr = QR::new("svg");
        let pattern = qr.get_pattern("Test").unwrap();
        assert!(!pattern.is_empty());
        // QR version 1 has 21 modules
        assert!(pattern.len() >= 21);
        // Pattern should be square
        assert_eq!(pattern.len(), pattern[0].len());
    }

    #[test]
    fn test_error_correction_levels() {
        for level in [QR_ECC_L, QR_ECC_M, QR_ECC_Q, QR_ECC_H] {
            let mut qr = QR::new("svg");
            qr.error_correction = level;
            qr.draw("Test", 200).unwrap();
            let svg = qr.base_2d.base.get_svg().unwrap();
            assert!(svg.contains("<svg"));
        }
    }

    #[test]
    fn test_version_manual() {
        let mut qr = QR::new("svg");
        qr.version = 5; // Force version 5
        qr.draw("Hello", 200).unwrap();
        let pattern = qr.get_pattern("Hello").unwrap();
        // Version 5 = 4*5+17 = 37 modules
        assert_eq!(pattern.len(), 37);
    }

    #[test]
    fn test_version_bug_fix_data_too_large() {
        // Try to force a very small version with data that won't fit
        let mut qr = QR::new("svg");
        qr.version = 1;
        let long_data = "A".repeat(100);
        let result = qr.draw(&long_data, 200);
        assert!(result.is_err());
    }

    #[test]
    fn test_version_auto() {
        let mut qr = QR::new("svg");
        qr.version = 0; // Auto
        // Short data should produce a small version
        let pattern = qr.get_pattern("Hi").unwrap();
        assert_eq!(pattern.len(), 21); // Version 1
    }

    #[test]
    fn test_larger_data() {
        let mut qr = QR::new("svg");
        let data = "The quick brown fox jumps over the lazy dog";
        qr.draw(data, 400).unwrap();
        let svg = qr.base_2d.base.get_svg().unwrap();
        assert!(svg.contains("<svg"));
    }

    #[test]
    fn test_mask_bit() {
        // Mask 0: (px+py) % 2 == 0
        assert!(qr_code_get_mask_bit(0, 0, 0));
        assert!(!qr_code_get_mask_bit(0, 0, 1));
        assert!(qr_code_get_mask_bit(0, 1, 1));
    }

    #[test]
    fn test_gf_tables_consistency() {
        // Verify GF_EXP and GF_LOG are inverses
        for i in 0..255 {
            let exp_val = GF_EXP[i] as usize;
            assert_eq!(GF_LOG[exp_val], i as i32, "GF table mismatch at {}", i);
        }
    }

    #[test]
    fn test_alignment_patterns() {
        // Version 1 has no alignment patterns (only finder)
        assert_eq!(QR_ALIGNMENT_PATTERNS[1].len(), 1);
        // Version 2 has 2 positions
        assert_eq!(QR_ALIGNMENT_PATTERNS[2].len(), 2);
        // Version 40 has 7 positions
        assert_eq!(QR_ALIGNMENT_PATTERNS[40].len(), 7);
    }

    #[test]
    fn test_format_info_length() {
        assert_eq!(QR_FORMAT_INFO.len(), 32);
        for fi in &QR_FORMAT_INFO {
            assert_eq!(fi.len(), 15);
        }
    }

    #[cfg(any(feature = "png", feature = "jpeg"))]
    #[test]
    fn test_draw_png() {
        let mut qr = QR::new("png");
        qr.draw("Hello PNG", 200).unwrap();
        let img = qr.base_2d.base.get_image_memory();
        assert!(!img.is_empty());
        // PNG magic bytes
        assert_eq!(&img[..4], &[0x89, 0x50, 0x4E, 0x47]);
    }

    #[test]
    fn test_url_encoding() {
        let mut qr = QR::new("svg");
        qr.draw("https://example.com/path?q=1&r=2", 300).unwrap();
        let svg = qr.base_2d.base.get_svg().unwrap();
        assert!(svg.contains("<svg"));
    }

    #[test]
    fn test_multibyte_utf8() {
        let mut qr = QR::new("svg");
        qr.draw("日本語テスト", 300).unwrap();
        let svg = qr.base_2d.base.get_svg().unwrap();
        assert!(svg.contains("<svg"));
    }
}
