use crate::base_2d::BarcodeBase2D;
use crate::error::{BarcodeError, Result};

use super::pdf417_tables::{CLUSTER_TABLE, ERROR_CORRECTION_CODEWORDS};

// --- Constants ---

const START_PATTERN: i32 = 0x1fea8;
const STOP_PATTERN: i32 = 0x3fa29;
const START_CODE_SIZE: i32 = 17;
const STOP_SIZE: i32 = 18;
const MOD: i32 = 929;
const ALPHA: i32 = 0x10000;
const LOWER: i32 = 0x20000;
const MIXED: i32 = 0x40000;
const PUNCTUATION: i32 = 0x80000;
const IS_BYTE: i32 = 0x100000;
const BYTE_SHIFT: i32 = 913;
const PL: i32 = 25;
const LL: i32 = 27;
const AS: i32 = 27;
const ML: i32 = 28;
const AL: i32 = 28;
const PS: i32 = 29;
const PAL: i32 = 29;
const SPACE: i32 = 26;
const TEXT_MODE: i32 = 900;
const BYTE_MODE_6: i32 = 924;
const BYTE_MODE: i32 = 901;
const NUMERIC_MODE: i32 = 902;
const ABSOLUTE_MAX_TEXT_SIZE: usize = 5420;
const MAX_DATA_CODEWORDS: usize = 926;

// Size kind constants
const SIZE_AUTO: i32 = 0;
const SIZE_COLUMNS: i32 = 1;
const SIZE_ROWS: i32 = 2;
const SIZE_COLUMNS_AND_ROWS: i32 = 3;

const MIXED_SET: &[u8] = b"0123456789&\r\t,:#-.$/+%*=^";
const PUNCTUATION_SET: &[u8] = b";<>@[\\]_`~!\r\t,:\n-.$/\"|*()?{}'";

// --- Segment ---

#[derive(Clone)]
struct Segment {
    seg_type: u8, // b'T', b'N', b'B'
    start: usize,
    end: usize,
}

// --- Internal encoder ---

struct Pdf417Encoder {
    out_bits: Vec<u8>,
    bit_ptr: usize,
    bit_columns: i32,
    error_level: i32,
    len_codewords: usize,
    code_columns: i32,
    code_rows: i32,
    use_auto_error_level: bool,
    aspect_ratio: f64,
    y_height: i32,
    options: i32,
    size_kind: i32,
    codewords: Vec<i32>,
    text: Vec<u8>,
    cw_ptr: usize,
    segment_list: Vec<Segment>,
}

impl Pdf417Encoder {
    fn new() -> Self {
        Self {
            out_bits: Vec::new(),
            bit_ptr: 0,
            bit_columns: 0,
            error_level: 2,
            len_codewords: 0,
            code_columns: 0,
            code_rows: 0,
            use_auto_error_level: true,
            aspect_ratio: 0.5,
            y_height: 3,
            options: 0,
            size_kind: SIZE_AUTO,
            codewords: Vec::new(),
            text: Vec::new(),
            cw_ptr: 0,
            segment_list: Vec::new(),
        }
    }

    fn init_block(&mut self) {
        self.codewords = vec![0i32; MAX_DATA_CODEWORDS + 2];
    }

    // --- Bit output ---

    fn out_codeword_17(&mut self, codeword: i32) {
        let byte_ptr_start = self.bit_ptr >> 3;
        let bit = self.bit_ptr - byte_ptr_start * 8;
        let mut byte_ptr = byte_ptr_start;
        self.out_bits[byte_ptr] |= ((codeword >> (9 + bit)) & 0xFF) as u8;
        byte_ptr += 1;
        self.out_bits[byte_ptr] |= ((codeword >> (1 + bit)) & 0xFF) as u8;
        byte_ptr += 1;
        let codeword = codeword << 8;
        self.out_bits[byte_ptr] |= ((codeword >> (1 + bit)) & 0xFF) as u8;
        self.bit_ptr += 17;
    }

    fn out_codeword_18(&mut self, codeword: i32) {
        let byte_ptr_start = self.bit_ptr >> 3;
        let bit = self.bit_ptr - byte_ptr_start * 8;
        let mut byte_ptr = byte_ptr_start;
        self.out_bits[byte_ptr] |= ((codeword >> (10 + bit)) & 0xFF) as u8;
        byte_ptr += 1;
        self.out_bits[byte_ptr] |= ((codeword >> (2 + bit)) & 0xFF) as u8;
        byte_ptr += 1;
        let codeword = codeword << 8;
        self.out_bits[byte_ptr] |= ((codeword >> (2 + bit)) & 0xFF) as u8;
        if bit == 7 {
            byte_ptr += 1;
            self.out_bits[byte_ptr] |= 0x80;
        }
        self.bit_ptr += 18;
    }

    fn out_codeword(&mut self, codeword: i32) {
        self.out_codeword_17(codeword);
    }

    fn out_start_pattern(&mut self) {
        self.out_codeword_17(START_PATTERN);
    }

    fn out_stop_pattern(&mut self) {
        self.out_codeword_18(STOP_PATTERN);
    }

    // --- Paint code ---

    fn out_paint_code(&mut self) {
        self.bit_columns = START_CODE_SIZE * (self.code_columns + 3) + STOP_SIZE;
        let bytes_per_row = ((self.bit_columns - 1) / 8 + 1) as usize;
        let len_bits = bytes_per_row * self.code_rows as usize;
        self.out_bits = vec![0u8; len_bits];

        let mut code_ptr: usize = 0;
        for row in 0..self.code_rows {
            self.bit_ptr = bytes_per_row * 8 * row as usize;
            let row_mod = (row % 3) as usize;
            let cluster = &CLUSTER_TABLE[row_mod];

            self.out_start_pattern();

            // Left edge indicator
            let edge = match row_mod {
                0 => 30 * (row / 3) + (self.code_rows - 1) / 3,
                1 => 30 * (row / 3) + self.error_level * 3 + (self.code_rows - 1) % 3,
                _ => 30 * (row / 3) + self.code_columns - 1,
            };
            self.out_codeword(cluster[edge as usize]);

            // Data codewords
            for _col in 0..self.code_columns {
                if code_ptr >= self.len_codewords {
                    break;
                }
                let cw = self.codewords[code_ptr];
                if cw < 0 || cw >= cluster.len() as i32 {
                    break;
                }
                self.out_codeword(cluster[cw as usize]);
                code_ptr += 1;
            }

            // Right edge indicator
            let edge = match row_mod {
                0 => 30 * (row / 3) + self.code_columns - 1,
                1 => 30 * (row / 3) + (self.code_rows - 1) / 3,
                _ => 30 * (row / 3) + self.error_level * 3 + (self.code_rows - 1) % 3,
            };
            self.out_codeword(cluster[edge as usize]);

            self.out_stop_pattern();
        }

        if (self.options & 0x02) != 0 {
            // PDF417_INVERT_BITMAP
            for b in &mut self.out_bits {
                *b ^= 0xFF;
            }
        }
    }

    // --- Error correction ---

    fn calculate_error_correction(&mut self, dest: usize) {
        let el = self.error_level.clamp(0, 8) as usize;
        let a = ERROR_CORRECTION_CODEWORDS[el];
        let a_length = 2usize << el;
        for k in 0..a_length {
            self.codewords[dest + k] = 0;
        }
        let last_e = a_length - 1;
        for k in 0..self.len_codewords {
            let t1 = self.codewords[k] + self.codewords[dest];
            for e in 0..a_length {
                let t2 = (t1 * a[last_e - e]) % MOD;
                let t3 = MOD - t2;
                let nxt = if e != last_e {
                    self.codewords[dest + e + 1]
                } else {
                    0
                };
                self.codewords[dest + e] = (nxt + t3) % MOD;
            }
        }
        for k in 0..a_length {
            self.codewords[dest + k] = (MOD - self.codewords[dest + k]) % MOD;
        }
    }

    // --- Text type/value ---

    fn get_text_type_and_value(&self, max_length: usize, idx: usize) -> i32 {
        if idx >= max_length {
            return 0;
        }
        let c = self.text[idx] as i32;
        let ch = c as u8;

        if ch.is_ascii_uppercase() {
            return ALPHA + c - i32::from(b'A');
        }
        if ch.is_ascii_lowercase() {
            return LOWER + c - i32::from(b'a');
        }
        if ch == b' ' {
            return ALPHA + LOWER + MIXED + SPACE;
        }

        let ms = index_of(MIXED_SET, ch);
        let ps = index_of(PUNCTUATION_SET, ch);

        if ms < 0 && ps < 0 {
            return IS_BYTE + c;
        }
        if ms == ps {
            return MIXED + PUNCTUATION + ms;
        }
        if ms >= 0 {
            return MIXED + ms;
        }
        PUNCTUATION + ps
    }

    // --- Text compaction ---

    fn text_compaction(&mut self, start: usize, length: usize) -> Result<()> {
        let mut dest = vec![0i32; ABSOLUTE_MAX_TEXT_SIZE * 2];
        let mut mode = ALPHA;
        let mut ptr: usize = 0;
        let mut full_bytes: usize = 0;
        let length_end = length + start;
        let mut k = start;
        while k < length_end {
            let v = self.get_text_type_and_value(length_end, k);
            if (v & mode) != 0 {
                dest[ptr] = v & 0xFF;
                ptr += 1;
                k += 1;
                continue;
            }
            if (v & IS_BYTE) != 0 {
                if (ptr & 1) != 0 {
                    if (mode & PUNCTUATION) != 0 {
                        dest[ptr] = PAL;
                    } else {
                        dest[ptr] = PS;
                    }
                    ptr += 1;
                    if (mode & PUNCTUATION) != 0 {
                        mode = ALPHA;
                    }
                }
                dest[ptr] = BYTE_SHIFT;
                ptr += 1;
                dest[ptr] = v & 0xFF;
                ptr += 1;
                full_bytes += 2;
                k += 1;
                continue;
            }

            if mode == ALPHA {
                if (v & LOWER) != 0 {
                    dest[ptr] = LL;
                    ptr += 1;
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                    mode = LOWER;
                } else if (v & MIXED) != 0 {
                    dest[ptr] = ML;
                    ptr += 1;
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                    mode = MIXED;
                } else if (self.get_text_type_and_value(length_end, k + 1)
                    & self.get_text_type_and_value(length_end, k + 2)
                    & PUNCTUATION)
                    != 0
                {
                    dest[ptr] = ML;
                    ptr += 1;
                    dest[ptr] = PL;
                    ptr += 1;
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                    mode = PUNCTUATION;
                } else {
                    dest[ptr] = PS;
                    ptr += 1;
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                }
            } else if mode == LOWER {
                if (v & ALPHA) != 0 {
                    if (self.get_text_type_and_value(length_end, k + 1)
                        & self.get_text_type_and_value(length_end, k + 2)
                        & ALPHA)
                        != 0
                    {
                        dest[ptr] = ML;
                        ptr += 1;
                        dest[ptr] = AL;
                        ptr += 1;
                        mode = ALPHA;
                    } else {
                        dest[ptr] = AS;
                        ptr += 1;
                    }
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                } else if (v & MIXED) != 0 {
                    dest[ptr] = ML;
                    ptr += 1;
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                    mode = MIXED;
                } else if (self.get_text_type_and_value(length_end, k + 1)
                    & self.get_text_type_and_value(length_end, k + 2)
                    & PUNCTUATION)
                    != 0
                {
                    dest[ptr] = ML;
                    ptr += 1;
                    dest[ptr] = PL;
                    ptr += 1;
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                    mode = PUNCTUATION;
                } else {
                    dest[ptr] = PS;
                    ptr += 1;
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                }
            } else if mode == MIXED {
                if (v & LOWER) != 0 {
                    dest[ptr] = LL;
                    ptr += 1;
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                    mode = LOWER;
                } else if (v & ALPHA) != 0 {
                    dest[ptr] = AL;
                    ptr += 1;
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                    mode = ALPHA;
                } else if (self.get_text_type_and_value(length_end, k + 1)
                    & self.get_text_type_and_value(length_end, k + 2)
                    & PUNCTUATION)
                    != 0
                {
                    dest[ptr] = PL;
                    ptr += 1;
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                    mode = PUNCTUATION;
                } else {
                    dest[ptr] = PS;
                    ptr += 1;
                    dest[ptr] = v & 0xFF;
                    ptr += 1;
                }
            } else if mode == PUNCTUATION {
                dest[ptr] = PAL;
                ptr += 1;
                mode = ALPHA;
                // k will be incremented at the bottom, but we need to re-process this character,
                // so we decrement first (Go uses k-- then falls through to k++)
                k = k.wrapping_sub(1);
            }

            k += 1;
        }

        if (ptr & 1) != 0 {
            dest[ptr] = PS;
            ptr += 1;
        }
        let size = (ptr + full_bytes) / 2;
        if size + self.cw_ptr > MAX_DATA_CODEWORDS {
            return Err(BarcodeError::EncodingError("the text is too big".into()));
        }
        let length_end = ptr;
        ptr = 0;
        while ptr < length_end {
            let v = dest[ptr];
            ptr += 1;
            if v >= 30 {
                self.codewords[self.cw_ptr] = v;
                self.cw_ptr += 1;
                self.codewords[self.cw_ptr] = dest[ptr];
                self.cw_ptr += 1;
                ptr += 1;
            } else {
                self.codewords[self.cw_ptr] = v * 30 + dest[ptr];
                self.cw_ptr += 1;
                ptr += 1;
            }
        }
        Ok(())
    }

    // --- Number compaction ---

    fn basic_number_compaction(&mut self, start: usize, length: usize) {
        let ret = self.cw_ptr;
        let ret_last = length / 3;
        self.cw_ptr += ret_last + 1;
        for k in 0..=ret_last {
            self.codewords[ret + k] = 0;
        }
        self.codewords[ret + ret_last] = 1;
        let end = length + start;
        for ni in start..end {
            for k in (0..=ret_last).rev() {
                self.codewords[ret + k] *= 10;
            }
            self.codewords[ret + ret_last] += i32::from(self.text[ni]) - i32::from(b'0');
            for k in (1..=ret_last).rev() {
                self.codewords[ret + k - 1] += self.codewords[ret + k] / 900;
                self.codewords[ret + k] %= 900;
            }
        }
    }

    fn number_compaction(&mut self, start: usize, length: usize) -> Result<()> {
        let full = (length / 44) * 15;
        let sz = length % 44;
        let size = if sz == 0 {
            full
        } else {
            full + sz / 3 + 1
        };
        if size + self.cw_ptr > MAX_DATA_CODEWORDS {
            return Err(BarcodeError::EncodingError("the text is too big".into()));
        }
        let end = length + start;
        let mut k = start;
        while k < end {
            let seg_sz = (end - k).min(44);
            self.basic_number_compaction(k, seg_sz);
            k += 44;
        }
        Ok(())
    }

    // --- Byte compaction ---

    fn byte_compaction_6(&mut self, start: usize) {
        let length = 6;
        let ret = self.cw_ptr;
        let ret_last = 4;
        self.cw_ptr += ret_last + 1;
        for k in 0..=ret_last {
            self.codewords[ret + k] = 0;
        }
        let end = length + start;
        for ni in start..end {
            for k in (0..=ret_last).rev() {
                self.codewords[ret + k] *= 256;
            }
            self.codewords[ret + ret_last] += i32::from(self.text[ni]);
            for k in (1..=ret_last).rev() {
                self.codewords[ret + k - 1] += self.codewords[ret + k] / 900;
                self.codewords[ret + k] %= 900;
            }
        }
    }

    fn byte_compaction(&mut self, start: usize, length: usize) -> Result<()> {
        let size = (length / 6) * 5 + (length % 6);
        if size + self.cw_ptr > MAX_DATA_CODEWORDS {
            return Err(BarcodeError::EncodingError("the text is too big".into()));
        }
        let end = length + start;
        let mut k = start;
        while k < end {
            let sz = (end - k).min(6);
            if sz < 6 {
                for j in 0..sz {
                    self.codewords[self.cw_ptr] = i32::from(self.text[k + j]);
                    self.cw_ptr += 1;
                }
            } else {
                self.byte_compaction_6(k);
            }
            k += 6;
        }
        Ok(())
    }

    // --- Break string ---

    fn break_string(&mut self) {
        let text_length = self.text.len();
        let mut last_p: usize = 0;
        let mut start_n: usize = 0;
        let mut nd: usize = 0;

        for k in 0..text_length {
            let c = self.text[k];
            if c.is_ascii_digit() {
                if nd == 0 {
                    start_n = k;
                }
                nd += 1;
                continue;
            }
            if nd >= 13 {
                if last_p != start_n {
                    let c2 = self.text[last_p];
                    let mut last_txt = is_text_char(c2);
                    let mut j = last_p;
                    while j < start_n {
                        let c2 = self.text[j];
                        let txt = is_text_char(c2);
                        if txt != last_txt {
                            let seg_t = if last_txt { b'T' } else { b'B' };
                            self.segment_list.push(Segment {
                                seg_type: seg_t,
                                start: last_p,
                                end: j,
                            });
                            last_p = j;
                            last_txt = txt;
                        }
                        j += 1;
                    }
                    let seg_t = if last_txt { b'T' } else { b'B' };
                    self.segment_list.push(Segment {
                        seg_type: seg_t,
                        start: last_p,
                        end: start_n,
                    });
                }
                self.segment_list.push(Segment {
                    seg_type: b'N',
                    start: start_n,
                    end: k,
                });
                last_p = k;
            }
            nd = 0;
        }

        if nd < 13 {
            start_n = text_length;
        }
        if last_p != start_n {
            let c2 = self.text[last_p];
            let mut last_txt = is_text_char(c2);
            let mut j = last_p;
            while j < start_n {
                let c2 = self.text[j];
                let txt = is_text_char(c2);
                if txt != last_txt {
                    let seg_t = if last_txt { b'T' } else { b'B' };
                    self.segment_list.push(Segment {
                        seg_type: seg_t,
                        start: last_p,
                        end: j,
                    });
                    last_p = j;
                    last_txt = txt;
                }
                j += 1;
            }
            let seg_t = if last_txt { b'T' } else { b'B' };
            self.segment_list.push(Segment {
                seg_type: seg_t,
                start: last_p,
                end: start_n,
            });
        }
        if nd >= 13 {
            self.segment_list.push(Segment {
                seg_type: b'N',
                start: start_n,
                end: text_length,
            });
        }

        // Merge pass 1: single-byte segments between text segments
        let mut k = 0usize;
        while k < self.segment_list.len() {
            let v_type = self.segment_list[k].seg_type;
            let v_len = self.segment_list[k].end - self.segment_list[k].start;
            if v_type == b'B' && v_len == 1 {
                let has_prev = k > 0;
                let has_next = k + 1 < self.segment_list.len();
                if has_prev && has_next {
                    let vp_type = self.segment_list[k - 1].seg_type;
                    let vp_len =
                        self.segment_list[k - 1].end - self.segment_list[k - 1].start;
                    let vn_type = self.segment_list[k + 1].seg_type;
                    let vn_len =
                        self.segment_list[k + 1].end - self.segment_list[k + 1].start;
                    if vp_type == b'T' && vn_type == b'T' && (vp_len + vn_len) >= 3 {
                        let vn_end = self.segment_list[k + 1].end;
                        self.segment_list[k - 1].end = vn_end;
                        self.segment_list.drain(k..k + 2);
                        k = 0;
                        continue;
                    }
                }
            }
            k += 1;
        }

        // Merge pass 2: absorb short neighbors into long text segments
        k = 0;
        while k < self.segment_list.len() {
            let v_type = self.segment_list[k].seg_type;
            let v_len = self.segment_list[k].end - self.segment_list[k].start;
            if v_type == b'T' && v_len >= 5 {
                let mut redo = false;
                if k > 0 {
                    let vp_type = self.segment_list[k - 1].seg_type;
                    let vp_len =
                        self.segment_list[k - 1].end - self.segment_list[k - 1].start;
                    if (vp_type == b'B' && vp_len == 1) || vp_type == b'T' {
                        redo = true;
                        let vp_start = self.segment_list[k - 1].start;
                        self.segment_list[k].start = vp_start;
                        self.segment_list.remove(k - 1);
                        k -= 1;
                    }
                }
                if k + 1 < self.segment_list.len() {
                    let vn_type = self.segment_list[k + 1].seg_type;
                    let vn_len =
                        self.segment_list[k + 1].end - self.segment_list[k + 1].start;
                    if (vn_type == b'B' && vn_len == 1) || vn_type == b'T' {
                        redo = true;
                        let vn_end = self.segment_list[k + 1].end;
                        self.segment_list[k].end = vn_end;
                        self.segment_list.remove(k + 1);
                    }
                }
                if redo {
                    k = 0;
                    continue;
                }
            }
            k += 1;
        }

        // Merge pass 3: absorb short text segments into byte segments
        k = 0;
        while k < self.segment_list.len() {
            let v_type = self.segment_list[k].seg_type;
            if v_type == b'B' {
                let mut redo = false;
                if k > 0 {
                    let vp_type = self.segment_list[k - 1].seg_type;
                    let vp_len =
                        self.segment_list[k - 1].end - self.segment_list[k - 1].start;
                    if (vp_type == b'T' && vp_len < 5) || vp_type == b'B' {
                        redo = true;
                        let vp_start = self.segment_list[k - 1].start;
                        self.segment_list[k].start = vp_start;
                        self.segment_list.remove(k - 1);
                        k -= 1;
                    }
                }
                if k + 1 < self.segment_list.len() {
                    let vn_type = self.segment_list[k + 1].seg_type;
                    let vn_len =
                        self.segment_list[k + 1].end - self.segment_list[k + 1].start;
                    if (vn_type == b'T' && vn_len < 5) || vn_type == b'B' {
                        redo = true;
                        let vn_end = self.segment_list[k + 1].end;
                        self.segment_list[k].end = vn_end;
                        self.segment_list.remove(k + 1);
                    }
                }
                if redo {
                    k = 0;
                    continue;
                }
            }
            k += 1;
        }

        // Special case: single text segment of all digits
        if self.segment_list.len() == 1 {
            let v = &mut self.segment_list[0];
            if v.seg_type == b'T' && (v.end - v.start) >= 8 {
                let all_digits = self.text[v.start..v.end]
                    .iter()
                    .all(|&b| b.is_ascii_digit());
                if all_digits {
                    v.seg_type = b'N';
                }
            }
        }
    }

    // --- Assemble ---

    fn assemble(&mut self) -> Result<()> {
        if self.segment_list.is_empty() {
            return Ok(());
        }
        self.cw_ptr = 1;
        for k in 0..self.segment_list.len() {
            let seg_type = self.segment_list[k].seg_type;
            let seg_start = self.segment_list[k].start;
            let seg_end = self.segment_list[k].end;
            let seg_len = seg_end - seg_start;
            match seg_type {
                b'T' => {
                    if k != 0 {
                        self.codewords[self.cw_ptr] = TEXT_MODE;
                        self.cw_ptr += 1;
                    }
                    self.text_compaction(seg_start, seg_len)?;
                }
                b'N' => {
                    self.codewords[self.cw_ptr] = NUMERIC_MODE;
                    self.cw_ptr += 1;
                    self.number_compaction(seg_start, seg_len)?;
                }
                b'B' => {
                    if !seg_len.is_multiple_of(6) {
                        self.codewords[self.cw_ptr] = BYTE_MODE;
                    } else {
                        self.codewords[self.cw_ptr] = BYTE_MODE_6;
                    }
                    self.cw_ptr += 1;
                    self.byte_compaction(seg_start, seg_len)?;
                }
                _ => {}
            }
        }
        Ok(())
    }

    // --- Max possible error level ---

    fn max_possible_error_level(&self, remain: i32) -> i32 {
        let mut level = 8;
        let mut size = 512;
        while level > 0 {
            if remain >= size {
                return level;
            }
            level -= 1;
            size >>= 1;
        }
        0
    }

    // --- Max square ---

    fn max_square(&mut self) -> i32 {
        if self.code_columns > 21 {
            self.code_columns = 29;
            self.code_rows = 32;
        } else {
            self.code_columns = 16;
            self.code_rows = 58;
        }
        MAX_DATA_CODEWORDS as i32 + 2
    }

    // --- Paint code (main entry) ---

    fn paint_code(&mut self) -> Result<()> {
        self.init_block();

        if self.text.is_empty() {
            return Err(BarcodeError::EncodingError(
                "text cannot be empty".into(),
            ));
        }
        if self.text.len() > ABSOLUTE_MAX_TEXT_SIZE {
            return Err(BarcodeError::EncodingError("the text is too big".into()));
        }

        self.segment_list.clear();
        self.break_string();
        self.assemble()?;
        self.segment_list.clear();
        self.codewords[0] = self.cw_ptr as i32;
        self.len_codewords = self.cw_ptr;

        let max_err = self.max_possible_error_level(
            MAX_DATA_CODEWORDS as i32 + 2 - self.len_codewords as i32,
        );

        if self.use_auto_error_level {
            if self.len_codewords < 41 {
                self.error_level = 2;
            } else if self.len_codewords < 161 {
                self.error_level = 3;
            } else if self.len_codewords < 321 {
                self.error_level = 4;
            } else {
                self.error_level = 5;
            }
        }

        self.error_level = self.error_level.clamp(0, max_err);

        self.code_columns = self.code_columns.clamp(1, 30);
        self.code_rows = self.code_rows.clamp(3, 90);

        let len_err = 2i32 << self.error_level;
        let fixed_column = self.size_kind != SIZE_ROWS;
        let mut skip_row_col_adjust = false;
        let mut tot = self.len_codewords as i32 + len_err;

        if self.size_kind == SIZE_COLUMNS_AND_ROWS {
            tot = self.code_columns * self.code_rows;
            if tot > MAX_DATA_CODEWORDS as i32 + 2 {
                tot = self.max_square();
            }
            if tot < self.len_codewords as i32 + len_err {
                tot = self.len_codewords as i32 + len_err;
            } else {
                skip_row_col_adjust = true;
            }
        } else if self.size_kind == SIZE_AUTO {
            // fixed_column is already true
            self.aspect_ratio = self.aspect_ratio.clamp(0.001, 1000.0);
            let b = 73.0 * self.aspect_ratio - 4.0;
            let c = (-b
                + (b * b
                    + 4.0
                        * 17.0
                        * self.aspect_ratio
                        * (self.len_codewords as i32 + len_err) as f64
                        * self.y_height as f64)
                    .sqrt())
                / (2.0 * 17.0 * self.aspect_ratio);
            self.code_columns = (c + 0.5) as i32;
            self.code_columns = self.code_columns.clamp(1, 30);
        }

        if !skip_row_col_adjust {
            if fixed_column {
                self.code_rows = (tot - 1) / self.code_columns + 1;
                if self.code_rows < 3 {
                    self.code_rows = 3;
                } else if self.code_rows > 90 {
                    self.code_rows = 90;
                    self.code_columns = (tot - 1) / 90 + 1;
                }
            } else {
                self.code_columns = (tot - 1) / self.code_rows + 1;
                if self.code_columns > 30 {
                    self.code_columns = 30;
                    self.code_rows = (tot - 1) / 30 + 1;
                }
            }
            tot = self.code_rows * self.code_columns;
        }

        if tot > MAX_DATA_CODEWORDS as i32 + 2 {
            tot = self.max_square();
        }

        self.error_level =
            self.max_possible_error_level(tot - self.len_codewords as i32);
        let len_err = 2i32 << self.error_level;
        let mut pad = tot - len_err - self.len_codewords as i32;
        self.cw_ptr = self.len_codewords;
        while pad > 0 {
            self.codewords[self.cw_ptr] = TEXT_MODE;
            self.cw_ptr += 1;
            pad -= 1;
        }
        self.codewords[0] = self.cw_ptr as i32;
        self.len_codewords = self.cw_ptr;
        self.calculate_error_correction(self.len_codewords);
        self.len_codewords = tot as usize;

        self.out_paint_code();
        Ok(())
    }

    // --- Convert to bool matrix ---

    fn convert_to_bool_matrix(&self) -> Option<Vec<Vec<bool>>> {
        if self.out_bits.is_empty() || self.code_rows == 0 || self.bit_columns == 0 {
            return None;
        }
        let rows = self.code_rows as usize;
        let bit_cols = self.bit_columns as usize;
        let bytes_per_row = bit_cols.div_ceil(8);
        let mut result = Vec::with_capacity(rows);
        for r in 0..rows {
            let row_data: Vec<bool> = (0..bit_cols)
                .map(|c| {
                    let byte_index = r * bytes_per_row + c / 8;
                    let bit_index = 7 - (c % 8);
                    byte_index < self.out_bits.len()
                        && (self.out_bits[byte_index] & (1 << bit_index)) != 0
                })
                .collect();
            result.push(row_data);
        }
        Some(result)
    }
}

// --- Helper functions ---

fn index_of(set: &[u8], b: u8) -> i32 {
    for (i, &s) in set.iter().enumerate() {
        if s == b {
            return i as i32;
        }
    }
    -1
}

fn is_text_char(c: u8) -> bool {
    (b' '..0x7f).contains(&c) || c == b'\r' || c == b'\n' || c == b'\t'
}

// ========== Public PDF417 struct ==========

/// PDF417 2D barcode encoder.
///
/// # Example
/// ```
/// use barcode_pao::{PDF417, FORMAT_SVG};
///
/// let mut pdf = PDF417::new(FORMAT_SVG);
/// pdf.draw("Hello PDF417", 400, 200).unwrap();
/// let svg = pdf.base_2d.base.get_svg().unwrap();
/// assert!(!svg.is_empty());
/// ```
pub struct PDF417 {
    pub base_2d: BarcodeBase2D,
    /// Error correction level (0-8, or -1 for auto). Default: 2.
    pub error_correction_level: i32,
    columns: i32,
    rows: i32,
    size_mode: i32,
    aspect_ratio: f64,
    y_height: i32,
    quiet_zone_width: i32,
    use_auto_error_level: bool,
}

/// Size mode constants for [`PDF417::set_size_mode`].
pub const SIZE_MODE_AUTO: i32 = 0;
pub const SIZE_MODE_ROWS: i32 = 1;
pub const SIZE_MODE_COLUMNS: i32 = 2;
pub const SIZE_MODE_COLUMNS_AND_ROWS: i32 = 3;

impl PDF417 {
    /// Creates a new PDF417 encoder with the given output format ("png", "jpeg", or "svg").
    pub fn new(output_format: &str) -> Self {
        Self {
            base_2d: BarcodeBase2D::new(output_format),
            error_correction_level: 2,
            columns: 0,
            rows: 0,
            size_mode: SIZE_MODE_AUTO,
            aspect_ratio: 0.5,
            y_height: 3,
            quiet_zone_width: 2,
            use_auto_error_level: true,
        }
    }

    // --- Settings ---

    /// Sets the error correction level (-1=Auto, 0..8).
    pub fn set_error_correction_level(&mut self, level: i32) {
        if (-1..=8).contains(&level) {
            self.error_correction_level = level;
        }
    }

    /// Sets the number of data columns (0=auto, 1..30).
    pub fn set_columns(&mut self, cols: i32) {
        if (0..=30).contains(&cols) {
            self.columns = cols;
        }
    }

    /// Returns the current column count.
    pub fn columns(&self) -> i32 {
        self.columns
    }

    /// Sets the number of rows (0=auto, 3..90).
    pub fn set_rows(&mut self, rows: i32) {
        if (0..=90).contains(&rows) {
            self.rows = rows;
        }
    }

    /// Returns the current row count.
    pub fn rows(&self) -> i32 {
        self.rows
    }

    /// Sets the size mode (SIZE_MODE_AUTO, SIZE_MODE_ROWS, etc.).
    pub fn set_size_mode(&mut self, mode: i32) {
        if (SIZE_MODE_AUTO..=SIZE_MODE_COLUMNS_AND_ROWS).contains(&mode) {
            self.size_mode = mode;
        }
    }

    /// Returns the current size mode.
    pub fn size_mode(&self) -> i32 {
        self.size_mode
    }

    /// Sets the aspect ratio (0.001..1000).
    pub fn set_aspect_ratio(&mut self, ratio: f64) {
        if (0.001..=1000.0).contains(&ratio) {
            self.aspect_ratio = ratio;
        }
    }

    /// Returns the current aspect ratio.
    pub fn aspect_ratio(&self) -> f64 {
        self.aspect_ratio
    }

    /// Sets the Y-dimension height multiplier.
    pub fn set_y_height(&mut self, h: i32) {
        if h > 0 {
            self.y_height = h;
        }
    }

    /// Returns the current Y-height.
    pub fn y_height(&self) -> i32 {
        self.y_height
    }

    /// Sets the quiet zone width in modules.
    pub fn set_quiet_zone_width(&mut self, w: i32) {
        if w >= 0 {
            self.quiet_zone_width = w;
        }
    }

    /// Returns the current quiet zone width.
    pub fn quiet_zone_width(&self) -> i32 {
        self.quiet_zone_width
    }

    /// Sets whether to automatically determine error level.
    pub fn set_use_auto_error_level(&mut self, on: bool) {
        self.use_auto_error_level = on;
    }

    /// Returns the current auto error level setting.
    pub fn use_auto_error_level(&self) -> bool {
        self.use_auto_error_level
    }

    // --- Internal: apply settings to encoder ---

    fn apply_settings_full(&self, enc: &mut Pdf417Encoder) {
        enc.error_level = self.error_correction_level;
        enc.code_columns = self.columns;
        enc.code_rows = self.rows;
        enc.aspect_ratio = self.aspect_ratio;
        enc.y_height = self.y_height;
        enc.use_auto_error_level = self.use_auto_error_level;
        enc.size_kind = match self.size_mode {
            SIZE_MODE_AUTO => SIZE_AUTO,
            SIZE_MODE_COLUMNS => SIZE_COLUMNS,
            SIZE_MODE_ROWS => SIZE_ROWS,
            SIZE_MODE_COLUMNS_AND_ROWS => SIZE_COLUMNS_AND_ROWS,
            _ => SIZE_AUTO,
        };
    }

    // --- Get pattern ---

    /// Generates the PDF417 pattern as a `Vec<Vec<bool>>` matrix (true=black).
    pub fn get_pattern(&self, code: &str) -> Result<Vec<Vec<bool>>> {
        let data = code.as_bytes();
        if data.is_empty() {
            return Err(BarcodeError::EmptyCode);
        }

        let mut enc = Pdf417Encoder::new();
        self.apply_settings_full(&mut enc);
        enc.text = data.to_vec();
        enc.paint_code()?;
        enc.convert_to_bool_matrix()
            .ok_or_else(|| BarcodeError::EncodingError("failed to generate pattern".into()))
    }

    // --- Draw ---

    /// Renders the PDF417 barcode to the internal buffer (SVG or PNG/JPEG).
    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        if width <= 0 || height <= 0 {
            return Err(BarcodeError::InvalidData(
                "width and height must be positive".into(),
            ));
        }

        let patt = self.get_pattern(code)?;

        let num_rows = patt.len();
        if num_rows == 0 {
            return Err(BarcodeError::EncodingError("empty pattern".into()));
        }
        let num_cols = patt[0].len();
        if num_cols == 0 {
            return Err(BarcodeError::EncodingError("empty pattern".into()));
        }

        if self.base_2d.base.is_svg_output() {
            self.draw_svg(&patt, num_rows, num_cols, width, height)
        } else {
            #[cfg(any(feature = "png", feature = "jpeg"))]
            {
                self.draw_png(&patt, num_rows, num_cols, width, height)
            }
            #[cfg(not(any(feature = "png", feature = "jpeg")))]
            Err(BarcodeError::FormatError(
                "PNG/JPEG support requires 'png' or 'jpeg' feature".into(),
            ))
        }
    }

    // --- SVG rendering ---

    fn draw_svg(
        &mut self,
        patt: &[Vec<bool>],
        num_rows: usize,
        num_cols: usize,
        width: i32,
        height: i32,
    ) -> Result<()> {
        let (module_w, module_h) = if self.base_2d.base.fit_width {
            (
                width as f64 / num_cols as f64,
                height as f64 / num_rows as f64,
            )
        } else {
            let mut mw = (width / num_cols as i32) as f64;
            let mut mh = (height / num_rows as i32) as f64;
            if mw < 1.0 {
                mw = 1.0;
            }
            if mh < 1.0 {
                mh = 1.0;
            }
            (mw, mh)
        };

        let aw = (module_w * num_cols as f64).ceil() as i32;
        let ah = (module_h * num_rows as f64).ceil() as i32;

        let back_color = self.base_2d.base.back_color;
        let fore_color = self.base_2d.base.fore_color;
        let adj = self.base_2d.base.px_adj_black as f64;

        self.base_2d.base.svg_begin(aw, ah);
        self.base_2d
            .base
            .svg_rect(0.0, 0.0, aw as f64, ah as f64, back_color);

        for (r, row) in patt.iter().enumerate() {
            for (c, &is_black) in row.iter().enumerate() {
                if is_black {
                    let dw = (module_w + adj + 0.5).max(0.0);
                    let dh = (module_h + adj + 0.5).max(0.0);
                    self.base_2d.base.svg_rect(
                        c as f64 * module_w,
                        r as f64 * module_h,
                        dw,
                        dh,
                        fore_color,
                    );
                }
            }
        }

        self.base_2d.base.svg_end();
        Ok(())
    }

    // --- PNG rendering ---

    #[cfg(any(feature = "png", feature = "jpeg"))]
    fn draw_png(
        &mut self,
        patt: &[Vec<bool>],
        num_rows: usize,
        num_cols: usize,
        width: i32,
        height: i32,
    ) -> Result<()> {
        let (module_w, module_h) = if self.base_2d.base.fit_width {
            (
                width as f64 / num_cols as f64,
                height as f64 / num_rows as f64,
            )
        } else {
            let mut mw = (width / num_cols as i32) as f64;
            let mut mh = (height / num_rows as i32) as f64;
            if mw < 1.0 {
                mw = 1.0;
            }
            if mh < 1.0 {
                mh = 1.0;
            }
            (mw, mh)
        };

        let aw = (module_w * num_cols as f64).ceil() as i32;
        let ah = (module_h * num_rows as f64).ceil() as i32;

        let bc = self.base_2d.base.back_color;
        let mut img = image::RgbaImage::from_pixel(
            aw as u32,
            ah as u32,
            image::Rgba([bc.r, bc.g, bc.b, bc.a]),
        );

        let adj = self.base_2d.base.px_adj_black;
        let fc = self.base_2d.base.fore_color;
        let fore = image::Rgba([fc.r, fc.g, fc.b, fc.a]);

        for (r, row) in patt.iter().enumerate() {
            for (c, &is_black) in row.iter().enumerate() {
                if is_black {
                    let draw_x = (c as f64 * module_w) as i32;
                    let draw_y = (r as f64 * module_h) as i32;
                    let draw_w = ((module_w + adj as f64) as i32).max(1);
                    let draw_h = ((module_h + adj as f64) as i32).max(1);
                    let end_x = (draw_x + draw_w).min(aw);
                    let end_y = (draw_y + draw_h).min(ah);
                    crate::base_1d::fill_rect(&mut img, draw_x, draw_y, end_x, end_y, fore);
                }
            }
        }

        // Trial mode watermark
        if crate::product_info::is_trial_mode() {
            #[cfg(feature = "font")]
            crate::base_1d::draw_sample_overlay_png(&mut img, 0, 0, aw, ah);
        }

        self.base_2d.base.encode_image(&img)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_new_defaults() {
        let pdf = PDF417::new("svg");
        assert_eq!(pdf.error_correction_level, 2);
        assert_eq!(pdf.columns(), 0);
        assert_eq!(pdf.rows(), 0);
        assert_eq!(pdf.size_mode(), SIZE_MODE_AUTO);
        assert!((pdf.aspect_ratio() - 0.5).abs() < f64::EPSILON);
        assert_eq!(pdf.y_height(), 3);
        assert_eq!(pdf.quiet_zone_width(), 2);
        assert!(pdf.use_auto_error_level());
    }

    #[test]
    fn test_settings() {
        let mut pdf = PDF417::new("svg");
        pdf.set_error_correction_level(5);
        assert_eq!(pdf.error_correction_level, 5);
        pdf.set_error_correction_level(9); // out of range, no change
        assert_eq!(pdf.error_correction_level, 5);
        pdf.set_columns(10);
        assert_eq!(pdf.columns(), 10);
        pdf.set_rows(20);
        assert_eq!(pdf.rows(), 20);
        pdf.set_size_mode(SIZE_MODE_COLUMNS);
        assert_eq!(pdf.size_mode(), SIZE_MODE_COLUMNS);
        pdf.set_aspect_ratio(0.8);
        assert!((pdf.aspect_ratio() - 0.8).abs() < f64::EPSILON);
        pdf.set_y_height(5);
        assert_eq!(pdf.y_height(), 5);
        pdf.set_quiet_zone_width(4);
        assert_eq!(pdf.quiet_zone_width(), 4);
        pdf.set_use_auto_error_level(false);
        assert!(!pdf.use_auto_error_level());
    }

    #[test]
    fn test_get_pattern_basic() {
        let pdf = PDF417::new("svg");
        let patt = pdf.get_pattern("Hello").unwrap();
        assert!(!patt.is_empty());
        assert!(!patt[0].is_empty());
    }

    #[test]
    fn test_get_pattern_empty() {
        let pdf = PDF417::new("svg");
        assert!(pdf.get_pattern("").is_err());
    }

    #[test]
    fn test_get_pattern_numeric() {
        let pdf = PDF417::new("svg");
        let patt = pdf.get_pattern("1234567890123456789").unwrap();
        assert!(!patt.is_empty());
    }

    #[test]
    fn test_get_pattern_mixed() {
        let pdf = PDF417::new("svg");
        let patt = pdf.get_pattern("Hello 12345 World!").unwrap();
        assert!(!patt.is_empty());
    }

    #[test]
    fn test_draw_svg() {
        let mut pdf = PDF417::new("svg");
        pdf.draw("Hello PDF417", 400, 200).unwrap();
        let svg = pdf.base_2d.base.get_svg().unwrap();
        assert!(svg.contains("<svg"));
        assert!(svg.contains("</svg>"));
        assert!(svg.contains("<rect"));
    }

    #[test]
    fn test_draw_invalid_dimensions() {
        let mut pdf = PDF417::new("svg");
        assert!(pdf.draw("Hello", 0, 200).is_err());
        assert!(pdf.draw("Hello", 400, -1).is_err());
    }

    #[test]
    fn test_draw_long_text() {
        let mut pdf = PDF417::new("svg");
        let long_text = "A".repeat(500);
        pdf.draw(&long_text, 800, 400).unwrap();
        let svg = pdf.base_2d.base.get_svg().unwrap();
        assert!(!svg.is_empty());
    }

    #[test]
    fn test_error_correction_levels() {
        for level in 0..=8 {
            let mut pdf = PDF417::new("svg");
            pdf.set_error_correction_level(level);
            pdf.set_use_auto_error_level(false);
            let patt = pdf.get_pattern("Test");
            assert!(patt.is_ok(), "Failed for error level {}", level);
        }
    }

    #[test]
    fn test_byte_mode() {
        let pdf = PDF417::new("svg");
        // Create data with non-text bytes
        let mut enc = Pdf417Encoder::new();
        enc.error_level = 2;
        enc.use_auto_error_level = true;
        enc.aspect_ratio = 0.5;
        enc.y_height = 3;
        enc.text = vec![0x80, 0x81, 0x82, 0x83, 0x84, 0x85];
        enc.init_block();
        enc.segment_list.clear();
        enc.break_string();
        assert!(!enc.segment_list.is_empty());
        // Should be detected as byte segment
        assert_eq!(enc.segment_list[0].seg_type, b'B');
        drop(pdf);
    }

    #[test]
    fn test_text_compaction_all_uppercase() {
        let pdf = PDF417::new("svg");
        let patt = pdf.get_pattern("ABCDEFGHIJ").unwrap();
        assert!(!patt.is_empty());
    }

    #[test]
    fn test_text_compaction_mixed_case() {
        let pdf = PDF417::new("svg");
        let patt = pdf.get_pattern("AbCdEfGhIj").unwrap();
        assert!(!patt.is_empty());
    }

    #[test]
    fn test_columns_and_rows_mode() {
        let mut pdf = PDF417::new("svg");
        pdf.set_size_mode(SIZE_MODE_COLUMNS_AND_ROWS);
        pdf.set_columns(5);
        pdf.set_rows(10);
        pdf.draw("Test data", 400, 200).unwrap();
        let svg = pdf.base_2d.base.get_svg().unwrap();
        assert!(!svg.is_empty());
    }

    #[test]
    fn test_index_of() {
        assert_eq!(index_of(b"0123456789", b'0'), 0);
        assert_eq!(index_of(b"0123456789", b'9'), 9);
        assert_eq!(index_of(b"0123456789", b'a'), -1);
    }

    #[test]
    fn test_is_text_char() {
        assert!(is_text_char(b' '));
        assert!(is_text_char(b'A'));
        assert!(is_text_char(b'~'));
        assert!(is_text_char(b'\r'));
        assert!(is_text_char(b'\n'));
        assert!(is_text_char(b'\t'));
        assert!(!is_text_char(0x80));
        assert!(!is_text_char(0x7f));
    }

    #[test]
    fn test_pattern_deterministic() {
        let pdf = PDF417::new("svg");
        let patt1 = pdf.get_pattern("Hello World").unwrap();
        let patt2 = pdf.get_pattern("Hello World").unwrap();
        assert_eq!(patt1.len(), patt2.len());
        for (r1, r2) in patt1.iter().zip(patt2.iter()) {
            assert_eq!(r1, r2);
        }
    }
}
