use crate::base_1d::{BarcodeBase1D, Encoder1D};
use crate::error::{BarcodeError, Result};

const NW7_CHARS: &str = "ABCD.+:/$-0123456789";
const NW7_PATTERNS: &[u8] = &[
    0x1a, 0x29, 0x0b, 0x0e, 0x54, 0x15, 0x45, 0x51, 0x18, 0x0c,
    0x03, 0x06, 0x09, 0x60, 0x12, 0x42, 0x21, 0x24, 0x30, 0x48,
];

/// NW-7 (Codabar) barcode encoder.
pub struct NW7 {
    pub base_1d: BarcodeBase1D,
    pub show_start_stop: bool,
}

impl NW7 {
    pub fn new(output_format: &str) -> Self {
        Self {
            base_1d: BarcodeBase1D::new(output_format),
            show_start_stop: true,
        }
    }

    pub fn encode(&self, code: &str) -> Result<Vec<i32>> {
        NW7Encoder.encode(code)
    }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        self.base_1d.draw(code, width, height, &NW7Encoder)
    }
}

struct NW7Encoder;

impl Encoder1D for NW7Encoder {
    fn encode(&self, code: &str) -> Result<Vec<i32>> {
        encode_nw7(code)
    }
}

pub fn encode_nw7(code: &str) -> Result<Vec<i32>> {
    if code.is_empty() {
        return Err(BarcodeError::EmptyCode);
    }

    let s = code.to_uppercase();
    let bytes = s.as_bytes();

    let start_char = bytes[0] as char;
    let stop_char = bytes[bytes.len() - 1] as char;
    let is_start_valid = ('A'..='D').contains(&start_char);
    let is_stop_valid = ('A'..='D').contains(&stop_char);

    let encoded = if is_start_valid && is_stop_valid {
        s.clone()
    } else if is_start_valid || is_stop_valid {
        return Err(BarcodeError::InvalidData(format!(
            "invalid start/stop character combination in NW7: {}",
            s
        )));
    } else {
        format!("C{}C", s)
    };

    let mut result = Vec::new();
    for (i, b) in encoded.bytes().enumerate() {
        if i > 0 {
            result.push(2); // inter-character gap
        }

        let char_index = NW7_CHARS
            .find(b as char)
            .ok_or_else(|| BarcodeError::InvalidCharacter {
                ch: b as char,
                symbology: "NW7".into(),
            })?;

        let pattern = NW7_PATTERNS[char_index];

        // 7-bit pattern: MSB first, 1=wide(3), 0=narrow(1)
        let mask: u8 = 0x40;
        for j in 0..7 {
            if (pattern & (mask >> j)) != 0 {
                result.push(3);
            } else {
                result.push(1);
            }
        }
    }

    Ok(result)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_basic() {
        let pattern = encode_nw7("A12345B").unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_auto_wrap() {
        let pattern = encode_nw7("12345").unwrap();
        assert!(!pattern.is_empty());
        // Should auto-wrap with C...C
    }

    #[test]
    fn test_encode_empty() {
        assert!(encode_nw7("").is_err());
    }

    #[test]
    fn test_encode_invalid_start_stop() {
        assert!(encode_nw7("A12345").is_err());
    }
}
