use crate::base_1d::{BarcodeBase1D, Encoder1D};
use crate::error::{BarcodeError, Result};

const NEC2OF5_DIGIT_PATTERNS: &[i32] = &[
    0x18, 0x03, 0x05, 0x06, 0x09, 0x0a, 0x0c, 0x11, 0x12, 0x14,
];
const NEC2OF5_START: i32 = 0x5; // 101
const NEC2OF5_STOP: i32 = 0x3; // 011

/// NEC 2 of 5 barcode encoder.
pub struct NEC2of5 {
    pub base_1d: BarcodeBase1D,
}

impl NEC2of5 {
    pub fn new(output_format: &str) -> Self {
        Self {
            base_1d: BarcodeBase1D::new(output_format),
        }
    }

    pub fn encode(&self, code: &str) -> Result<Vec<i32>> {
        let encoder = NEC2of5Encoder {
            min_line_width: self.base_1d.min_line_width,
        };
        encoder.encode(code)
    }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        let encoder = NEC2of5Encoder {
            min_line_width: self.base_1d.min_line_width,
        };
        self.base_1d.draw(code, width, height, &encoder)
    }
}

struct NEC2of5Encoder {
    min_line_width: i32,
}

impl Encoder1D for NEC2of5Encoder {
    fn encode(&self, code: &str) -> Result<Vec<i32>> {
        encode_nec2of5(code, self.min_line_width)
    }
}

pub fn encode_nec2of5(code: &str, min_line_width: i32) -> Result<Vec<i32>> {
    if code.is_empty() {
        return Err(BarcodeError::EmptyCode);
    }

    for c in code.chars() {
        if !c.is_ascii_digit() {
            return Err(BarcodeError::InvalidData(
                "NEC 2 of 5 barcode requires numeric digits only".into(),
            ));
        }
    }

    let wide = if min_line_width % 2 != 0 { 3 } else { 2 };
    let narrow = 1;

    let mut result = Vec::new();

    // Start code (3 bits: 101)
    let chk = 0x4;
    for j in 0..3 {
        if j % 2 == 0 {
            // bar position
            if (NEC2OF5_START & (chk >> j)) != 0 {
                result.push(wide);
            } else {
                result.push(narrow);
            }
        } else {
            // space position - always narrow for NEC
            result.push(narrow);
        }
    }

    // Inter-character gap
    result.push(narrow);

    // Data encoding
    let bytes = code.as_bytes();
    for i in 0..bytes.len() {
        let digit = (bytes[i] - b'0') as usize;
        let pattern = NEC2OF5_DIGIT_PATTERNS[digit];

        let chk = 0x10;
        for j in 0..5 {
            if j % 2 == 0 {
                // bar position
                if (pattern & (chk >> j)) != 0 {
                    result.push(wide);
                } else {
                    result.push(narrow);
                }
            } else {
                // space position - always narrow for NEC data
                result.push(narrow);
            }
        }

        // Inter-character gap (except last)
        if i < bytes.len() - 1 {
            result.push(narrow);
        }
    }

    // Stop code (3 bits: 011) - both bar and space respond to bit pattern
    let chk = 0x4;
    for j in 0..3 {
        if j % 2 == 0 {
            // bar position
            if (NEC2OF5_STOP & (chk >> j)) != 0 {
                result.push(wide);
            } else {
                result.push(narrow);
            }
        } else {
            // space position - responds to bit pattern for stop
            if (NEC2OF5_STOP & (chk >> j)) != 0 {
                result.push(wide);
            } else {
                result.push(narrow);
            }
        }
    }

    Ok(result)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_basic() {
        let pattern = encode_nec2of5("12345", 1).unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_empty() {
        assert!(encode_nec2of5("", 1).is_err());
    }

    #[test]
    fn test_encode_non_numeric() {
        assert!(encode_nec2of5("12A45", 1).is_err());
    }
}
