use crate::base_1d::BarcodeBase1D;
use crate::error::Result;

/// Parameters for JAN/UPC rendering.
pub(crate) struct JANUPCParams {
    pub(crate) _text_height: f64,
    pub(crate) normal_bar_height: f64,
    pub(crate) tall_bar_height: f64,
    pub(crate) svg_font_size: i32,
    pub(crate) text_y: f64,
    pub(crate) digit_width: f64,
    pub(crate) left_margin: f64,
    #[allow(dead_code)]
    pub(crate) char_width: f64,
    pub(crate) barcode_width: f64,
    pub(crate) barcode_start_x: f64,
    pub(crate) module_width: f64,
    pub(crate) pos_a: f64,
}

pub(crate) fn compute_params(
    height: i32,
    tf_scale: f64,
    tvo_scale: f64,
    pattern: &[i32],
    width: i32,
    cls_name: &str,
) -> JANUPCParams {
    let total_units: i32 = pattern.iter().sum();
    let module_width = width as f64 / total_units as f64;

    let base_font_size = (height as f64 * 0.2 * tf_scale).max(8.0);
    let svg_font_size = base_font_size as i32;
    let text_height = base_font_size * 1.3 * tvo_scale;
    let normal_bar_height = height as f64 - text_height;
    let tall_bar_height = height as f64 - text_height * 0.3;
    let text_y = normal_bar_height + 1.0;

    let digit_width = module_width * 7.0;
    let left_margin = module_width * 2.0;

    let char_width = digit_width;

    let (barcode_width, barcode_start_x, pos_a) = match cls_name {
        "JAN8" => {
            let bw = total_units as f64 * module_width;
            (bw, 0.0, 0.0)
        }
        "JAN13" => {
            let bw = 95.0 * module_width;
            let start_x = (width as f64 - bw) / 2.0;
            let pa = start_x - digit_width;
            (bw, start_x, pa)
        }
        "UPCA" => {
            let bw = 95.0 * module_width;
            let start_x = (width as f64 - bw) / 2.0;
            let pa = start_x - digit_width;
            (bw, start_x, pa)
        }
        "UPCE" => {
            let bw = 51.0 * module_width;
            let start_x = (width as f64 - bw) / 2.0;
            let pa = start_x - digit_width;
            (bw, start_x, pa)
        }
        _ => {
            let bw = total_units as f64 * module_width;
            (bw, 0.0, 0.0)
        }
    };

    JANUPCParams {
        _text_height: text_height,
        normal_bar_height,
        tall_bar_height,
        svg_font_size,
        text_y,
        digit_width,
        left_margin,
        char_width,
        barcode_width,
        barcode_start_x,
        module_width,
        pos_a,
    }
}

// --- SVG rendering for JAN/UPC with extended guard bars ---

fn draw_svg_bars_with_guards(
    b: &mut BarcodeBase1D,
    pattern: &[i32],
    width: i32,
    height: i32,
    p: &JANUPCParams,
    guard_segments: &[(i32, i32)], // (start_seg_count, seg_size) pairs that define tall bar segments
) {
    b.base.svg_begin(width, height);
    b.base
        .svg_rect(0.0, 0.0, width as f64, height as f64, b.base.back_color);

    let mut current_x = p.barcode_start_x;
    let mut is_bar = true;
    let mut seg_idx = 0;
    let mut seg_cnt = 0;

    for &v in pattern {
        let bar_w = v as f64 * p.module_width;

        // Check segment transitions
        if seg_idx < guard_segments.len() {
            let (_, seg_size) = guard_segments[seg_idx];
            if seg_cnt >= seg_size {
                seg_idx += 1;
                seg_cnt = 0;
            }
        }

        let is_tall = is_guard_segment(seg_idx) && is_bar;
        if is_bar && bar_w > 0.0 {
            let bh = if is_tall {
                p.tall_bar_height
            } else {
                p.normal_bar_height
            };
            b.base
                .svg_rect(current_x, 0.0, bar_w, bh, b.base.fore_color);
        }

        seg_cnt += 1;
        current_x += bar_w;
        is_bar = !is_bar;
    }
}

fn is_guard_segment(seg_idx: usize) -> bool {
    // Guard segments are at even indices (0, 2, 4): start guard, center guard, end guard
    seg_idx.is_multiple_of(2)
}

pub fn draw_svg_jan8(
    b: &mut BarcodeBase1D,
    pattern: &[i32],
    display_code: &str,
    width: i32,
    height: i32,
) -> Result<()> {
    let p = compute_params(height, b.text_font_scale, b.text_vertical_offset_scale, pattern, width, "JAN8");

    let guards = vec![(0, 3), (1, 16), (2, 5), (3, 16), (4, 3)];
    draw_svg_bars_with_guards(b, pattern, width, height, &p, &guards);

    // Draw text
    if display_code.len() >= 8 {
        let font_size = p.svg_font_size;
        let text_y = p.text_y;
        let mw = p.module_width;

        // Left group (4 digits)
        draw_svg_text_segment(b, &display_code[..4], p.barcode_start_x + 3.0 * mw,
            p.barcode_start_x + 31.0 * mw, p.left_margin, text_y, font_size, 4);
        // Right group (4 digits)
        draw_svg_text_segment(b, &display_code[4..8], p.barcode_start_x + 36.0 * mw,
            p.barcode_start_x + 64.0 * mw, p.left_margin, text_y, font_size, 4);
    }

    b.base.svg_end();
    Ok(())
}

pub fn draw_svg_jan13(
    b: &mut BarcodeBase1D,
    pattern: &[i32],
    display_code: &str,
    width: i32,
    height: i32,
) -> Result<()> {
    let p = compute_params(height, b.text_font_scale, b.text_vertical_offset_scale, pattern, width, "JAN13");

    let guards = vec![(0, 3), (1, 24), (2, 5), (3, 24), (4, 3)];
    draw_svg_bars_with_guards(b, pattern, width, height, &p, &guards);

    if display_code.len() >= 13 {
        let font_size = p.svg_font_size;
        let text_y = p.text_y;
        let mw = p.module_width;
        let fore = b.base.fore_color;

        // First digit (standalone, left of barcode)
        b.base.svg_text(
            p.pos_a + p.digit_width * 0.3,
            text_y,
            &display_code[..1],
            font_size,
            fore,
            "middle",
        );

        // Left group (6 digits)
        draw_svg_text_segment(b, &display_code[1..7], p.barcode_start_x + 3.0 * mw,
            p.barcode_start_x + 45.0 * mw, p.left_margin, text_y, font_size, 6);
        // Right group (6 digits)
        draw_svg_text_segment(b, &display_code[7..13], p.barcode_start_x + 50.0 * mw,
            p.barcode_start_x + 92.0 * mw, p.left_margin, text_y, font_size, 6);
    }

    b.base.svg_end();
    Ok(())
}

pub fn draw_svg_upca(
    b: &mut BarcodeBase1D,
    pattern: &[i32],
    display_code: &str,
    width: i32,
    height: i32,
) -> Result<()> {
    let p = compute_params(height, b.text_font_scale, b.text_vertical_offset_scale, pattern, width, "UPCA");

    let guards = vec![(0, 7), (1, 21), (2, 3), (3, 21), (4, 7)];
    draw_svg_bars_with_guards(b, pattern, width, height, &p, &guards);

    if display_code.len() >= 12 {
        let font_size = p.svg_font_size;
        let text_y = p.text_y;
        let mw = p.module_width;
        let fore = b.base.fore_color;

        // First digit
        b.base.svg_text(
            p.pos_a + p.digit_width * 0.3,
            text_y,
            &display_code[..1],
            font_size,
            fore,
            "middle",
        );

        // Left group (5 digits)
        draw_svg_text_segment(b, &display_code[1..6], p.barcode_start_x + 10.0 * mw,
            p.barcode_start_x + 31.0 * mw, p.left_margin, text_y, font_size, 5);
        // Right group (5 digits)
        draw_svg_text_segment(b, &display_code[6..11], p.barcode_start_x + 34.0 * mw,
            p.barcode_start_x + 55.0 * mw, p.left_margin, text_y, font_size, 5);

        // Last digit
        let end_x = p.barcode_start_x + p.barcode_width;
        b.base.svg_text(
            end_x + p.digit_width * 0.7,
            text_y,
            &display_code[11..12],
            font_size,
            fore,
            "middle",
        );
    }

    b.base.svg_end();
    Ok(())
}

pub fn draw_svg_upce(
    b: &mut BarcodeBase1D,
    pattern: &[i32],
    display_code: &str,
    width: i32,
    height: i32,
) -> Result<()> {
    let p = compute_params(height, b.text_font_scale, b.text_vertical_offset_scale, pattern, width, "UPCE");

    // UPCE guard bars: based on accumulated units
    b.base.svg_begin(width, height);
    b.base
        .svg_rect(0.0, 0.0, width as f64, height as f64, b.base.back_color);

    let mut current_x = p.barcode_start_x;
    let mut is_bar = true;
    let mut acc_units = 0;

    for &v in pattern {
        let bar_w = v as f64 * p.module_width;
        let is_start_guard = acc_units < 3;
        let is_end_guard = acc_units >= 45;
        let is_tall = (is_start_guard || is_end_guard) && is_bar;

        if is_bar && bar_w > 0.0 {
            let bh = if is_tall {
                p.tall_bar_height
            } else {
                p.normal_bar_height
            };
            b.base
                .svg_rect(current_x, 0.0, bar_w, bh, b.base.fore_color);
        }

        acc_units += v;
        current_x += bar_w;
        is_bar = !is_bar;
    }

    if display_code.len() >= 8 {
        let font_size = p.svg_font_size;
        let text_y = p.text_y;
        let mw = p.module_width;
        let fore = b.base.fore_color;

        // First digit
        b.base.svg_text(
            p.pos_a + p.digit_width * 0.3,
            text_y,
            &display_code[..1],
            font_size,
            fore,
            "middle",
        );

        // Middle 6 digits
        let start_b = p.barcode_start_x + 3.0 * mw + p.digit_width * 0.5;
        let end_c = p.barcode_start_x + 45.0 * mw - p.digit_width * 0.5;
        draw_svg_text_segment(b, &display_code[1..7], start_b, end_c, p.left_margin, text_y, font_size, 6);

        // Last digit
        let end_x = p.barcode_start_x + p.barcode_width;
        b.base.svg_text(
            end_x + p.digit_width * 0.7,
            text_y,
            &display_code[7..8],
            font_size,
            fore,
            "middle",
        );
    }

    b.base.svg_end();
    Ok(())
}

/// Distributes text characters evenly across a section.
#[allow(clippy::too_many_arguments)]
fn draw_svg_text_segment(
    b: &mut BarcodeBase1D,
    text: &str,
    start_pos: f64,
    end_pos: f64,
    margin: f64,
    text_y: f64,
    font_size: i32,
    count: usize,
) {
    let section = end_pos - start_pos;
    let eff_section = section - margin * 2.0;
    if eff_section <= 0.0 || count == 0 {
        return;
    }
    let fore = b.base.fore_color;

    if b.text_even_spacing && count > 1 {
        let spacing = eff_section / (count - 1) as f64;
        for (i, ch) in text.chars().take(count).enumerate() {
            let digit_x = start_pos + margin + i as f64 * spacing;
            b.base.svg_text(
                digit_x,
                text_y,
                &ch.to_string(),
                font_size,
                fore,
                "middle",
            );
        }
    } else {
        let mid = start_pos + section / 2.0;
        b.base
            .svg_text(mid, text_y, text, font_size, fore, "middle");
    }
}

// --- PNG rendering for JAN/UPC with extended guard bars ---

pub fn draw_png_jan_upc(
    b: &mut BarcodeBase1D,
    pattern: &[i32],
    display_code: &str,
    width: i32,
    height: i32,
    cls_name: &str,
) -> Result<()> {
    #[cfg(any(feature = "png", feature = "jpeg"))]
    {
        let p = compute_params(height, b.text_font_scale, b.text_vertical_offset_scale, pattern, width, cls_name);
        // Save svg dimensions for text drawing
        b.base.svg_height = height;

        let mut img = image::RgbaImage::from_pixel(
            width as u32,
            height as u32,
            image::Rgba([
                b.base.back_color.r,
                b.base.back_color.g,
                b.base.back_color.b,
                b.base.back_color.a,
            ]),
        );

        let fore = image::Rgba([
            b.base.fore_color.r,
            b.base.fore_color.g,
            b.base.fore_color.b,
            b.base.fore_color.a,
        ]);

        let mut current_x = p.barcode_start_x;
        let mut is_bar = true;

        match cls_name {
            "UPCE" => {
                let mut acc_units = 0;
                for &v in pattern {
                    let bar_w = (v as f64 * p.module_width).max(1.0);
                    let is_start_guard = acc_units < 3;
                    let is_end_guard = acc_units >= 45;
                    let is_tall = (is_start_guard || is_end_guard) && is_bar;
                    if is_bar {
                        let bh = if is_tall { p.tall_bar_height } else { p.normal_bar_height };
                        let x1 = current_x.round() as i32;
                        let x2 = (current_x + bar_w).round() as i32;
                        if x2 > x1 {
                            crate::base_1d::fill_rect(&mut img, x1, 0, x2, bh.round() as i32, fore);
                        }
                    }
                    acc_units += v;
                    current_x += bar_w;
                    is_bar = !is_bar;
                }
            }
            _ => {
                let guard_config: Vec<(i32, i32)> = match cls_name {
                    "JAN8" => vec![(0, 3), (1, 16), (2, 5), (3, 16), (4, 3)],
                    "JAN13" => vec![(0, 3), (1, 24), (2, 5), (3, 24), (4, 3)],
                    "UPCA" => vec![(0, 7), (1, 21), (2, 3), (3, 21), (4, 7)],
                    _ => vec![(0, 3), (1, 24), (2, 5), (3, 24), (4, 3)],
                };

                let mut seg_idx = 0;
                let mut seg_cnt = 0;
                for &v in pattern {
                    let bar_w = (v as f64 * p.module_width).max(1.0);
                    if seg_idx < guard_config.len() {
                        let (_, seg_size) = guard_config[seg_idx];
                        if seg_cnt >= seg_size {
                            seg_idx += 1;
                            seg_cnt = 0;
                        }
                    }
                    let is_tall = is_guard_segment(seg_idx) && is_bar;
                    if is_bar {
                        let bh = if is_tall { p.tall_bar_height } else { p.normal_bar_height };
                        let x1 = current_x.round() as i32;
                        let x2 = (current_x + bar_w).round() as i32;
                        if x2 > x1 {
                            crate::base_1d::fill_rect(&mut img, x1, 0, x2, bh.round() as i32, fore);
                        }
                    }
                    seg_cnt += 1;
                    current_x += bar_w;
                    is_bar = !is_bar;
                }
            }
        }

        // Draw text using font
        #[cfg(feature = "font")]
        {
            let font = crate::font::get_default_font();
            if let Some(ref font) = font {
                let text_y = p.text_y.round() as i32;
                let font_size = (height as f64 * 0.2 * b.text_font_scale * 0.8).max(8.0) as i32;
                let fore_color = b.base.fore_color;
                let mw = p.module_width;

                match cls_name {
                    "JAN8" => {
                        if display_code.len() >= 8 {
                            draw_jan_upc_text_png(&mut img, font, &display_code[..4],
                                p.barcode_start_x + 3.0 * mw, p.barcode_start_x + 31.0 * mw,
                                p.left_margin, text_y, 4, font_size, fore_color, b.text_even_spacing);
                            draw_jan_upc_text_png(&mut img, font, &display_code[4..8],
                                p.barcode_start_x + 36.0 * mw, p.barcode_start_x + 64.0 * mw,
                                p.left_margin, text_y, 4, font_size, fore_color, b.text_even_spacing);
                        }
                    }
                    "JAN13" => {
                        if display_code.len() >= 13 {
                            crate::font::draw_char_centered_png(&mut img, font,
                                display_code.chars().next().unwrap(),
                                (p.pos_a + p.digit_width * 0.3).round() as i32, text_y,
                                font_size, fore_color);
                            draw_jan_upc_text_png(&mut img, font, &display_code[1..7],
                                p.barcode_start_x + 3.0 * mw, p.barcode_start_x + 45.0 * mw,
                                p.left_margin, text_y, 6, font_size, fore_color, b.text_even_spacing);
                            draw_jan_upc_text_png(&mut img, font, &display_code[7..13],
                                p.barcode_start_x + 50.0 * mw, p.barcode_start_x + 92.0 * mw,
                                p.left_margin, text_y, 6, font_size, fore_color, b.text_even_spacing);
                        }
                    }
                    "UPCA" => {
                        if display_code.len() >= 12 {
                            crate::font::draw_char_centered_png(&mut img, font,
                                display_code.chars().next().unwrap(),
                                (p.pos_a + p.digit_width * 0.3).round() as i32, text_y,
                                font_size, fore_color);
                            draw_jan_upc_text_png(&mut img, font, &display_code[1..6],
                                p.barcode_start_x + 10.0 * mw, p.barcode_start_x + 31.0 * mw,
                                p.left_margin, text_y, 5, font_size, fore_color, b.text_even_spacing);
                            draw_jan_upc_text_png(&mut img, font, &display_code[6..11],
                                p.barcode_start_x + 34.0 * mw, p.barcode_start_x + 55.0 * mw,
                                p.left_margin, text_y, 5, font_size, fore_color, b.text_even_spacing);
                            let end_x = p.barcode_start_x + p.barcode_width;
                            crate::font::draw_char_centered_png(&mut img, font,
                                display_code.chars().nth(11).unwrap(),
                                (end_x + p.digit_width * 0.7).round() as i32, text_y,
                                font_size, fore_color);
                        }
                    }
                    "UPCE" => {
                        if display_code.len() >= 8 {
                            crate::font::draw_char_centered_png(&mut img, font,
                                display_code.chars().next().unwrap(),
                                (p.pos_a + p.digit_width * 0.3).round() as i32, text_y,
                                font_size, fore_color);
                            let start_b = p.barcode_start_x + 3.0 * mw + p.digit_width * 0.5;
                            let end_c = p.barcode_start_x + 45.0 * mw - p.digit_width * 0.5;
                            draw_jan_upc_text_png(&mut img, font, &display_code[1..7],
                                start_b, end_c, p.left_margin, text_y, 6,
                                font_size, fore_color, b.text_even_spacing);
                            let end_x = p.barcode_start_x + p.barcode_width;
                            crate::font::draw_char_centered_png(&mut img, font,
                                display_code.chars().nth(7).unwrap(),
                                (end_x + p.digit_width * 0.7).round() as i32, text_y,
                                font_size, fore_color);
                        }
                    }
                    _ => {}
                }
            }
        }

        // Trial mode watermark
        if crate::product_info::is_trial_mode() {
            #[cfg(feature = "font")]
            crate::base_1d::draw_sample_overlay_png(&mut img, 0, 0, width, height);
        }

        b.base.encode_image(&img)
    }
    #[cfg(not(any(feature = "png", feature = "jpeg")))]
    {
        let _ = (b, pattern, display_code, width, height, cls_name);
        Err(BarcodeError::FormatError(
            "PNG/JPEG support requires 'png' or 'jpeg' feature".into(),
        ))
    }
}

#[cfg(all(any(feature = "png", feature = "jpeg"), feature = "font"))]
#[allow(clippy::too_many_arguments)]
fn draw_jan_upc_text_png(
    img: &mut image::RgbaImage,
    font: &ab_glyph::FontVec,
    text: &str,
    start_pos: f64,
    end_pos: f64,
    margin: f64,
    text_y: i32,
    count: usize,
    font_size: i32,
    color: crate::color::Rgba,
    even_spacing: bool,
) {
    let section = end_pos - start_pos;
    let eff_section = section - margin * 2.0;
    if eff_section <= 0.0 || count == 0 {
        return;
    }

    if even_spacing && count > 1 {
        let spacing = eff_section / (count - 1) as f64;
        for (i, ch) in text.chars().take(count).enumerate() {
            let digit_x = start_pos + margin + i as f64 * spacing;
            crate::font::draw_char_centered_png(
                img,
                font,
                ch,
                digit_x.round() as i32,
                text_y,
                font_size,
                color,
            );
        }
    } else {
        crate::font::draw_text_centered_png(
            img,
            font,
            text,
            (start_pos + section / 2.0).round() as i32,
            text_y,
            font_size,
            color,
        );
    }
}
