use crate::base_1d::BarcodeBase1D;
use crate::error::{BarcodeError, Result};
use super::bits_util::bits_to_runlength;

const JAN8_LEFT_PATTERNS: &[i32] = &[0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b];
const JAN8_RIGHT_PATTERNS: &[i32] = &[0x72, 0x66, 0x6c, 0x42, 0x5c, 0x4e, 0x50, 0x44, 0x48, 0x74];

/// Calculates the JAN8 check digit (Modulus 10, weight 3-1).
pub fn calculate_check_digit_jan8(src: &str) -> String {
    let mut total = 0;
    for (i, c) in src.bytes().take(7).enumerate() {
        let d = (c - b'0') as i32;
        if i % 2 == 0 {
            total += d * 3;
        } else {
            total += d;
        }
    }
    let cd = (10 - (total % 10)) % 10;
    cd.to_string()
}

/// JAN-8 (EAN-8) barcode encoder.
pub struct JAN8 {
    pub base_1d: BarcodeBase1D,
    pub extended_guard: bool,
}

impl JAN8 {
    pub fn new(output_format: &str) -> Self {
        Self {
            base_1d: BarcodeBase1D::new(output_format),
            extended_guard: true,
        }
    }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        let pattern = encode_jan8(code)?;
        let s = if code.len() == 7 {
            format!("{}{}", code, calculate_check_digit_jan8(code))
        } else {
            code.to_string()
        };

        if self.base_1d.base.is_svg_output() {
            if self.base_1d.show_text && self.extended_guard {
                return super::jan_upc_render::draw_svg_jan8(
                    &mut self.base_1d, &pattern, &s, width, height,
                );
            }
            let display = if self.base_1d.show_text { &s } else { "" };
            return self.base_1d.draw_svg_bars(&pattern, width, height, display);
        }
        super::jan_upc_render::draw_png_jan_upc(
            &mut self.base_1d, &pattern, &s, width, height, "JAN8",
        )
    }
}

pub fn encode_jan8(code: &str) -> Result<Vec<i32>> {
    if code.is_empty() {
        return Err(BarcodeError::EmptyCode);
    }
    for c in code.chars() {
        if !c.is_ascii_digit() {
            return Err(BarcodeError::InvalidData(
                "JAN8 barcode requires numeric digits only".into(),
            ));
        }
    }

    let s;
    if code.len() == 8 {
        let expected = calculate_check_digit_jan8(&code[..7]);
        if &code[7..8] != expected.as_str() {
            return Err(BarcodeError::InvalidData("invalid check digit".into()));
        }
        s = code.to_string();
    } else if code.len() == 7 {
        s = format!("{}{}", code, calculate_check_digit_jan8(code));
    } else {
        return Err(BarcodeError::InvalidData(
            "JAN8 barcode requires 7 or 8 digits".into(),
        ));
    }

    let left_half = &s[..4];
    let right_half = &s[4..8];

    let mut result = Vec::new();
    // Start guard (101)
    result.extend_from_slice(&[1, 1, 1]);

    // Left 4 digits (L-pattern)
    for c in left_half.bytes() {
        let digit = (c - b'0') as usize;
        let pattern = JAN8_LEFT_PATTERNS[digit];
        result.extend(bits_to_runlength(pattern, 7));
    }

    // Center guard (01010)
    result.extend_from_slice(&[1, 1, 1, 1, 1]);

    // Right 4 digits (R-pattern)
    for c in right_half.bytes() {
        let digit = (c - b'0') as usize;
        let pattern = JAN8_RIGHT_PATTERNS[digit];
        result.extend(bits_to_runlength(pattern, 7));
    }

    // End guard (101)
    result.extend_from_slice(&[1, 1, 1]);

    Ok(result)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_check_digit() {
        // 4*3+9*1+0*3+1*1+2*3+3*1+4*3 = 43; (10-43%10)%10 = 7
        assert_eq!(calculate_check_digit_jan8("4901234"), "7");
    }

    #[test]
    fn test_encode() {
        let pattern = encode_jan8("49012347").unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_7_digits() {
        let pattern = encode_jan8("4901234").unwrap();
        assert!(!pattern.is_empty());
    }
}
