use crate::base_1d::BarcodeBase1D;
use crate::error::{BarcodeError, Result};
use super::bits_util::bits_to_runlength;

const JAN13_PARITY_PATTERNS: &[i32] = &[0x00, 0x0b, 0x0d, 0x0e, 0x13, 0x19, 0x1c, 0x15, 0x16, 0x1a];
const JAN13_LEFT_L: &[i32] = &[0x0d, 0x19, 0x13, 0x3d, 0x23, 0x31, 0x2f, 0x3b, 0x37, 0x0b];
const JAN13_LEFT_G: &[i32] = &[0x27, 0x33, 0x1b, 0x21, 0x1d, 0x39, 0x05, 0x11, 0x09, 0x17];
const JAN13_RIGHT: &[i32] = &[0x72, 0x66, 0x6c, 0x42, 0x5c, 0x4e, 0x50, 0x44, 0x48, 0x74];

/// Calculates the JAN13 check digit (Modulus 10, weight 1-3).
pub fn calculate_check_digit_jan13(src: &str) -> String {
    let mut total = 0;
    for (i, c) in src.bytes().take(12).enumerate() {
        let d = (c - b'0') as i32;
        if i % 2 == 0 {
            total += d;
        } else {
            total += d * 3;
        }
    }
    let cd = (10 - (total % 10)) % 10;
    cd.to_string()
}

/// JAN-13 (EAN-13) barcode encoder.
pub struct JAN13 {
    pub base_1d: BarcodeBase1D,
    pub extended_guard: bool,
}

impl JAN13 {
    pub fn new(output_format: &str) -> Self {
        Self {
            base_1d: BarcodeBase1D::new(output_format),
            extended_guard: true,
        }
    }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        let pattern = encode_jan13(code)?;
        let s = if code.len() == 12 {
            format!("{}{}", code, calculate_check_digit_jan13(code))
        } else {
            code.to_string()
        };

        if self.base_1d.base.is_svg_output() {
            if self.base_1d.show_text && self.extended_guard {
                return super::jan_upc_render::draw_svg_jan13(
                    &mut self.base_1d, &pattern, &s, width, height,
                );
            }
            let display = if self.base_1d.show_text { &s } else { "" };
            return self.base_1d.draw_svg_bars(&pattern, width, height, display);
        }
        super::jan_upc_render::draw_png_jan_upc(
            &mut self.base_1d, &pattern, &s, width, height, "JAN13",
        )
    }
}

pub fn encode_jan13(code: &str) -> Result<Vec<i32>> {
    if code.is_empty() {
        return Err(BarcodeError::EmptyCode);
    }
    for c in code.chars() {
        if !c.is_ascii_digit() {
            return Err(BarcodeError::InvalidData(
                "JAN13 barcode requires numeric digits only".into(),
            ));
        }
    }

    let s;
    if code.len() == 13 {
        let expected = calculate_check_digit_jan13(&code[..12]);
        if &code[12..13] != expected.as_str() {
            return Err(BarcodeError::InvalidData("invalid check digit".into()));
        }
        s = code.to_string();
    } else if code.len() == 12 {
        s = format!("{}{}", code, calculate_check_digit_jan13(code));
    } else {
        return Err(BarcodeError::InvalidData(
            "JAN13 barcode requires 12 or 13 digits".into(),
        ));
    }

    let first_digit = (s.as_bytes()[0] - b'0') as usize;
    let left_half = &s[1..7];
    let right_half = &s[7..13];

    let mut result = Vec::new();
    // Start guard (101)
    result.extend_from_slice(&[1, 1, 1]);

    // Left 6 digits (L or G pattern based on first digit parity)
    let chk_oe: i32 = 0x20;
    for (i, c) in left_half.bytes().enumerate() {
        let digit = (c - b'0') as usize;
        let flg_oe = if (JAN13_PARITY_PATTERNS[first_digit] & (chk_oe >> i)) != 0 {
            1
        } else {
            0
        };
        let pattern = if flg_oe == 0 {
            JAN13_LEFT_L[digit]
        } else {
            JAN13_LEFT_G[digit]
        };
        result.extend(bits_to_runlength(pattern, 7));
    }

    // Center guard (01010)
    result.extend_from_slice(&[1, 1, 1, 1, 1]);

    // Right 6 digits (R-pattern)
    for c in right_half.bytes() {
        let digit = (c - b'0') as usize;
        let pattern = JAN13_RIGHT[digit];
        result.extend(bits_to_runlength(pattern, 7));
    }

    // End guard (101)
    result.extend_from_slice(&[1, 1, 1]);

    Ok(result)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_check_digit() {
        assert_eq!(calculate_check_digit_jan13("490123456789"), "4");
    }

    #[test]
    fn test_encode_13() {
        let pattern = encode_jan13("4901234567894").unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_12() {
        let pattern = encode_jan13("490123456789").unwrap();
        assert!(!pattern.is_empty());
    }
}
