use crate::base_1d::{BarcodeBase1D, Encoder1D};
use crate::error::{BarcodeError, Result};

const ITF_DIGIT_PATTERNS: &[i32] = &[
    0x06, 0x11, 0x09, 0x18, 0x05, 0x14, 0x0c, 0x03, 0x12, 0x0a,
];

/// ITF (Interleaved 2 of 5) barcode encoder.
pub struct ITF {
    pub base_1d: BarcodeBase1D,
}

impl ITF {
    pub fn new(output_format: &str) -> Self {
        Self {
            base_1d: BarcodeBase1D::new(output_format),
        }
    }

    pub fn encode(&self, code: &str) -> Result<Vec<i32>> {
        let encoder = ITFEncoder {
            min_line_width: self.base_1d.min_line_width,
        };
        encoder.encode(code)
    }

    pub fn draw(&mut self, code: &str, width: i32, height: i32) -> Result<()> {
        let encoder = ITFEncoder {
            min_line_width: self.base_1d.min_line_width,
        };
        self.base_1d.draw(code, width, height, &encoder)
    }
}

struct ITFEncoder {
    min_line_width: i32,
}

impl Encoder1D for ITFEncoder {
    fn encode(&self, code: &str) -> Result<Vec<i32>> {
        encode_itf(code, self.min_line_width)
    }
}

pub fn encode_itf(code: &str, min_line_width: i32) -> Result<Vec<i32>> {
    if code.is_empty() {
        return Err(BarcodeError::EmptyCode);
    }

    for c in code.chars() {
        if !c.is_ascii_digit() {
            return Err(BarcodeError::InvalidData(
                "ITF barcode requires numeric digits only".into(),
            ));
        }
    }

    let s = if !code.len().is_multiple_of(2) {
        format!("0{}", code)
    } else {
        code.to_string()
    };

    let wide = if min_line_width % 2 != 0 { 3 } else { 2 };

    let mut result = Vec::new();
    // Start pattern
    result.extend_from_slice(&[1, 1, 1, 1]);

    // Encode digit pairs
    let bytes = s.as_bytes();
    let mut i = 0;
    while i < bytes.len() {
        let d1 = (bytes[i] - b'0') as usize;
        let d2 = (bytes[i + 1] - b'0') as usize;
        let p1 = ITF_DIGIT_PATTERNS[d1];
        let p2 = ITF_DIGIT_PATTERNS[d2];

        // Interleave the two 5-bit patterns
        let mut combined: i32 = 0x01;
        for j in 0..5 {
            let bit1 = if (p1 & (0x10 >> j)) != 0 { 1 } else { 0 };
            let bit2 = if (p2 & (0x10 >> j)) != 0 { 1 } else { 0 };
            combined = (combined << 1) | bit1;
            combined = (combined << 1) | bit2;
        }
        combined &= 0x3FF; // 10 bits

        for j in 0..10 {
            if (combined & (0x200 >> j)) != 0 {
                result.push(wide);
            } else {
                result.push(1);
            }
        }

        i += 2;
    }

    // Stop pattern
    result.extend_from_slice(&[wide, 1, 1]);

    Ok(result)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_encode_even() {
        let pattern = encode_itf("1234", 1).unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_odd_padding() {
        let pattern = encode_itf("12345", 1).unwrap();
        assert!(!pattern.is_empty());
    }

    #[test]
    fn test_encode_empty() {
        assert!(encode_itf("", 1).is_err());
    }

    #[test]
    fn test_encode_non_numeric() {
        assert!(encode_itf("12A45", 1).is_err());
    }
}
